﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using Curse.ClientService.Models;
using System.Configuration;
using System.Data.SqlClient;
using System.Data;
using System.Web.Caching;
using System.IO;

using Curse.Extensions;
using Curse.ClientService.Extensions;
using System.Threading;
using Curse.AddOns;

namespace Curse.ClientService
{
    public class CSavedGameSyncCache
    {
        private string _databaseConnectionString;
        private string _savedGameSyncPath;
        private Cache _distributedCache = null;
        private readonly TimeSpan _defaultCacheDuration = TimeSpan.FromMinutes(5);

        //private int _maxSavedGamesFree;
        //private int _maxRevisionsFree;
        //private int _maxSavedGamesPremium;
        //private int _maxRevisionsPremium;
        
        private Thread _cullingThread;

        private ESavedGameRestrictionLevel _restrictionLevel;

        private static readonly CSavedGameSyncCache _instance = new CSavedGameSyncCache();
        public static CSavedGameSyncCache Instance { get { return _instance; } }
        public ESavedGameRestrictionLevel RestrictionLevel { get { return _restrictionLevel; } set { _restrictionLevel = value; } }

        public CSavedGameSyncCache()
        {
            _databaseConnectionString = ConfigurationManager.ConnectionStrings["ClientService"].ConnectionString;
            _savedGameSyncPath = ConfigurationManager.AppSettings["SavedGameSyncPath"];
            //_maxSavedGamesFree = Int32.Parse(ConfigurationManager.AppSettings["MaxSavedGamesFree"]);
            //_maxRevisionsFree = Int32.Parse(ConfigurationManager.AppSettings["MaxRevisionsFree"]);
            //_maxSavedGamesPremium = Int32.Parse(ConfigurationManager.AppSettings["MaxSavedGamesPremium"]);
            //_maxRevisionsPremium = Int32.Parse(ConfigurationManager.AppSettings["MaxRevisionsPremium"]);

            _distributedCache = HttpRuntime.Cache;
            _defaultCacheDuration = TimeSpan.Parse(ConfigurationManager.AppSettings["SyncCacheDefaultDuration"]);

            _restrictionLevel = (ESavedGameRestrictionLevel)Int32.Parse(ConfigurationManager.AppSettings["DefaultRestrictionLevel"]);

            _cullingThread = new Thread(CullOrphanedFiles)
            {
                IsBackground = true,
                Priority = ThreadPriority.Lowest
            };
            _cullingThread.Start();
        }

        public static void Initialize() { }

        private void CullOrphanedFiles()
        {
            var aborted = false;
            while(!aborted)
            {
                Thread.Sleep(60000);
                try
                {
                    var deletedFiles = new List<string>();
                    var revisionIds = new List<int>();
                    using (var conn = new SqlConnection(_databaseConnectionString))
                    {
                        var cmd = new SqlCommand("SELECT S.UserID, S.GameID, SR.ID FROM SavedGame S INNER JOIN SavedGameRevision SR ON S.ID = SR.SavedGameID WHERE SR.[Status] = 2", conn);
                        
                        conn.Open();
                        using (var reader = cmd.ExecuteReader())
                        {
                            while (reader.Read())
                            {
                                int userId = reader.GetInt32(reader.GetOrdinal("UserID"));
                                int gameId = reader.GetInt32(reader.GetOrdinal("GameID"));
                                int revisionId = reader.GetInt32(reader.GetOrdinal("ID"));

                                var di = new DirectoryInfo(_savedGameSyncPath.FormatWith(userId.ToFullModPath(), gameId));
                                string fullFileName = Path.Combine(di.FullName, revisionId + ".zip");

                                if (!deletedFiles.Contains(fullFileName))
                                {
                                    deletedFiles.Add(fullFileName);
                                    revisionIds.Add(revisionId);
                                }
                            }
                        }
                    }

                    DeleteOrphanedFiles(deletedFiles);

                    foreach (var revisionId in revisionIds)
                    {
                        var revision = new CSavedGameRevision { ID = revisionId };
                        revision.HardDelete();
                    }
                }
                catch (ThreadAbortException)
                {
                    aborted = true;
                    Logger.Log(ELogLevel.Info, null, "Thread Abort Exception. Service shutting down.");
                }
                catch (Exception ex)
                {
                    Logger.Log(ELogLevel.Info, null, "Update Thread Exception: {0}", ex.Message + "\n" + ex.StackTrace);
                }
            }
        }

        private void DeleteOrphanedFiles(List<string> deletedFiles)
        {
            foreach (string fileName in deletedFiles)
            {
                if (File.Exists(fileName))
                {
                    File.Delete(fileName);
                }
                else
                {
                    Logger.Log("DeleteOrphanedFiles: Could not find revision file {0}", ELogLevel.Error, fileName);
                }
            }
        }
        private void DeleteOrphanedFiles(string path, List<string> validFiles)
        {
            string[] files = Directory.GetFiles(path);
            foreach (var file in files)
            {
                if (!validFiles.Contains(file))
                {
                    File.Delete(file);
                }
            }

            string[] dirs = Directory.GetDirectories(path);
            foreach (var dir in dirs)
            {
                DeleteOrphanedFiles(dir, validFiles);
            }
        }
        
        private void CacheSavedGames(List<CSavedGame> savedGames, int userId)
        {
            string cacheKey = string.Format("SyncProfile-{0}", userId);
            _distributedCache.Remove(cacheKey);

            if (savedGames != null)
            {
                _distributedCache.Add(cacheKey, savedGames, null, Cache.NoAbsoluteExpiration, _defaultCacheDuration, CacheItemPriority.High, null);
            }
        }

        private List<CSavedGame> GetSavedGamesFromCache(int userId)
        {
            var cacheKey = string.Format("SavedGames-{0}", userId);

            var savedGames = _distributedCache.Get(cacheKey) as List<CSavedGame>;
            if (savedGames == null)
            {
                savedGames = GetSavedGamesFromDatabase(userId);
                CacheSavedGames(savedGames, userId);
            }

            return savedGames;
        }

        private List<CSavedGame> GetSavedGamesFromDatabase(int userId)
        {
            List<CSavedGame> savedGames = new List<CSavedGame>();

            using (var conn = new SqlConnection(_databaseConnectionString))
            {
                var revisions = GetRevisionsForUser(userId, conn);

                var cmd = new SqlCommand("spGetSavedGames", conn);
                cmd.CommandType = CommandType.StoredProcedure;
                cmd.Parameters.Add("UserID", SqlDbType.Int).Value = userId;

                if (conn.State != ConnectionState.Open)
                {
                    conn.Open();
                }

                using (var reader = cmd.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        var savedGame = new CSavedGame();
                        savedGame.SetFromDataReader(reader, revisions);

                        savedGames.Add(savedGame);
                    }
                }
            }

            return savedGames;
        }

        private Dictionary<int, List<CSavedGameRevision>> GetRevisionsForUser(int userId, SqlConnection conn)
        {
            var revisions = new Dictionary<int, List<CSavedGameRevision>>();

            using (var cmd = new SqlCommand("spGetSavedGameRevisions", conn))
            {
                cmd.CommandType = CommandType.StoredProcedure;
                cmd.Parameters.Add("UserID", SqlDbType.Int).Value = userId;

                if (conn.State != ConnectionState.Open)
                {
                    conn.Open();
                }

                using (var reader = cmd.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        var revision = new CSavedGameRevision();
                        revision.SetFromDataReader(reader);

                        if (!revisions.ContainsKey(revision.SavedGameID))
                        {
                            revisions.Add(revision.SavedGameID, new List<CSavedGameRevision>());
                        }

                        revisions[revision.SavedGameID].Add(revision);
                    }
                }
            }

            return revisions;
        }

        //private CSavedGameConstraints Constraints(bool isPremium)
        //{
        //    var constraints = new CSavedGameConstraints
        //    {
        //        MaxRevionCount = isPremium ? _maxRevisionsPremium : _maxRevisionsFree,
        //        MaxSavedGameCount = isPremium ? _maxSavedGamesPremium : _maxSavedGamesFree
        //    };
        //    return constraints;
        //}

        public CServiceResponse<List<CSavedGame>> GetSavedGames(int userId)
        {
            var savedGames = GetSavedGamesFromCache(userId);
            return new CServiceResponse<List<CSavedGame>>(EServiceResponseStatus.Successful, savedGames);
        }

        public CServiceResponse DeleteSavedGame(int userId, int savedGameId)
        {
            var savedGames = GetSavedGamesFromCache(userId);
            var savedGame = savedGames.FirstOrDefault(p => p.ID == savedGameId);
            if (savedGame == null)
            {
                return new CServiceResponse(EServiceResponseStatus.DeleteSavedGame_SavedGameNotFound);
            }

            savedGame.Delete();
            savedGames.Remove(savedGame);
            CacheSavedGames(savedGames, userId);

            return new CServiceResponse(EServiceResponseStatus.Successful);
        }

        public CServiceResponse DeleteSavedGameRevision(int userId, int savedGameId, int savedGameRevisionId)
        {
            var savedGames = GetSavedGamesFromCache(userId);
            var savedGame = savedGames.FirstOrDefault(p => p.ID == savedGameId);
            if (savedGame == null)
            {
                return new CServiceResponse(EServiceResponseStatus.DeleteSavedGameRevision_SavedGameNotFound);
            }

            var revision = savedGame.SavedGameRevisions.FirstOrDefault(p => p.ID == savedGameRevisionId);
            if (revision == null)
            {
                return new CServiceResponse(EServiceResponseStatus.DeleteSavedGameRevision_RevisionNotFound);
            }

            var fullFileName = Path.Combine(_savedGameSyncPath.FormatWith(userId.ToFullModPath(), savedGame.GameID), revision.ID + ".zip");
            if (File.Exists(fullFileName))
            {
                File.Delete(fullFileName);
            }
            else
            {
                Logger.Log("Could not find revision file: {0}", ELogLevel.Error, fullFileName);
            }

            revision.SoftDelete();
            savedGame.SavedGameRevisions.Remove(revision);
            if (savedGame.SavedGameRevisions.Count == 0)
            {
                savedGames.Remove(savedGame);
            }
            CacheSavedGames(savedGames, userId);

            return new CServiceResponse(EServiceResponseStatus.Successful);
        }

        public CServiceResponse SetSavedGameStatus(int userId, int savedGameId, ESavedGameStatus status)
        {
            var savedGames = GetSavedGamesFromCache(userId);
            var savedGame = savedGames.FirstOrDefault(p => p.ID == savedGameId);
            if (savedGame == null)
            {
                return new CServiceResponse(EServiceResponseStatus.SetSavedGameStatus_SavedGameNotFound);
            }

            savedGame.Status = status;
            savedGame.Save();
            CacheSavedGames(savedGames, userId);

            return new CServiceResponse(EServiceResponseStatus.Successful);
        }

        public CServiceResponse SetSavedGameName(int userId, int savedGameId, string name)
        {
            var savedGames = GetSavedGamesFromCache(userId);
            var savedGame = savedGames.FirstOrDefault(p => p.ID == savedGameId);
            if (savedGame == null)
            {
                return new CServiceResponse(EServiceResponseStatus.SetSavedGameName_SavedGameNotFound);
            }

            savedGame.Name = name;
            savedGame.Save();
            CacheSavedGames(savedGames, userId);

            return new CServiceResponse(EServiceResponseStatus.Successful);
        }

        public CServiceResponse SetSavedGameDescription(int userId, int savedGameId, string description)
        {
            var savedGames = GetSavedGamesFromCache(userId);
            var savedGame = savedGames.FirstOrDefault(p => p.ID == savedGameId);
            if (savedGame == null)
            {
                return new CServiceResponse(EServiceResponseStatus.SetSavedGameDescription_SavedGameNotFound);
            }

            savedGame.Description = description;
            savedGame.Save();
            CacheSavedGames(savedGames, userId);

            return new CServiceResponse(EServiceResponseStatus.Successful);
        }

        public CServiceResponse SetSavedGameDefaultRevision(int userId, int savedGameId, int revisionId)
        {
            var savedGames = GetSavedGamesFromCache(userId);
            var savedGame = savedGames.FirstOrDefault(p => p.ID == savedGameId);
            if (savedGame == null)
            {
                return new CServiceResponse(EServiceResponseStatus.SetSavedGameDefaultRevision_SavedGameNotFound);
            }

            var revision = savedGame.SavedGameRevisions.FirstOrDefault(p => p.ID == revisionId);
            if (revision == null)
            {
                return new CServiceResponse(EServiceResponseStatus.SetSavedGameDefaultRevision_RevisionNotFound);
            }

            savedGame.SetDefaultRevision(revision.ID);

            return new CServiceResponse(EServiceResponseStatus.Successful);
        }

        public CServiceResponse SetSavedGameFavoriteRevision(int userId, int savedGameId, int revisionId)
        {
            var savedGames = GetSavedGamesFromCache(userId);
            var savedGame = savedGames.FirstOrDefault(p => p.ID == savedGameId);
            if (savedGame == null)
            {
                return new CServiceResponse(EServiceResponseStatus.SetSavedGameFavoriteRevision_SavedGameNotFound);
            }

            var revision = savedGame.SavedGameRevisions.FirstOrDefault(p => p.ID == revisionId);
            if (revision == null)
            {
                return new CServiceResponse(EServiceResponseStatus.SetSavedGameFavoriteRevision_RevisionNotFound);
            }

            revision.SetFavoriteRevision();

            return new CServiceResponse(EServiceResponseStatus.Successful);
        }

        //public CServiceResponse<CSavedGameConstraints> GetSavedGameConstraints(bool isPremium)
        //{
        //    return new CServiceResponse<CSavedGameConstraints>(EServiceResponseStatus.Successful, Constraints(isPremium));
        //}

        public void SyncSavedGame(int userId, int gameId, string filename, string computerName, long fingerprint, DateTime dateModified, Stream stream, bool isPremium)
        {
            var game = CGameCache.Instance.Games.FirstOrDefault(p => p.ID == gameId);
            if (game == null)
            {
                throw new Exception("Failed to pull the game specified.");
            }

            var savedGames = GetSavedGamesFromCache(userId);
            if(savedGames.Sum(p => p.TotalSize) > (isPremium ? game.MaxPremiumStorage : game.MaxFreeStorage))
            {
                return;
            }

            var savedGame = savedGames.FirstOrDefault(p => p.FileName == filename);
            if (savedGame == null)
            {
                savedGame = new CSavedGame
                {
                    UserID = userId,
                    GameID = gameId,
                    Name = filename,
                    FileName = filename,
                    Status = ESavedGameStatus.Private,
                    SavedGameRevisions = new List<CSavedGameRevision>()
                };
            }
            savedGame.DateModified = dateModified;

            var revision = new CSavedGameRevision
            {
                ComputerName = computerName,
                Fingerprint = fingerprint
            };

            DirectoryInfo di = new DirectoryInfo(_savedGameSyncPath.FormatWith(userId.ToFullModPath(), gameId));
            if (!di.Exists)
            {
                di.Create();
            }
            var uniqueNumber = Utility.UniqueNumber;
            var fullFileName = Path.Combine(di.FullName, uniqueNumber + ".zip");

            FileStream targetStream = null;
            using (targetStream = new FileStream(fullFileName, FileMode.Create, FileAccess.Write, FileShare.None))
            {
                const int bufferLen = 4096; //read from the input stream in 4K chunks and save to output stream
                byte[] buffer = new byte[bufferLen];
                int count = 0;
                while ((count = stream.Read(buffer, 0, bufferLen)) > 0)
                {
                    targetStream.Write(buffer, 0, count);

                    if (targetStream.Length > game.MaxFileSize)
                    {
                        targetStream.Close();
                        File.Delete(fullFileName);

                        throw new Exception("Max file size exceeded.");
                    }
                }
                revision.FileSize = targetStream.Length;
                targetStream.Close();
            }

            savedGame.SavedGameRevisions.Add(revision);
            savedGame.Save();

            savedGame.SetDefaultRevision(revision.ID);

            if (File.Exists(fullFileName))
            {
                var newFileName = fullFileName.Replace(uniqueNumber + ".zip", revision.ID + ".zip");
                File.Move(fullFileName, newFileName);

                savedGames.Add(savedGame);
                CacheSavedGames(savedGames, userId);
            }
            else
            {
                savedGame.SavedGameRevisions.Remove(revision);
                revision.SoftDelete();                
            }
        }

        public void AddRevision(int userId, int gameId, string filename, string computerName, long fingerprint, DateTime dateModified, Stream stream, bool isPremium)
        {
            SyncSavedGame(userId, gameId, filename, computerName, fingerprint, dateModified, stream, isPremium);
        }
    }
}