﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using Curse.Extensions;
using System.Configuration;
using System.Data.SqlClient;
using System.Threading;
using System.Data;
using Curse.ClientService.Models;

namespace Curse.ClientService
{
    public class CloudUserCache
    {
        private List<int> _userIds;
        private object _userLock = new object();

        private string _databaseConnectionString = null;
        private int _updateThreadInterval;
        private Thread _updateThread = null;

        private int _maxCloudUsers;

        static CloudUserCache _instance = new CloudUserCache();
        public static CloudUserCache Instance { get { return _instance; } }

        public CloudUserCache()
        {
            _updateThreadInterval = int.Parse(ConfigurationManager.AppSettings["UpdateThreadInterval"]);
            _databaseConnectionString = ConfigurationManager.ConnectionStrings["ClientService"].ConnectionString;
            _maxCloudUsers = Int32.Parse(ConfigurationManager.AppSettings["MaxCloudUsers"]);

            _userIds = new List<int>();

            UpdateCache();

            _updateThread = new Thread(CacheThread) { IsBackground = true };
            _updateThread.Priority = ThreadPriority.Lowest;
            _updateThread.Start();
        }
        public void Initialize() { }

        #region Caching
        private void CacheThread()
        {
            Boolean aborted = false;
            while (!aborted)
            {
                Thread.Sleep(_updateThreadInterval);
                try
                {
                    UpdateCache();
                }
                catch (ThreadAbortException)
                {
                    aborted = true;
                    _updateThread.Join(100);
                    Logger.Log(ELogLevel.Info, null, "Thread Abort Exception. Service shutting down.");
                }
                catch (Exception ex)
                {
                    Logger.Log(ELogLevel.Info, null, "Update Thread Exception: {0}", ex.Message + "\n" + ex.StackTrace);
                }
            }
        }
        private void UpdateCache()
        {
            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                try
                {
                    conn.Open();
                }
                catch (Exception)
                {
                    Logger.Log(ELogLevel.Info, "localhost", "Unable to establish connection to database:" + DateTime.Now.ToString());
                    return;
                }
                List<int> users = new List<int>();

                SqlCommand command = new SqlCommand("SELECT * FROM CloudUsage", conn);
                using (SqlDataReader reader = command.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        int userId = reader.GetInt32(reader.GetOrdinal("UserID"));
                        users.Add(userId);
                    }
                }

                lock (_userLock)
                {
                    _userIds = users;
                }
            }
        }
        #endregion

        public CServiceResponse UserOrSlotsAvailable(int userId)
        {
            if (_userIds.Contains(userId))
            {
                return new CServiceResponse(EServiceResponseStatus.Successful);
            }
            else if ((_maxCloudUsers - _userIds.Count) > 0)
            {
                return ReserveSlot(userId);
            }
            return new CServiceResponse(EServiceResponseStatus.UserOrSlotsAvailable_NoCloudSlotsAvailable);
        }
        public CServiceResponse ReserveSlot(int userId)
        {
            using (var conn = new SqlConnection(_databaseConnectionString))
            {
                var cmd = new SqlCommand("spAddCloudUser", conn);
                cmd.CommandType = CommandType.StoredProcedure;
                cmd.Parameters.Add("UserID", SqlDbType.Int).Value = userId;
                cmd.Parameters.Add("MaxNumUsers", SqlDbType.Int).Value = _maxCloudUsers;

                conn.Open();
                try
                {
                    cmd.ExecuteNonQuery(); // Procedure returns 1 for success & 0 for failure
                    lock (_userLock)
                    {
                        _userIds.Add(userId);
                    }
                    return new CServiceResponse(EServiceResponseStatus.Successful);
                }
                catch (Exception exc)
                {
                    Logger.Log("Unable to reserve slot for user {0}. Details: {1}", ELogLevel.Error, userId, exc.GetExceptionDetails());
                    return new CServiceResponse(EServiceResponseStatus.UnknownException, exc.GetExceptionDetails());
                }
            }
        }
    }
}
