﻿using System;
using System.Collections.Generic;
using Curse.CommunityTracker.Models;
using System.Threading;
using System.Configuration;
using System.Data.SqlClient;
using Curse.CommunityTracker.Extensions;

namespace Curse.CommunityTracker.Caching
{
    public class ApplicationActivityReportingCache
    {
        private Dictionary<int, ApplicationActivityReporting> _applicationActivityReporting = new Dictionary<int, ApplicationActivityReporting>();
        private Dictionary<string, List<ApplicationActivityReporting>> _applicationActivityReportingByApplication = new Dictionary<string, List<ApplicationActivityReporting>>();

        private bool _isCacheBuilt = false;
        private bool _isCacheUpdating = false;
        public bool IsCacheUpdating { get { return _isCacheUpdating; } }
        public bool IsCacheBuilt { get { return _isCacheBuilt; } set { _isCacheBuilt = value; } }

        private readonly Thread _updateThread = null;
        private readonly int _updateThreadInterval;
        private DateTime _lastQueryTime = new DateTime(1975, 12, 12);
        private readonly string _databaseConnectionString = string.Empty;

        private static readonly ApplicationActivityReportingCache _instance = new ApplicationActivityReportingCache();
        public static ApplicationActivityReportingCache Instance { get { return _instance; } }
        public DateTime LastQueryTime { get { return _lastQueryTime; } }

        public ApplicationActivityReportingCache()
        {
            _isCacheBuilt = false;
            _updateThreadInterval = int.Parse(ConfigurationManager.AppSettings["UpdateThreadInterval"]);
            _databaseConnectionString = ConfigurationManager.ConnectionStrings["CommunityTracker"].ConnectionString;

            UpdateCache();

            _updateThread = new Thread(CacheThread) { IsBackground = true, Priority = ThreadPriority.Lowest };
            _updateThread.Start();
        }

        public static void Initialize() { }

        private void CacheThread()
        {
            Boolean aborted = false;
            while (!aborted)
            {
                Thread.Sleep(_updateThreadInterval);
                GC.Collect();
                _isCacheUpdating = true;
                try
                {
                    UpdateCache();
                }
                catch (ThreadAbortException)
                {
                    aborted = true;
                    _updateThread.Join(100);
                    Logger.Log(ELogLevel.Info, null, "Thread Abort Exception. Service shutting down.");
                }
                catch (Exception ex)
                {

                    Logger.Log(ELogLevel.Info, null, "Update Thread Exception: {0}", ex.Message + "\n" + ex.StackTrace);
                }
                _isCacheUpdating = false;
            }
        }

        public void UpdateCache()
        {
            Dictionary<int, ApplicationActivityReporting> applicationActivityReportingCache = new Dictionary<int, ApplicationActivityReporting>();

            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                try
                {
                    conn.Open();
                }
                catch (Exception)
                {
                    Logger.Log(ELogLevel.Info, "localhost", "Unable to establish connection to database:" + DateTime.Now.ToString());
                    return;
                }

                DateTime lastQueryTime = DateTime.UtcNow;
                SqlCommand command = new SqlCommand("SELECT * FROM ApplicationActivityReporting Order By DateModified DESC", conn);
                using (SqlDataReader reader = command.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        var applicationActivityReporting = new ApplicationActivityReporting();
                        applicationActivityReporting.SetFromDataReader(reader);
                        applicationActivityReportingCache[applicationActivityReporting.ID] = applicationActivityReporting;
                    }
                }

                lock (_applicationActivityReporting)
                {
                    _applicationActivityReporting = applicationActivityReportingCache;
                }

                // Build a Application specicfic Cache
                var applicationActivityReportingByApplication = new Dictionary<string, List<ApplicationActivityReporting>>();
                foreach (ApplicationActivityReporting report in applicationActivityReportingCache.Values)
                {
                    if (!applicationActivityReportingByApplication.ContainsKey(report.ReportingApplication))
                    {
                        applicationActivityReportingByApplication[report.ReportingApplication] = new List<ApplicationActivityReporting>();
                    }

                    applicationActivityReportingByApplication[report.ReportingApplication].Add(report);
                }

                lock (applicationActivityReportingByApplication)
                {
                    _applicationActivityReportingByApplication = applicationActivityReportingByApplication;
                }

                _lastQueryTime = lastQueryTime;
            }

            if (!IsCacheBuilt)
            {
                Logger.Log("Author Cache Built", ELogLevel.Info);
            }

            IsCacheBuilt = true;
            GC.Collect();
        }

        public void InvalidateCache()
        {
            _isCacheUpdating = true;
            try
            {
                UpdateCache();
            }
            catch (Exception exc)
            {
                Logger.Log("Failed to update Author Cache Details: ", ELogLevel.Error, exc.GetExceptionDetails());
            }
            _isCacheUpdating = false;
        }

        public IEnumerable<ApplicationActivityReporting> GetApplicationActivityReportingByApplication(string application)
        {
            if (!IsCacheBuilt)
            {
                return null;
            }

            return _applicationActivityReportingByApplication.ContainsKey(application) ? _applicationActivityReportingByApplication[application] : null;
        }
    }
}