﻿using System;
using System.Linq;
using Curse.CommunityTracker.Models;
using System.Collections.Generic;
using System.Threading;
using System.Configuration;
using System.Data.SqlClient;
using System.Data;
using Curse.CommunityTracker.Extensions;

namespace Curse.CommunityTracker.Caching {
    public class CForumThreadPostCache {
        private Dictionary<int, CForumThreadPost> _threadPostCache = new Dictionary<int, CForumThreadPost>();
        private Dictionary<int, List<CForumThreadPost>> _postsByThread = new Dictionary<int, List<CForumThreadPost>>();

        private bool _isCacheBuilt = false;
        private bool _isCacheUpdating = false;
        public bool IsCacheUpdating { get { return _isCacheUpdating; } }
        public bool IsCacheBuilt { get { return _isCacheBuilt; } set { _isCacheBuilt = value; } }

        private Thread _updateThread = null;
        private int _updateThreadInterval;
        private DateTime _lastQueryTime = new DateTime(1979, 5, 17);
        private string _databaseConnectionString = string.Empty;

        private static readonly CForumThreadPostCache _instance = new CForumThreadPostCache();
        public static CForumThreadPostCache Instance {
            get { return _instance; }
        }
        public DateTime LastQueryTime {
            get { return _lastQueryTime; }
        }

        public CForumThreadPostCache() {
            _isCacheBuilt = false;
            _updateThreadInterval = int.Parse(ConfigurationManager.AppSettings["UpdateThreadInterval"]);
            _databaseConnectionString = ConfigurationManager.ConnectionStrings["CommunityTracker"].ConnectionString;

            UpdateCache();

            _updateThread = new Thread(CacheThread) { IsBackground = true };
            _updateThread.Priority = ThreadPriority.Lowest;
            _updateThread.Start();
        }

        public static void Initialize() { }

        private void CacheThread() {
            Boolean aborted = false;
            while (!aborted) {
                Thread.Sleep(_updateThreadInterval);
                GC.Collect();
                _isCacheUpdating = true;
                try {
                    UpdateCache();
                }
                catch (ThreadAbortException) {
                    aborted = true;
                    _updateThread.Join(100);
                    Logger.Log(ELogLevel.Info, null, "Thread Abort Exception. Service shutting down.");
                }
                catch (Exception ex) {

                    Logger.Log(ELogLevel.Info, null, "Update Thread Exception: {0}", ex.Message + "\n" + ex.StackTrace);
                }
                _isCacheUpdating = false;
            }
        }

        public void InvalidateCache() {
            _isCacheUpdating = true;
            try {
                UpdateCache();
            }
            catch (Exception exc) {
                Logger.Log("Failed to update Post Cache Details: ", ELogLevel.Error, exc.GetExceptionDetails());
            }
            _isCacheUpdating = false;
        }

        public void UpdateCache() {
            Dictionary<int, CForumThreadPost> threadPostCache = new Dictionary<int, CForumThreadPost>();

            using (SqlConnection conn = new SqlConnection(_databaseConnectionString)) {
                try {
                    conn.Open();
                }
                catch (Exception) {
                    Logger.Log(ELogLevel.Info, "localhost", "Unable to establish connection to database:" + DateTime.Now.ToString());
                    return;
                }

                DateTime lastQueryTime = DateTime.UtcNow;
                SqlCommand command = new SqlCommand("SELECT CommunityForumPost.*, CommunityAuthor.Name as Author, CommunityAuthor.AvatarUrl as Avatar FROM CommunityForumPost with(nolock) INNER JOIN CommunityAuthor with(nolock) ON CommunityForumPost.CommunityAuthorID = CommunityAuthor.ID where CommunityForumPost.DateModified >= @datemodified", conn);
                command.Parameters.AddWithValue("dateModified", _lastQueryTime.AddMinutes(-30));
                
                using (SqlDataReader reader = command.ExecuteReader()) {
                    while (reader.Read()) {
                        var post = new CForumThreadPost();
                        post.SetFromDataReader(reader);

                        threadPostCache[post.ID] = post;
                    }
                }

                lock (_threadPostCache) {
                    foreach (CForumThreadPost post in threadPostCache.Values)
                    {
                        _threadPostCache[post.ID] = post;    
                    }
                }

                // Build a source specific cache
                var postsByThread = new Dictionary<int, List<CForumThreadPost>>();
                foreach (CForumThreadPost post in threadPostCache.Values) {
                    if (!postsByThread.ContainsKey(post.ThreadID)) {
                        postsByThread[post.ThreadID] = new List<CForumThreadPost>();
                    }

                    postsByThread[post.ThreadID].Add(post);
                }

                lock (_postsByThread) {
                    _postsByThread = postsByThread;
                }

                _lastQueryTime = lastQueryTime;
            }

            if (!IsCacheBuilt) {
                Logger.Log("CommunityForumThread Cache Built", ELogLevel.Info);
            }

            IsCacheBuilt = true;
            GC.Collect();
        }

        public CForumThreadPost GetPostById(int id) {
            if (!_isCacheBuilt) {
                return null;
            }

            if (_threadPostCache.ContainsKey(id)) {
                return _threadPostCache[id];
            }
            else {
                return null;
            }
        }

        public CForumThreadPost GetPostById(int id, DateTime lastUpdated) {
            if (!_isCacheBuilt) {
                return null;
            }

            //Invalidate the cache if it has been longer that the last query time
            if (LastQueryTime < lastUpdated) {
                InvalidateCache();
            }

            if (_threadPostCache.ContainsKey(id)) {
                return _threadPostCache[id];
            }
            else {
                return null;
            }
        }

        public IEnumerable<CForumThreadPost> GetPostsByThreadId(int id, DateTime since) {
            if (!IsCacheBuilt) {
                return null;
            }

            if (_postsByThread.ContainsKey(id)) {
                return _postsByThread[id].Where(p => p.DateCreated >= since || p.DateModified >= since);
            }
            else {
                return null;
            }
        }

        public IEnumerable<CForumThreadPost> GetPostByDataSourceIdLastX(int id, int rowsToGet)
        {
            List<CForumThreadPost> postByDataSource = new List<CForumThreadPost>();
            var query = @"SELECT TOP(@rowsToGet) CommunityForumPost.*, CommunityAuthor.Name as Author, CommunityAuthor.AvatarUrl as Avatar 
                            FROM CommunityForumPost with(nolock) 
                            INNER JOIN CommunityAuthor with(nolock) ON CommunityForumPost.CommunityAuthorID = CommunityAuthor.ID 
                            WHERE CommunityForumPost.CommunityDataSourceID = @id";
            
            using (var conn = new SqlConnection(_databaseConnectionString))
            {
                var cmd = new SqlCommand(query, conn);
                cmd.Parameters.AddWithValue("@id", id);
                cmd.Parameters.AddWithValue("@rowsToGet", rowsToGet);
                conn.Open();
                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                   
                    while (reader.Read())
                    {
                        var post = new CForumThreadPost();
                        post.SetFromDataReader(reader);
                        postByDataSource.Add(post);
                    }
                }
            }

            return postByDataSource;
        } 
    }
}