﻿using System;
using System.Collections.Generic;
using Curse.CommunityTracker.Models;
using System.Threading;
using System.Configuration;
using System.Data.SqlClient;
using Curse.CommunityTracker.Extensions;

namespace Curse.CommunityTracker.Caching
{
    public class CpuMonitorCache
    {
        private Dictionary<int, CpuMonitor> _cpuMonitor = new Dictionary<int, CpuMonitor>();
        private Dictionary<string, List<CpuMonitor>> _cpuMonitorByApplication = new Dictionary<string, List<CpuMonitor>>();
        private Dictionary<string, List<CpuMonitor>> _cpuMonitorByHost = new Dictionary<string, List<CpuMonitor>>();

        private bool _isCacheBuilt = false;
        private bool _isCacheUpdating = false;
        public bool IsCacheUpdating { get { return _isCacheUpdating; } }
        public bool IsCacheBuilt { get { return _isCacheBuilt; } set { _isCacheBuilt = value; } }

        private readonly Thread _updateThread = null;
        private readonly int _updateThreadInterval;
        private DateTime _lastQueryTime = new DateTime(1975,12,12);
        private readonly string _databaseConnectionString = string.Empty;

        private static readonly CpuMonitorCache _instance = new CpuMonitorCache();
        public static CpuMonitorCache Instance { get { return _instance; } }
        public DateTime LastQueryTime { get { return _lastQueryTime; } }
        
        public CpuMonitorCache() 
        { 
            _isCacheBuilt = false;
            _updateThreadInterval = int.Parse(ConfigurationManager.AppSettings["UpdateThreadInterval"]);
            _databaseConnectionString = ConfigurationManager.ConnectionStrings["CommunityTracker"].ConnectionString;

            UpdateCache();

            _updateThread = new Thread(CacheThread) {IsBackground = true, Priority = ThreadPriority.Lowest};
            _updateThread.Start();
        }

        public static void Initialize(){ }

        private void CacheThread()
        {
            Boolean aborted = false;
            while (!aborted)
            {
                Thread.Sleep(_updateThreadInterval);
                GC.Collect();
                _isCacheUpdating = true;
                try
                {
                    UpdateCache();
                }
                catch (ThreadAbortException)
                {
                    aborted = true;
                    _updateThread.Join(100);
                    Logger.Log(ELogLevel.Info, null, "Thread Abort Exception. Service shutting down.");
                }
                catch (Exception ex)
                {

                    Logger.Log(ELogLevel.Info, null, "Update Thread Exception: {0}", ex.Message + "\n" + ex.StackTrace);
                }
                _isCacheUpdating = false;
            }
        }

        public void UpdateCache()
        {
            Dictionary<int, CpuMonitor> cpuMonitorCache = new Dictionary<int, CpuMonitor>();

            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                try
                {
                    conn.Open();
                }
                catch (Exception)
                {
                    Logger.Log(ELogLevel.Info, "localhost", "Unable to establish connection to database:" + DateTime.Now.ToString());
                    return;
                }

                int getTopRows = 1;
                SqlCommand countAppsCmd = new SqlCommand("SELECT COUNT(DISTINCT ReportingApplication) FROM CpuMonitor", conn);
                int? multiplier = (int?)countAppsCmd.ExecuteScalar();
                if (multiplier.HasValue)
                {
                    getTopRows = (int)multiplier * int.Parse(ConfigurationManager.AppSettings["NumberOfCpuMonitorRows"]);
                }


                DateTime lastQueryTime = DateTime.UtcNow;
                // just getting last days worth (24 hours * 60 * number of applications, cause it inserts a new roe every minuet)
                SqlCommand command = new SqlCommand("SELECT TOP(@TopRows) * FROM CpuMonitor Order By DateCreated DESC" , conn);
                command.Parameters.AddWithValue("@TopRows", getTopRows);
                using (SqlDataReader reader = command.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        var cpuMonitor = new CpuMonitor();
                        cpuMonitor.SetFromDataReader(reader);
                        cpuMonitorCache[cpuMonitor.ID] = cpuMonitor;
                    }
                }

                lock (_cpuMonitor)
                {
                    _cpuMonitor = cpuMonitorCache;
                }

                // Build a Application specicfic Cache and by Host
                var cpuMonitorsByApplication = new Dictionary<string, List<CpuMonitor>>();
                var cpuMonitorByHost = new Dictionary<string, List<CpuMonitor>>();
                foreach (CpuMonitor monitor in cpuMonitorCache.Values)
                {
                    if (!cpuMonitorsByApplication.ContainsKey(monitor.ReportingApplication))
                    {
                        cpuMonitorsByApplication[monitor.ReportingApplication] = new List<CpuMonitor>();
                    }

                    cpuMonitorsByApplication[monitor.ReportingApplication].Add(monitor);

                    if (!cpuMonitorByHost.ContainsKey(monitor.HostName))
                    {
                        cpuMonitorByHost[monitor.HostName] = new List<CpuMonitor>();
                    }

                    cpuMonitorByHost[monitor.HostName].Add(monitor);
                }

                lock (cpuMonitorsByApplication)
                {
                    _cpuMonitorByApplication = cpuMonitorsByApplication;
                }

                lock (cpuMonitorByHost)
                {
                    _cpuMonitorByHost = cpuMonitorByHost;
                }

                _lastQueryTime = lastQueryTime;
            }

            if (!IsCacheBuilt)
            {
                Logger.Log("Author Cache Built", ELogLevel.Info);
            }

            IsCacheBuilt = true;
            GC.Collect();
        }

        public void InvalidateCache()
        {
            _isCacheUpdating = true;
            try
            {
                UpdateCache();
            }
            catch (Exception exc)
            {
                Logger.Log("Failed to update Author Cache Details: ", ELogLevel.Error, exc.GetExceptionDetails());
            }
            _isCacheUpdating = false;
        }

        public IEnumerable<CpuMonitor> GetCpuMonitorByApplication(string application)
        {
            if (!IsCacheBuilt)
            {
                return null;
            }

            return _cpuMonitorByApplication.ContainsKey(application) ? _cpuMonitorByApplication[application] : null;
        }

        public IEnumerable<CpuMonitor> GetCpuMonitorByHost(string host)
        {
            if (!IsCacheBuilt)
            {
                return null;
            }
            return _cpuMonitorByHost.ContainsKey(host) ? _cpuMonitorByHost[host] : null;
        }

        public CpuMonitor GetLatestCpuMonitorByHost(string host)
        {
            if (!IsCacheBuilt)
            {
                return null;
            }
            return _cpuMonitorByHost.ContainsKey(host) ? _cpuMonitorByHost[host][0] : null;
        }

        public IEnumerable<CpuMonitor> GetCpuMonitorByHostLastX(string host, int rowsToGet)
        {
            if (!IsCacheBuilt)
            {
                return null;
            }
            int cpuMonitorCount = _cpuMonitorByHost.ContainsKey(host) ? _cpuMonitorByHost[host].Count : 0;
            if (rowsToGet > cpuMonitorCount)
            {
                rowsToGet = cpuMonitorCount;
            }
            return _cpuMonitorByHost.ContainsKey(host) ? _cpuMonitorByHost[host].GetRange(0, rowsToGet) : null;
        }
    }
}