﻿using System;
using Curse.CommunityTracker.Models;
using System.Collections.Generic;
using System.Threading;
using System.Configuration;
using System.Data.SqlClient;
using Curse.CommunityTracker.Extensions;

namespace Curse.CommunityTracker.Caching
{
    public class ELMAH_ErrorCache
    {
        private Dictionary<Guid, ELMAH_Error> _elmahError = new Dictionary<Guid, ELMAH_Error>();
        private Dictionary<string, List<ELMAH_Error>> _elmahErrorsByHost = new Dictionary<string, List<ELMAH_Error>>();

        private bool _isCacheBuilt = false;
        private bool _isCacheUpdating = false;
        public bool IsCacheUpdating { get { return _isCacheUpdating; } }
        public bool IsCacheBuilt { get { return _isCacheBuilt; } set { _isCacheBuilt = value; } }

        private readonly Thread _updateThread = null;
        private readonly int _updateThreadInterval;
        private readonly int _numberOfRowsToFetch;
        private DateTime _lastQueryTime = new DateTime(1975, 12, 12);
        private readonly string _databaseConnectionString = string.Empty;

        private static readonly ELMAH_ErrorCache _instance = new ELMAH_ErrorCache();
        public static ELMAH_ErrorCache Instance { get { return _instance; } }
        public DateTime LastQueryTime { get { return _lastQueryTime; } }

        public ELMAH_ErrorCache()
        {
            _isCacheBuilt = false;
            _updateThreadInterval = int.Parse(ConfigurationManager.AppSettings["UpdateThreadInterval"]);
            _numberOfRowsToFetch = int.Parse(ConfigurationManager.AppSettings["NumberOfErrorsToReturn"]);
            _databaseConnectionString = ConfigurationManager.ConnectionStrings["CommunityTracker"].ConnectionString;

            UpdateCache();

            _updateThread = new Thread(CacheThread) { IsBackground = true, Priority = ThreadPriority.Lowest };
            _updateThread.Start();
        }

        public static void Initialize() { }

        private void CacheThread()
        {
            Boolean aborted = false;
            while (!aborted)
            {
                Thread.Sleep(_updateThreadInterval);
                GC.Collect();
                _isCacheUpdating = true;
                try
                {
                    UpdateCache();
                }
                catch (ThreadAbortException)
                {
                    aborted = true;
                    _updateThread.Join(100);
                    Logger.Log(ELogLevel.Info, null, "Thread Abort Exception. Service shutting down.");
                }
                catch (Exception ex)
                {

                    Logger.Log(ELogLevel.Info, null, "Update Thread Exception: {0}", ex.Message + "\n" + ex.StackTrace);
                }
                _isCacheUpdating = false;
            }
        }

        public void UpdateCache()
        {
            Dictionary<Guid, ELMAH_Error> elmahErrorCache = new Dictionary<Guid, ELMAH_Error>();

            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                try
                {
                    conn.Open();
                }
                catch (Exception)
                {
                    Logger.Log(ELogLevel.Info, "localhost", "Unable to establish connection to database:" + DateTime.Now.ToString());
                    return;
                }

                DateTime lastQueryTime = DateTime.UtcNow;
                SqlCommand command = new SqlCommand("SELECT TOP(@TopRows) * FROM ELMAH_Error ORDER BY TimeUtc DESC", conn);
                command.Parameters.AddWithValue("@TopRows", _numberOfRowsToFetch);
                using (SqlDataReader reader = command.ExecuteReader()) 
                {
                    while (reader.Read())
                    {
                        var elmahError = new ELMAH_Error();
                        elmahError.SetFromDataReader(reader);
                        elmahErrorCache[elmahError.ErrorId] = elmahError;
                    }
                }

                lock (_elmahError)
                {
                    _elmahError = elmahErrorCache;
                }
            }

            // Build Host specific Cache
            var elmahErrorsByHost = new Dictionary<string, List<ELMAH_Error>>();
            foreach (ELMAH_Error error in elmahErrorCache.Values)
            {
                if (!elmahErrorsByHost.ContainsKey(error.Host))
                {
                    elmahErrorsByHost[error.Host] = new List<ELMAH_Error>();
                }

                elmahErrorsByHost[error.Host].Add(error);

                lock (elmahErrorsByHost)
                {
                    _elmahErrorsByHost = elmahErrorsByHost;
                }

                _lastQueryTime = LastQueryTime;
            }

            if (!IsCacheBuilt)
            {
                Logger.Log("Author Cache Built", ELogLevel.Info);
            }

            IsCacheBuilt = true;
            GC.Collect();
        }

        public void InvalidateCache()
        {
            _isCacheUpdating = true;
            try
            {
                UpdateCache();
            }
            catch (Exception exc)
            {
                Logger.Log("Failed to update Author Cache Details: ", ELogLevel.Error, exc.GetExceptionDetails());
            }
            _isCacheUpdating = false;
        }

        public IEnumerable<ELMAH_Error> GetElmahErrorsByHost(string host)
        {
            if (!IsCacheBuilt)
            {
                return null;
            }

            return _elmahErrorsByHost.ContainsKey(host) ? _elmahErrorsByHost[host] : null;
        }
    }
}