﻿using System;
using System.Linq;
using Curse.CommunityTracker.Models;
using System.Collections.Generic;
using System.Threading;
using System.Configuration;
using System.Data.SqlClient;
using System.Data;
using Curse.CommunityTracker.Extensions;

namespace Curse.CommunityTracker.Caching {
    public class GameCache {
        private Dictionary<int, Game> _gameCache = new Dictionary<int, Game>();
        
        private bool _isCacheBuilt = false;
        private bool _isCacheUpdating = false;
        public bool IsCacheUpdating { get { return _isCacheUpdating; } }
        public bool IsCacheBuilt { get { return _isCacheBuilt; } set { _isCacheBuilt = value; } }

        private Thread _updateThread = null;
        private int _updateThreadInterval;
        private string _databaseConnectionString = string.Empty;

        private static readonly GameCache _instance = new GameCache();
        public static GameCache Instance {
            get { return _instance; }
        }

        public GameCache() {
            _isCacheBuilt = false;
            _updateThreadInterval = int.Parse(ConfigurationManager.AppSettings["UpdateThreadInterval"]);
            _databaseConnectionString = ConfigurationManager.ConnectionStrings["CommunityTracker"].ConnectionString;

            UpdateCache();

            _updateThread = new Thread(CacheThread) { IsBackground = true };
            _updateThread.Priority = ThreadPriority.Lowest;
            _updateThread.Start();
        }

        public static void Initialize() { }

        private void CacheThread() {
            Boolean aborted = false;
            while (!aborted) {
                Thread.Sleep(_updateThreadInterval);
                GC.Collect();
                _isCacheUpdating = true;
                try {
                    UpdateCache();
                }
                catch (ThreadAbortException) {
                    aborted = true;
                    _updateThread.Join(100);
                    Logger.Log(ELogLevel.Info, null, "Thread Abort Exception. Service shutting down.");
                }
                catch (Exception ex) {

                    Logger.Log(ELogLevel.Info, null, "Update Thread Exception: {0}", ex.Message + "\n" + ex.StackTrace);
                }
                _isCacheUpdating = false;
            }
        }

        public void UpdateCache() {
            Dictionary<int, Game> gameCache = new Dictionary<int, Game>();

            using (SqlConnection conn = new SqlConnection(_databaseConnectionString)) {
                try {
                    conn.Open();
                }
                catch (Exception) {
                    Logger.Log(ELogLevel.Info, "localhost", "Unable to establish connection to database:" + DateTime.Now.ToString());
                    return;
                }

                SqlCommand command = new SqlCommand("select * from Game", conn);
                using (SqlDataReader reader = command.ExecuteReader()) {
                    while (reader.Read()) {
                        var game = new Game();
                        game.SetFromDataReader(reader);

                        gameCache[game.ID] = game;
                    }
                }

                lock (_gameCache) {
                    _gameCache = gameCache;
                }
            }

            if (!IsCacheBuilt) {
                Logger.Log("CommunityForumThread Cache Built", ELogLevel.Info);
            }

            IsCacheBuilt = true;
            GC.Collect();
        }

        public IEnumerable<Game> GetAllGames() {
            if (!IsCacheBuilt) {
                return null;
            }

            return _gameCache.Values;
        }
    }
}