﻿using System;
using System.Collections;
using System.Collections.Generic;
using Curse.CommunityTracker.Models;
using System.Threading;
using System.Configuration;
using System.Data.SqlClient;
using Curse.CommunityTracker.Extensions;

namespace Curse.CommunityTracker.Caching
{
    public class HostCache
    {
        private Dictionary<int, DataSource> _dataSource = new Dictionary<int, DataSource>();
        private Dictionary<string, HostList> _hostWithDataSources = new Dictionary<string, HostList>();

        private bool _isCacheBuilt = false;
        private bool _isCacheUpdating = false;
        public bool IsCacheUpdating { get { return _isCacheUpdating; } }
        public bool IsCacheBuilt { get { return _isCacheBuilt; } set { _isCacheBuilt = value; } }

        private readonly Thread _updateThread = null;
        private readonly int _updateThreadInterval;
        private DateTime _lastQueryTime = new DateTime(1975, 12, 12);
        private readonly string _databaseConnectionString = string.Empty;

        private static readonly HostCache _instance = new HostCache();
        public static HostCache Instance { get { return _instance; } }
        public DateTime LastQueryTime { get { return _lastQueryTime; } }

        public HostCache()
        {
            _isCacheBuilt = false;
            _updateThreadInterval = int.Parse(ConfigurationManager.AppSettings["UpdateThreadInterval"]);
            _databaseConnectionString = ConfigurationManager.ConnectionStrings["CommunityTracker"].ConnectionString;

            UpdateCache();

            _updateThread = new Thread(CacheThread) { IsBackground = true, Priority = ThreadPriority.Lowest };
            _updateThread.Start();
        }

        public static void Initialize() { }

        private void CacheThread()
        {
            Boolean aborted = false;
            while (!aborted)
            {
                Thread.Sleep(_updateThreadInterval);
                GC.Collect();
                _isCacheUpdating = true;
                try
                {
                    UpdateCache();
                }
                catch (ThreadAbortException)
                {
                    aborted = true;
                    _updateThread.Join(100);
                    Logger.Log(ELogLevel.Info, null, "Thread Abort Exception. Service shutting down.");
                }
                catch (Exception ex)
                {

                    Logger.Log(ELogLevel.Info, null, "Update Thread Exception: {0}", ex.Message + "\n" + ex.StackTrace);
                }
                _isCacheUpdating = false;
            }
        }

        public void UpdateCache()
        {
            Dictionary<int, DataSource> hostCache = new Dictionary<int, DataSource>();

            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                try
                {
                    conn.Open();
                }
                catch (Exception)
                {
                    Logger.Log(ELogLevel.Info, "localhost", "Unable to establish connection to database:" + DateTime.Now.ToString());
                    return;
                }

                DateTime lastQueryTime = DateTime.UtcNow;
                SqlCommand command = new SqlCommand("SELECT ID, Name, LastRunTimeUTC, Status, HostName FROM CommunityDataSource", conn);
                using (SqlDataReader reader = command.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        var dataSource = new DataSource();
                        dataSource.SetFromDataReader(reader);
                        hostCache[dataSource.ID] = dataSource;
                    }
                }

                lock (_dataSource)
                {
                    _dataSource = hostCache;
                }

                // Build a Hosted on specicfic Cache
                var hostWithDataSources = new Dictionary<string, HostList>();
                
                foreach (DataSource dataSourceHost in hostCache.Values)
                {
                    if (!hostWithDataSources.ContainsKey(dataSourceHost.HostName))
                    {
                        hostWithDataSources[dataSourceHost.HostName] = new HostList {DataSources = new List<DataSource>(), CpuMonitor = new CpuMonitor(), ElmahErrors = new List<ELMAH_Error>()};
                    }
                    hostWithDataSources[dataSourceHost.HostName].HostName = dataSourceHost.HostName;
                    hostWithDataSources[dataSourceHost.HostName].DataSources.Add(dataSourceHost);
                    
                }
                foreach (var hostWithDataSource in hostWithDataSources)
                {
                    CpuMonitor cpuMonitor = CpuMonitorCache.Instance.GetLatestCpuMonitorByHost(hostWithDataSource.Key) ?? new CpuMonitor();
                    hostWithDataSources[hostWithDataSource.Key].CpuMonitor = cpuMonitor;
                    IEnumerable<ELMAH_Error> elmahErrorList = ELMAH_ErrorCache.Instance.GetElmahErrorsByHost(hostWithDataSource.Key) ?? new ELMAH_Error[0];
                    hostWithDataSources[hostWithDataSource.Key].ElmahErrors.AddRange(elmahErrorList);
                }
                lock (hostWithDataSources)
                {
                    _hostWithDataSources = hostWithDataSources;
                }

                _lastQueryTime = lastQueryTime;
            }

            if (!IsCacheBuilt)
            {
                Logger.Log("Author Cache Built", ELogLevel.Info);
            }

            IsCacheBuilt = true;
            GC.Collect();
        }

        public void InvalidateCache()
        {
            _isCacheUpdating = true;
            try
            {
                UpdateCache();
            }
            catch (Exception exc)
            {
                Logger.Log("Failed to update Author Cache Details: ", ELogLevel.Error, exc.GetExceptionDetails());
            }
            _isCacheUpdating = false;
        }

        public IEnumerable<DataSource> GetAllDataSourceWithHostName()
        {
            if (!IsCacheBuilt)
            {
                return null;
            }

            return _dataSource.Values;
        }

        public IEnumerable<HostList> GetAllHostsWithDataLists()
        {
            if (!IsCacheBuilt)
            {
                return null;
            }
            
            return _hostWithDataSources.Values;
        }

        public bool ChangeDataSourceHost(int id, string host)
        {
            int rowschanged;
            using (SqlConnection conn = new SqlConnection(_databaseConnectionString))
            {
                try
                {
                    conn.Open();
                }
                catch (Exception)
                {
                    Logger.Log(ELogLevel.Info, "localhost", "Unable to establish connection to database:" + DateTime.Now.ToString());
                    return false;
                }

                SqlCommand command = new SqlCommand("UPDATE CommunityDataSource SET [HostName] = @HostName WHERE ID = @ID", conn);
                command.Parameters.AddWithValue("@HostName", host);
                command.Parameters.AddWithValue("@ID", id);
                rowschanged = command.ExecuteNonQuery();
            }

            return rowschanged == 1;
        }
    }
}