﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.CommunityTracker.Models;
using Curse.CommunityTracker.Extensions;
using Curse.CommunityTracker.Caching;

namespace Curse.CommunityTracker
{
    public class CommunityTracker : ICommunityTracker
    {
        private static object _startLock = new object();

        static CommunityTracker()
        {
            
            lock (_startLock)
            {
                try
                {

                    Logger.Log("Initializing Data Source Cache...", ELogLevel.Info);
                    ForumDataSourceCache.Initialize();

                    Logger.Log("Initializing Author Cache...", ELogLevel.Info);
                    CAuthorCache.Initialize();

                    Logger.Log("Initializing Forum Cache...", ELogLevel.Info);
                    CForumCache.Initialize();

                    Logger.Log("Initializing Thread Cache...", ELogLevel.Info);
                    CForumThreadCache.Initialize();

                    Logger.Log("Initializing Post Cache...", ELogLevel.Info);
                    CForumThreadPostCache.Initialize();

                    Logger.Log("Service Started", ELogLevel.Info);
                }
                catch (Exception ex)
                {
                    Logger.Log("Could not start CommunityTracker Service! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
                    throw;
                }
            }
        }

        public ForumDataSource[] GetAllDataSources()
        {
            return ForumDataSourceCache.Instance.GetAllForumDataSources().ToArray();
        }

        public HostList[] GetAllHostsWithDataSourceList()
        {
            return HostCache.Instance.GetAllHostsWithDataLists().ToArray();
        }

        public ForumDataSource[] GetDataSourceByGameId(int id)
        {
            //int idValue = ValidateInt(id);
            var sources = ForumDataSourceCache.Instance.GetDataSourcesByGameId(id);
            if (sources != null)
            {
                return sources.ToArray();
            }

            return new ForumDataSource[0];
        }

        public ForumDataSource[] GetDataSourceByLanguage(int gameId, string language)
        {
            var sources = ForumDataSourceCache.Instance.GetDataSourcesByLanguage(gameId, language);
            if (sources != null)
            {
                return sources.ToArray();
            }

            return new ForumDataSource[0];
        }

        public CForum[] GetForumsBySourceId(int id)
        {
            var forums = CForumCache.Instance.GetForumsBySourceId(id);
            if (forums != null)
            {
                return forums.ToArray();
            }

            return new CForum[0];
        }

        public CForumThread[] GetThreadsByForumId(int id, long since)
        {
            var threads = CForumThreadCache.Instance.GetThreadsByForumId(id, since.ToDateTime());
            if (threads != null)
            {
                return threads.ToArray();
            }

            return new CForumThread[0];
        }

        public CForumThreadPost[] GetPostsByThreadId(int id, long since)
        {
            var posts = CForumThreadPostCache.Instance.GetPostsByThreadId(id, since.ToDateTime());
            if (posts != null)
            {
                return posts.ToArray();
            }

            return new CForumThreadPost[0];
        }

        public CForumThreadPost[] GetPostByDataSourceId(int id, int rowsToGet)
        {
            var postList = CForumThreadPostCache.Instance.GetPostByDataSourceIdLastX(id, rowsToGet);
            return postList != null ? postList.ToArray() : new CForumThreadPost[0];
        }

        public UpdatedThread[] GetAllActivitySince(int dataSourceId, long since)
        {
            return GetAllActivitySince(new int[] { dataSourceId }, since);
        }

        public UpdatedThread[] GetAllActivitySince(int[] ids, long since)
        {
            var updatedThreads = CForumThreadCache.Instance.GetRecentThreadActivity(ids, since.ToDateTime());
            if (updatedThreads != null)
            {
                return updatedThreads.ToArray();
            }

            return new UpdatedThread[0];
        }

        public DataSourceActivity[] GetAllActivitySinceFull(int[] dataSourceIds, long since)
        {
            IEnumerable<DataSourceActivity> dataSourceActivity = ForumDataSourceCache.Instance.GetAllRecentActivity(dataSourceIds, since.ToDateTime());

            if (dataSourceActivity != null)
            {
                return dataSourceActivity.ToArray();
            }

            return new DataSourceActivity[0];
        }

        public CAuthor[] GetAuthorsByDataSourceId(int id, long since)
        {
            return CAuthorCache.Instance.GetAuthorsByDataSourceID(id, since.ToDateTime()).ToArray();
        }

        public ELMAH_Error[] GetElmahErrorsByHost(string host)
        {
            IEnumerable<ELMAH_Error> elmahError = ELMAH_ErrorCache.Instance.GetElmahErrorsByHost(host);
            return elmahError != null ? elmahError.ToArray() : new ELMAH_Error[0];
        }

        public bool ChangeDataSourceHost(int id, string host)
        {
            return HostCache.Instance.ChangeDataSourceHost(id, host); 
        }

        public CpuMonitor[] GetCpuMonitorByHostLastX(string host, int rowsToGet)
        {
            IEnumerable<CpuMonitor> cpuMonitor = CpuMonitorCache.Instance.GetCpuMonitorByHostLastX(host, rowsToGet);
            return cpuMonitor != null ? cpuMonitor.ToArray() : new CpuMonitor[0];
        }


        // these calls are ot used but potentially might be used later so just commenting them out

        //public DataSource[] GetAllDataSourceWithHostName()
        //{
        //    return HostCache.Instance.GetAllDataSourceWithHostName().ToArray();
        //}

        //public UpdatedThread[] GetDataSourceActivityLastX(int[] dataSourceIds, int rowsToGet)
        //{
        //    var dataSourceActivity = CForumThreadCache.Instance.GetThreadsByDataSourceIdLastX(dataSourceIds, rowsToGet);
        //    return dataSourceActivity != null ? dataSourceActivity.ToArray() : new UpdatedThread[0];
        //}

        //public ApplicationActivityReporting[] GetApplicationActivityReportingByApplication(string application)
        //{
        //    IEnumerable<ApplicationActivityReporting> applicationActivityReporting = ApplicationActivityReportingCache.Instance.GetApplicationActivityReportingByApplication(application);
        //    return applicationActivityReporting != null ? applicationActivityReporting.ToArray() : new ApplicationActivityReporting[0];
        //}

        //public CpuMonitor[] GetCpuMonitorByApplication(string application)
        //{
        //    IEnumerable<CpuMonitor> cpuMonitor = CpuMonitorCache.Instance.GetCpuMonitorByApplication(application);
        //    return cpuMonitor != null ? cpuMonitor.ToArray() : new CpuMonitor[0];
        //}

        //public CpuMonitor[] GetCpuMonitorByHost(string host)
        //{
        //    IEnumerable<CpuMonitor> cpuMonitor = CpuMonitorCache.Instance.GetCpuMonitorByHost(host);
        //    return cpuMonitor != null ? cpuMonitor.ToArray() : new CpuMonitor[0];
        //}

        //public CpuMonitor GetLatestCpuMonitorByHost(string host)
        //{
        //    return CpuMonitorCache.Instance.GetLatestCpuMonitorByHost(host) ?? new CpuMonitor();
        //}
    }
}
