﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using NUnit.Framework;
using CommunityTrackerUnitTests.CommunityTracker;
using System.Data.SqlClient;
using System.Configuration;

namespace CommunityTrackerUnitTests {
    [TestFixture]
    class SoapTests {
        CommunityTrackerClient _client;
        string _connectionString;

        [SetUp]
        public void Init() {
            _client = new CommunityTrackerClient();
            _connectionString = ConfigurationManager.ConnectionStrings["CommunityTracker"].ConnectionString;
        }

        [Test]
        public void GetAllGames() {
            game[] games = _client.GetAllGames();
             
            using (var conn = new SqlConnection(_connectionString)) {
                var cmd = conn.CreateCommand();
                cmd.CommandText = "select count(0) from Game";

                conn.Open();
                var count = (int)cmd.ExecuteScalar();
                Assert.AreEqual(count, games.Length);
            }
        }

        [Test]
        public void GetAllDataSources() {
            ForumDataSource[] dataSources = _client.GetAllDataSources();

            using (var conn = new SqlConnection(_connectionString)) {
                var cmd = conn.CreateCommand();
                cmd.CommandText = "select count(0) from CommunityDataSource";

                conn.Open();
                var count = (int)cmd.ExecuteScalar();
                Assert.AreEqual(count, dataSources.Length);
            }
        }

        [Test]
        public void GetDataSourceByGameId() {
            var games = _client.GetAllGames();
            
            var sourceCount = 0;
            foreach (var game in games) {
                var dataSource = _client.GetDataSourceByGameId(game.id);
                if(dataSource != null){
                    sourceCount += dataSource.Length;
                }
            }

            using (var conn = new SqlConnection(_connectionString)) {
                var cmd = conn.CreateCommand();
                cmd.CommandText = "select count(0) from CommunityDataSource";

                conn.Open();
                var count = (int)cmd.ExecuteScalar();
                Assert.AreEqual(count, sourceCount);
            }
        }

        [Test]
        public void GetDataSourceByLanguage() {
            var dataSources = _client.GetDataSourceByLanguage(1, "en-US");
            Assert.AreEqual(1, dataSources.Length);
        }

        [Test]
        public void GetForumsBySourceId() {
            var dataSources = _client.GetDataSourceByGameId(1);

            var forumCount = 0;
            foreach(var source in dataSources){
                var forums = _client.GetForumsBySourceId(source.id);
                if (forums != null) {
                    forumCount += forums.Length;
                }
            }

            using (var conn = new SqlConnection(_connectionString)) {
                var cmd = conn.CreateCommand();
                cmd.CommandText = "select count(0) from CommunityForum where CommunityDataSourceID in (select ID from CommunityDataSource where GameID = 1)";

                conn.Open();
                var count = (int)cmd.ExecuteScalar();
                Assert.AreEqual(count, forumCount);
            }
        }

        [Test]
        public void GetThreadsByForumId() {
            var time = DateTime.Now.AddDays(-30);
            var sources = _client.GetDataSourceByGameId(1);

            var threadCount = 0;
            foreach (var source in sources) {
                var forums = _client.GetForumsBySourceId(source.id);
                foreach (var forum in forums) {
                    var threads = _client.GetThreadsByForumId(forum.id, time.ToEpochTime());
                    if (threads != null) {
                        threadCount += threads.Length;
                    }
                }
            }

            using (var conn = new SqlConnection(_connectionString)) {
                var cmd = conn.CreateCommand();
                cmd.CommandText = "select count(0) from CommunityForumThread where CommunityForumID in (select ID from CommunityForum where CommunityDataSourceID in (select ID from CommunityDataSource where GameID = 1)) and DateModified > @time";
                cmd.Parameters.Add("time", System.Data.SqlDbType.DateTime).Value = time;

                conn.Open();
                var count = (int)cmd.ExecuteScalar();
                Assert.AreEqual(count, threadCount);
            }
        }

        [Test]
        public void GetPostsByThreadId() {
            var time = DateTime.Now.AddDays(-30);
            var sources = _client.GetDataSourceByGameId(1);

            var postCount = 0;
            foreach (var source in sources) {
                var forums = _client.GetForumsBySourceId(source.id);
                foreach (var forum in forums) {
                    var threads = _client.GetThreadsByForumId(forum.id, time.ToEpochTime());
                    foreach (var thread in threads) {
                        var posts = _client.GetPostsByThreadId(thread.id, time.ToEpochTime());
                        if (posts != null) {
                            postCount += posts.Length;
                        }
                    }
                }
            }

            using (var conn = new SqlConnection(_connectionString)) {
                var cmd = conn.CreateCommand();
                cmd.CommandText = "select count(0) from CommunityForumPost where CommunityForumThreadId in (select ID from CommunityForumThread where CommunityForumID in (select ID from CommunityForum where CommunityDataSourceID in (select ID from CommunityDataSource where GameID = 1))) and DateModified > @time";
                cmd.Parameters.Add("time", System.Data.SqlDbType.DateTime).Value = time;

                conn.Open();
                var count = (int)cmd.ExecuteScalar();
                Assert.AreEqual(count, postCount);
            }
        }

        [Test]
        public void GetAllActivitySinceFull() {
            var time = DateTime.Now.AddDays(-1);
            var sources = _client.GetDataSourceByGameId(1);

            //compile my list of source id's
            int[] sourceIds = new int[sources.Length];
            for (int i = 0; i < sources.Length; i++) {
                sourceIds[i] = sources[i].id;
            }

            var activity = _client.GetAllActivitySinceFull(sourceIds, time);

            var postCount = 0;
            foreach (var source in activity) {
                foreach (var forum in source.forums) {
                    foreach (var thread in forum.threads) {
                        postCount += thread.Posts.Length;
                    }
                }
            }

            using (var conn = new SqlConnection(_connectionString)) {
                var cmd = conn.CreateCommand();
                cmd.CommandText = "select count(0) from CommunityForumPost where CommunityForumThreadId in (select ID from CommunityForumThread where CommunityForumID in (select ID from CommunityForum where CommunityDataSourceID in (select ID from CommunityDataSource where GameID = 1))) and DateModified > @time";
                cmd.Parameters.Add("time", System.Data.SqlDbType.DateTime).Value = time;

                conn.Open();
                var count = (int)cmd.ExecuteScalar();
                Assert.AreEqual(count, postCount);
            }
        }
    }

    static class DateTimeExtensions {
        public static long ToEpochTime(this DateTime date) {
            return 0;
        }
    }
}
