﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.Serialization;
using System.ServiceModel;
using System.Text;
using System.Configuration;

using Curse.Extensions;
using Curse.Billing.Extensions;
using Curse.Billing.Configuration;
using Curse.Billing.Models;
using System.Web.Caching;
using System.Web;
using Curse.Logging.Uploader;
using Vindicia;
using System.Diagnostics;
using System.Net;
using System.IO;
using Curse.Caching;
using System.Data.SqlClient;
using Curse.Logging;


namespace Curse.Billing
{

    [ServiceBehavior(Name = "BillingService",
        Namespace = "http://billingservice.curse.com/",
        AddressFilterMode = AddressFilterMode.Any,
        ConcurrencyMode = ConcurrencyMode.Multiple,
        InstanceContextMode = InstanceContextMode.PerSession)]
    public class BillingService : IBillingService
    {

        private static object _startLock = new object();

        private static readonly string _checkoutLogPath;
        private static readonly bool _jobsEnabled = true;

        static BillingService()
        {
            lock (_startLock)
            {
                Logger.Init(ConfigurationManager.AppSettings["LogPath"], "BillingService");
                Logger.Info("Billing Service Starting...");

                LogUploader.Initialize(13, ConfigurationManager.AppSettings["LogServiceUrl"], ConfigurationManager.AppSettings["LogServiceApiKey"]);

                _checkoutLogPath = ConfigurationManager.AppSettings["CheckoutLogPath"];

                string jobNodePrefence = ConfigurationManager.AppSettings["JobNodePreference"];
                if (!string.IsNullOrEmpty(jobNodePrefence) && !System.Environment.MachineName.Equals(jobNodePrefence, StringComparison.InvariantCulture))
                {
                    Logger.Info("Disabling jobs on this node. Jobs will run on " + jobNodePrefence + "because the machine name is " + System.Environment.MachineName);
                    _jobsEnabled = false;
                }
                else
                {
                    Logger.Info("Enabling jobs on this node.");
                    _jobsEnabled = true;
                }


                Logger.Info("Billing Service Starting...");

                Logger.Info("Cluster Manager Initializing...");
                try
                {
                    using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
                    {
                        ClusterManager.Instance.Initialize("BillingService", conn);
                    }
                    Logger.Info("Cache Cluster Initializing...");
                    CacheCluster.Instance.Start();
                }
                catch (Exception ex)
                {
                    Logger.Error("Unable to start cluster service.", ex.Message);
                }



                VindiciaConfiguration.Initialize();

                if (_jobsEnabled)
                {
                    BillingSync.Instance.Start();
                }

                Logger.Info("Billing Service Started");

            }
        }

        #region Coupons

        public ServiceResponse<Coupon> CreateCoupon(string referenceBillingPlanID)
        {
            try
            {
                return new ServiceResponse<Coupon>(ServiceResponseStatus.Successful, Coupon.Create(referenceBillingPlanID));
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "CreateCoupon Exception: {0}. Stack Trace: {1}");
                return new ServiceResponse<Coupon>(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        public ServiceResponse ClaimCoupon(int userID, int cobaltInstanceID, int entityTypeID, int entityID, string code, string accountID)
        {
            try
            {
                Coupon coupon = Coupon.GetByCode(code);

                if (coupon == null)
                {
                    return new ServiceResponse(ServiceResponseStatus.ClaimCoupon_CodeNotFound);
                }

                AccountInformation account = AccountInformation.GetByID(accountID);

                if (account == null)
                {
                    return new ServiceResponse(ServiceResponseStatus.AccountNotFound);
                }

                ServiceResponseStatus status = coupon.Claim(userID, cobaltInstanceID, entityTypeID, entityID, account);
                return new ServiceResponse(status);
            }
            catch (Exception ex)
            {
                Logger.Error("Unknown exception occurred during the 'Claim Coupon' operation.", ex.GetExceptionDetails());
                return new ServiceResponse(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        public ServiceResponse ValidateCoupon(string code, string accountID)
        {
            try
            {
                Coupon coupon = Coupon.GetByCode(code);

                if (coupon == null)
                {
                    return new ServiceResponse(ServiceResponseStatus.ClaimCoupon_CodeNotFound);
                }

                return new ServiceResponse(coupon.Validate(accountID));
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unknown exception occurred during the 'Validate Coupon' operation. Exception: {0}", ex.GetExceptionDetails());
                return new ServiceResponse(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        public ServiceResponse<Coupon> GetCoupon(string code)
        {
            try
            {
                Coupon coupon = Coupon.GetByCode(code);

                if (coupon == null)
                {
                    return new ServiceResponse<Coupon>(ServiceResponseStatus.ClaimCoupon_CodeNotFound);
                }

                return new ServiceResponse<Coupon>(ServiceResponseStatus.Successful, coupon);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unknown exception occurred during the 'Get Coupon' operation.");
                return new ServiceResponse<Coupon>(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        public ServiceResponse<List<Coupon>> GetPurchasedCoupons(string accountID)
        {

            try
            {
                AccountInformation account = AccountInformation.GetByID(accountID);

                if (account == null)
                {
                    return new ServiceResponse<List<Coupon>>(ServiceResponseStatus.AccountNotFound);
                }

                return new ServiceResponse<List<Coupon>>(ServiceResponseStatus.Successful, account.GetPurchasedCoupons());

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Get Purchased Coupons Exception.");
                return new ServiceResponse<List<Coupon>>(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        public ServiceResponse<List<Coupon>> GetClaimedCoupons(string accountID)
        {

            try
            {
                AccountInformation account = AccountInformation.GetByID(accountID);

                if (account == null)
                {
                    return new ServiceResponse<List<Coupon>>(ServiceResponseStatus.AccountNotFound);
                }

                return new ServiceResponse<List<Coupon>>(ServiceResponseStatus.Successful, account.GetClaimedCoupons());

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Get Claimed Coupons Exception.");
                return new ServiceResponse<List<Coupon>>(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        #endregion

        #region Products

        public ServiceResponse<ProductOffering> GetProductOffering(string productID)
        {

            try
            {
                ProductOffering offering = ProductOffering.GetByID(productID);
                if (offering == null)
                {
                    return new ServiceResponse<ProductOffering>(ServiceResponseStatus.GetProductOffering_UnknownProductID);
                }
                else
                {
                    return new ServiceResponse<ProductOffering>(ServiceResponseStatus.Successful, offering);
                }

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Get Product Offering Exception.");
                return new ServiceResponse<ProductOffering>(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        public ServiceResponse<ProductOffering[]> GetProductOfferingByGroupID(string groupID)
        {
            try
            {

                return new ServiceResponse<ProductOffering[]>(ServiceResponseStatus.Successful, ProductOffering.GetByProductGroupID(groupID));

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Get Product Offering by Group Exception.");
                return new ServiceResponse<ProductOffering[]>(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        public ServiceResponse<ProductOffering[]> GetAllProductOfferings()
        {
            try
            {
                var productOfferings = ProductOffering.GetAll();
                return new ServiceResponse<ProductOffering[]>(ServiceResponseStatus.Successful, productOfferings);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Get All Product Offerings Exception.");
                return new ServiceResponse<ProductOffering[]>(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        public ServiceResponse<ProductBillingPlan> GetProductBillingPlan(string billingPlanID)
        {

            try
            {
                ProductBillingPlan plan = ProductBillingPlan.GetByID(billingPlanID);
                if (plan == null)
                {
                    return new ServiceResponse<ProductBillingPlan>(ServiceResponseStatus.GetProductBillingPlan_UnknownBillingPlanID);
                }
                else
                {
                    return new ServiceResponse<ProductBillingPlan>(ServiceResponseStatus.Successful, plan);
                }

            }
            catch (Exception ex)
            {
                Logger.Error("Get Product Billing Plan Exception.");
                return new ServiceResponse<ProductBillingPlan>(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        #endregion

        #region Accounts

        public ServiceResponse<List<AccountEntitlement>> GetAccountEntitlements(string accountID)
        {
            try
            {
                List<AccountEntitlement> entitlements = AccountEntitlement.GetByAccountID(accountID).Where(p => p.DateExpires > DateTime.UtcNow).ToList();
                return new ServiceResponse<List<AccountEntitlement>>(ServiceResponseStatus.Successful, entitlements);

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Get Account Entitlements Exceptio.");
                return new ServiceResponse<List<AccountEntitlement>>(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        public ServiceResponse<List<AccountEntitlement>> GetExpiredAccountEntitlements(string entitlementID, DateTime dateFrom, DateTime dateTo)
        {

            try
            {

                List<AccountEntitlement> expiredAccountEntitlements = AccountEntitlement.GetExpiredByTypeAndDate(entitlementID, dateFrom, dateTo);
                return new ServiceResponse<List<AccountEntitlement>>(ServiceResponseStatus.Successful, expiredAccountEntitlements);

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Get Expired Account Entitlements Exception.");
                return new ServiceResponse<List<AccountEntitlement>>(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        public ServiceResponse<List<AccountEntitlement>> v2GetExpiredAccountEntitlements(string entitlementID)
        {
            try
            {

                List<AccountEntitlement> expiredAccountEntitlements = AccountEntitlement.GetExpiredByType(entitlementID);
                return new ServiceResponse<List<AccountEntitlement>>(ServiceResponseStatus.Successful, expiredAccountEntitlements);

            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Get Expired Account Entitlements Exception.");
                return new ServiceResponse<List<AccountEntitlement>>(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        public ServiceResponse CreateOrUpdateAccount(AccountInformation accountInformation)
        {
            try
            {
                accountInformation.Update();
                return new ServiceResponse(ServiceResponseStatus.Successful);
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Create or Update Account Exception.");
                return new ServiceResponse(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        public ServiceResponse<AccountPaymentMethod> CreateOrUpdatePaymentMethod(string accountID, AccountPaymentMethod paymentMethod)
        {
            try
            {
                AccountInformation accountInformation = AccountInformation.GetByID(accountID);
                if (accountInformation == null)
                {
                    return new ServiceResponse<AccountPaymentMethod>(ServiceResponseStatus.AccountNotFound);
                }
                accountInformation.CreateOrUpdatePaymentMethod(paymentMethod);
                return new ServiceResponse<AccountPaymentMethod>(ServiceResponseStatus.Successful, paymentMethod);
            }
            catch (Exception ex)
            {
                Logger.Error("Create or Update Payment Method Exception.");
                return new ServiceResponse<AccountPaymentMethod>(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        public ServiceResponse<AccountInformation> GetAccountInformation(string accountID)
        {
            try
            {
                AccountInformation accountInformation = AccountInformation.GetByID(accountID);

                if (accountInformation == null)
                {
                    return new ServiceResponse<AccountInformation>(ServiceResponseStatus.AccountNotFound);
                }
                else
                {
                    return new ServiceResponse<AccountInformation>(ServiceResponseStatus.Successful, accountInformation);
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Get Account Information Exception.");
                return new ServiceResponse<AccountInformation>(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        public ServiceResponse ResyncAccountInformation(string accountID)
        {
            try
            {
                AccountInformation accountInformation = AccountInformation.GetByID(accountID);

                if (accountInformation == null)
                {
                    return new ServiceResponse(ServiceResponseStatus.AccountNotFound);
                }
                else
                {
                    accountInformation.Expire();
                    return new ServiceResponse(ServiceResponseStatus.Successful);
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Resync Account Information Exception.");
                return new ServiceResponse(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        public ServiceResponse ResyncAccountEntitlements(string accountID)
        {
            try
            {
                AccountInformation accountInformation = AccountInformation.GetByID(accountID);

                if (accountInformation == null)
                {
                    return new ServiceResponse(ServiceResponseStatus.AccountNotFound);
                }
                else
                {
                    accountInformation.ResyncEntitlements();
                    accountInformation.Expire();
                    return new ServiceResponse(ServiceResponseStatus.Successful);
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Resync Account Entitlements Exception.");
                return new ServiceResponse(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        #endregion

        #region Purchasing

        public ServiceResponse<Checkout> GetCheckout(int transactionID, string accountID)
        {

            try
            {
                Checkout offering = Checkout.GetByID(transactionID);

                if (offering == null)
                {
                    return new ServiceResponse<Checkout>(ServiceResponseStatus.GetCheckout_CheckoutNotFound);
                }
                else if (offering.AccountID != accountID)
                {
                    return new ServiceResponse<Checkout>(ServiceResponseStatus.GetCheckout_IncorrectAccount);
                }
                else
                {
                    return new ServiceResponse<Checkout>(ServiceResponseStatus.Successful, offering);
                }

            }
            catch (Exception ex)
            {
                Logger.Error("Get Checkout Exception.");
                return new ServiceResponse<Checkout>(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        private ServiceResponse<Checkout> ProcessPayPalSubscription(int userID, int cobaltInstanceID, int entityTypeID, int entityID, AccountInformation account, AccountPaymentMethod paymentMethod, ShoppingCart shoppingCart)
        {
            ShoppingCartItem item = shoppingCart.Items.First();
            ProductOffering product = ProductOffering.GetByID(item.ProductID);
            ProductBillingPlan plan = ProductBillingPlan.GetByID(item.BillingPlanID);
            ProductBillingPlan delayedPlan = plan.GetDelayedPlan();

            // For promotional plans, if the PreventFirstBilling flag is set, do not switch to the delayed plan, instead use the original plan to 
            // actually credit users with the promotional first period
            string preventFirstBillingStr;
            if (plan.MerchantData.TryGetValue("PreventFirstBilling", out preventFirstBillingStr))
            {
                bool preventFirstBilling;
                if (bool.TryParse(preventFirstBillingStr, out preventFirstBilling))
                {
                    if (preventFirstBilling)
                    {
                        delayedPlan = plan;
                    }
                }
            }

            CreateSubscriptionResult result = account.CreateSubscription(userID, cobaltInstanceID, entityTypeID, entityID, null, item.ProductID, delayedPlan.ID, paymentMethod, item.SubscriptionStartDate, AccountTransactionType.PurchaseProduct, shoppingCart.Currency);

            if (!result.IsSuccessful)
            {
                // Log this to a file, for later debugging            
                LogCheckout(AccountTransactionStatus.Failed, item.GetSubTotal(shoppingCart.Currency), userID, cobaltInstanceID, entityTypeID, entityID, account.ID, paymentMethod.ID, paymentMethod, shoppingCart);
                return new ServiceResponse<Checkout>(ServiceResponseStatus.PurchaseProducts_PaymentFailed);
            }

            Checkout checkout = new Checkout();
            checkout.PurchasedSubscriptions.Add(result.Subscription);
            checkout.PayPalRedirectUrl = result.PayPalRedirectUrl;
            checkout.SubTotal = item.GetSubTotal(shoppingCart.Currency);
            checkout.Tax = 0;
            checkout.Total = checkout.SubTotal + checkout.Tax;
            checkout.PaymentMethod = paymentMethod;
            checkout.Save(result.Transaction.ID, account.ID);

            // Save the cart, in case it is needed later
            shoppingCart.Save(userID, result.Transaction.ID, cobaltInstanceID, entityTypeID, entityID, paymentMethod, checkout.SubTotal, checkout.Total, checkout.Tax);

            // Log this to a file, for later debugging
            LogCheckout(AccountTransactionStatus.Pending, item.GetSubTotal(shoppingCart.Currency), userID, cobaltInstanceID, entityTypeID, entityID, account.ID, paymentMethod.ID, paymentMethod, shoppingCart);

            return new ServiceResponse<Checkout>(ServiceResponseStatus.PurchaseProducts_PaymentPending, checkout);

        }

        public ServiceResponse<Checkout> FinalizePayPalTransaction(string vid, string token)
        {

            try
            {
                // 1. Get the transaction, by VID
                AccountTransaction transaction = AccountTransaction.GetByPayPalToken(token);
                Logger.Debug("Finalize Paypal Transaction:", token);

                // 2. Validate the transaction
                if (transaction == null)
                {
                    Logger.Debug("Finalize Paypal Transaction -- transaction is null:",  token);
                    return new ServiceResponse<Checkout>(ServiceResponseStatus.FinalizePayPalTransaction_TransactionNotFound);
                }
                else if (transaction.Status != AccountTransactionStatus.Pending)
                {
                    Logger.Debug("Finalize Paypal Transaction -- transaction is finalized:", token);
                    return new ServiceResponse<Checkout>(ServiceResponseStatus.FinalizePayPalTransaction_TransactionAlreadyFinalized);
                }

                // 3. Get the shopping cart:
                ShoppingCart shoppingCart = ShoppingCart.GetByID(transaction.ID);

                if (shoppingCart == null)
                {
                    Logger.Debug("Finalize Paypal Transaction -- shopping cart is null:", token);
                    return new ServiceResponse<Checkout>(ServiceResponseStatus.FinalizePayPalTransaction_CartNotFound);
                }

                // 4. Determine if this is for a subscription, or a purchase:
                bool isSubscription = shoppingCart.SubscriptionItems.Any();

                if (isSubscription)
                {
                    Logger.Debug("Finalizie Paypal Transaction -- finalizing autobill:", token);
                    // Finalize the auto bill authorization                
                    Subscription.FinalizePayPalAuthorization(vid);

                    Logger.Debug("Finalizie Paypal Transaction -- finalizing transaction:", token);
                    // Finalize the transaction
                    transaction.Status = AccountTransactionStatus.Authorized;
                    transaction.Save();

                    Logger.Debug("Finalizie Paypal Transaction -- getting the checkout:", token);
                    // Retrieve the checkout
                    Checkout checkout = Checkout.GetByID(transaction.ID);

                    if (checkout == null)
                    {
                        checkout = new Checkout()
                        {
                            AccountID = transaction.AccountID
                        };
                    }

                    try
                    {
                        // update cached Checkout object with latest sub, now that it has been updated by Vindicia
                        var currentSub = Subscription.GetBySubscriptionID(checkout.PurchasedSubscriptions.FirstOrDefault().SubscriptionID, true);
                        if (currentSub != null)
                        {
                            checkout.PurchasedSubscriptions.Clear();
                            checkout.PurchasedSubscriptions.Add(currentSub);    
                        }
                    }
                    catch (Exception ex)
                    {
                        Logger.Error("Could not update subscription on cached checkout object.", ex);
                    }
                   

                    Logger.Debug("Finalizie Paypal Transaction -- resyncing entitlements:", token);
                    // Resync their entitlements
                    AccountInformation account = AccountInformation.GetByID(transaction.AccountID);
                    account.ResyncActiveEntitlementsOnly();

                    return new ServiceResponse<Checkout>(ServiceResponseStatus.Successful, checkout);
                }
                else
                {
                    Logger.Debug("Finalizie Paypal Transaction -- finalizying paypal capture in else statement:", token);
                    // Finalize the capture
                    transaction.FinalizePayPalCapture(vid);

                    // If the capture fails, get out!
                    if (transaction.Status == AccountTransactionStatus.Failed)
                    {
                        Logger.Debug("Finalizie Paypal Transaction -- paypal capture failed:", token);
                        return new ServiceResponse<Checkout>(ServiceResponseStatus.FinalizePayPalTransaction_UnableToCapture);
                    }

                    Logger.Debug("Finalizie Paypal Transaction -- getting account information:", token);
                    // Process the cart, with any fulfillments
                    AccountInformation account = AccountInformation.GetByID(transaction.AccountID);
                    Logger.Debug("Finalizie Paypal Transaction -- getting checkout:", token);
                    Checkout checkout = ProcessShoppingCart(shoppingCart, account);

                    return new ServiceResponse<Checkout>(ServiceResponseStatus.Successful, checkout);
                }

            }
            catch (Exception ex)
            {
                Logger.Error("Finalize Paypal Transaction Exception.", ex);
                return new ServiceResponse<Checkout>(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        public ServiceResponse<List<Subscription>> CreateTokenSubscriptions(int userID, int cobaltInstanceID, int entityTypeID, int entityID, string accountID, ShoppingCart cart)
        {
            try
            {
                AccountInformation account = AccountInformation.GetByID(accountID);
                List<Subscription> subscriptions = new List<Subscription>();
                if (account == null)
                {
                    return new ServiceResponse<List<Subscription>>(ServiceResponseStatus.AccountNotFound, "Account " + accountID + " not found", subscriptions);
                }

                foreach (ShoppingCartItem item in cart.Items)
                {
                    // Check Product
                    ProductOffering product = ProductOffering.GetByID(item.ProductID);
                    if (product == null)
                    {
                        return new ServiceResponse<List<Subscription>>(ServiceResponseStatus.PurchaseProducts_InvalidItems, "ProductID " + item.ProductID + " not found", subscriptions);
                    }

                    // Check Billing Plan
                    ProductBillingPlan billingPlan = ProductBillingPlan.GetByID(item.BillingPlanID);
                    if (billingPlan == null)
                    {
                        return new ServiceResponse<List<Subscription>>(ServiceResponseStatus.PurchaseProducts_InvalidItems, "ProductBillingPlanID " + item.BillingPlanID + " not found", subscriptions);
                    }

                    // Ensure Billing Plan is token based
                    if (billingPlan.FirstPayPeriod.Prices.First().Currency != CurrencyType.Token)
                    {
                        return new ServiceResponse<List<Subscription>>(ServiceResponseStatus.PurchaseProducts_InvalidItems, "ProductBillingPlanID " + item.BillingPlanID + " is not a Token plan", subscriptions);
                    }

                    AccountTokenType tokenType = billingPlan.FirstPayPeriod.Prices.First().TokenType;
                    AccountPaymentMethod tokenMethod = new AccountPaymentMethod()
                    {
                        TokenType = tokenType,
                        Type = AccountPaymentType.Token
                    };
                    account.CreateOrUpdatePaymentMethod(tokenMethod);

                    CreateSubscriptionResult result = account.CreateSubscription(userID, cobaltInstanceID, entityTypeID, entityID, null, item.ProductID, item.BillingPlanID, tokenMethod, null, AccountTransactionType.PurchaseProduct, cart.Currency);
                    if (!result.IsSuccessful)
                    {
                        return new ServiceResponse<List<Subscription>>(ServiceResponseStatus.UnknownException, "Unknown exception", subscriptions);
                    }
                    else
                    {
                        subscriptions.Add(result.Subscription);
                    }
                }

                return new ServiceResponse<List<Subscription>>(ServiceResponseStatus.Successful, "Subscription created", subscriptions);
            }
            catch (Exception ex)
            {
                Logger.Error("Create Token Subscriptions Exception.", ex);
                return new ServiceResponse<List<Subscription>>(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace, null);
            }
        }

        private static void LogCheckout(AccountTransactionStatus status, decimal checkoutTotal, int userID, int cobaltInstanceID, int entityTypeID, int entityID, string accountID, string paymentMethodID, AccountPaymentMethod paymentMethod, ShoppingCart shoppingCart)
        {

#if !DEBUG
            return;
#endif

            try
            {

                if (!bool.Parse(ConfigurationManager.AppSettings["CheckoutLoggingEnabled"]))
                {
                    return;
                }

                var cartDump = shoppingCart.DumpAllProperties();

                var loggedPaymentMethod = new AccountPaymentMethod()
                {
                    AccountHolderName = paymentMethod.AccountHolderName,
                    Address = paymentMethod.Address,
                    Description = paymentMethod.Description,
                    EmailAddress = paymentMethod.EmailAddress,
                    ID = paymentMethod.ID,
                    IsActive = paymentMethod.IsActive,
                    Type = paymentMethod.Type,
                    VindiciaID = paymentMethod.VindiciaID
                };

                var paymentDump = loggedPaymentMethod.DumpAllProperties();

                var logPath = Path.Combine(_checkoutLogPath, status.ToString() + "-" + userID.ToString() + "-" + DateTime.UtcNow.Ticks.ToString()) + ".html";

                using (var writer = new StreamWriter(logPath))
                {
                    writer.WriteLine("<b>Status:</b> " + status.ToString() + "<br>");

                    writer.WriteLine("<b>Checkout Total:</b> " + checkoutTotal.ToString() + "<br>");

                    writer.WriteLine("<b>User ID:</b> " + userID.ToString() + "<br>");

                    writer.WriteLine("<b>CobaltInstanceID:</b> " + cobaltInstanceID.ToString() + "<br>");

                    writer.WriteLine("<b>EntityTypeID:</b> " + entityTypeID.ToString() + "<br>");

                    writer.WriteLine("<b>EntityID:</b> " + entityID.ToString() + "<br>");

                    writer.WriteLine("<b>AccountID:</b> " + accountID.ToString() + "<br>");

                    if (!string.IsNullOrEmpty(paymentMethodID))
                    {
                        writer.WriteLine("<b>PaymentMethodID:</b> " + paymentMethodID.ToString() + "<br>");
                    }

                    writer.WriteLine("<b>Shopping Cart</b>");
                    writer.WriteLine(cartDump);

                    writer.WriteLine("<b>Payment Method</b>");
                    writer.WriteLine(paymentDump);
                }
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Unable to log checkout.");
            }
        }

        public ServiceResponse<Checkout> PurchaseProducts(int userID, int cobaltInstanceID, int entityTypeID, int entityID, string accountID, string paymentMethodID, AccountPaymentMethod paymentMethod, ShoppingCart shoppingCart)
        {
            try
            {
                // 1. Make sure this account exists
                AccountInformation account = AccountInformation.GetByID(accountID);

                if (account == null)
                {
                    return new ServiceResponse<Checkout>(ServiceResponseStatus.AccountNotFound);
                }

                // 2. Make sure this payment method exists
                if (paymentMethod == null)
                {
                    paymentMethod = account.PaymentMethods.FirstOrDefault(p => p.ID == paymentMethodID);
                }
                else
                {
                    account.CreateOrUpdatePaymentMethod(paymentMethod);
                }

                if (paymentMethod == null)
                {
                    return new ServiceResponse<Checkout>(ServiceResponseStatus.PurchaseProducts_PaymentMethodNotFound);
                }

                // 2.5 Make sure the payment type is available to the country specified
                if (!string.IsNullOrEmpty(paymentMethod.Address.Country))
                {
                    var country = CountryBillingOption.GetByName(paymentMethod.Address.Country);
                    if (country == null)
                    {
                        return new ServiceResponse<Checkout>(ServiceResponseStatus.PurchaseProducts_UnknownCountry);
                    }

                    // validate the payment method
                    if (!country.ValidatePaymentType(paymentMethod.Type))
                    {
                        return new ServiceResponse<Checkout>(ServiceResponseStatus.PurchaseProducts_InvalidCountryPaymentMethod);
                    }

                    // validate the currency
                    if (!country.ValidateCurrency(shoppingCart.Currency))
                    {
                        return new ServiceResponse<Checkout>(ServiceResponseStatus.PurchaseProducts_InvalidCountryCurrency);
                    }
                }
                else
                {
                    return new ServiceResponse<Checkout>(ServiceResponseStatus.PurchaseProducts_UnknownCountry);
                }

                // 3. Validate the shopping cart
                IEnumerable<ShoppingCartItem> invalidItems = shoppingCart.Validate();
                if (invalidItems.Any())
                {
                    return new ServiceResponse<Checkout>(ServiceResponseStatus.PurchaseProducts_InvalidItems);
                }

                // 5. If this is a PayPal purchase, divert the code path (Ugh!)
                if (paymentMethod.Type == AccountPaymentType.PayPal && shoppingCart.SubscriptionItems.Any())
                {
                    return ProcessPayPalSubscription(userID, cobaltInstanceID, entityTypeID, entityID, account, paymentMethod, shoppingCart);
                }

                // 5. Total up all items into a single transaction
                List<AccountTransactionItem> accountTransactionItems = new List<AccountTransactionItem>();
                decimal checkoutSubTotal = 0;
                decimal checkoutTotal = 0;
                decimal checkoutTax = 0;

                foreach (ShoppingCartItem item in shoppingCart.Items)
                {
                    ProductOffering productOffering = ProductOffering.GetByID(item.ProductID);
                    checkoutSubTotal += item.GetSubTotal(shoppingCart.Currency);
                    ProductBillingPlan billingPlan = ProductBillingPlan.GetByID(item.BillingPlanID);
                    accountTransactionItems.Add(new AccountTransactionItem()
                    {
                        Description = billingPlan.Description,
                        Quantity = item.Quantity,
                        ProductID = item.ProductID,
                        SubTotal = item.GetSubTotal(shoppingCart.Currency),
                        Price = item.GetPrice(shoppingCart.Currency)
                    });
                }

                checkoutTotal = checkoutTax + checkoutSubTotal;

                Checkout checkout = new Checkout();

                // 6. Setup a transaction            
                AccountTransaction accountTransaction = AccountTransaction.Create(userID, accountID, paymentMethod, shoppingCart.Currency, checkoutTotal, 0.00M, checkoutTotal, accountTransactionItems.ToArray(), AccountTransactionType.PurchaseProduct, AccountTransactionStatus.Pending, shoppingCart.UrlReferrer, null);
                Transaction txn = accountTransaction.ToVindicia();

                // 7. Re-hydrate the payment method object
                // This is done to add any fields that are not persisted to the database, such as CVN and PayPal
                txn.sourcePaymentMethod = paymentMethod.ToVindicia();


                if (checkoutTotal > 0)
                {
                    // 7. Try to charge the payment method
                    Return ret = txn.authCapture(false, true);

                    accountTransaction.Status = AccountTransactionStatus.Failed;

                    if (txn.statusLog.Length > 0 && txn.statusLog[0].payPalStatus != null)
                    {
                        if (txn.statusLog[0].status == TransactionStatusType.AuthorizationPending)
                        {
                            TransactionStatusType txnStatusType = txn.statusLog[0].status;
                            checkout.PayPalRedirectUrl = txn.statusLog[0].payPalStatus.redirectUrl;
                            accountTransaction.Status = AccountTransactionStatus.Pending;
                            accountTransaction.PayPalToken = txn.statusLog[0].payPalStatus.token;
                        }
                        else
                        {
                            accountTransaction.Status = AccountTransactionStatus.Failed;
                        }
                    }
                    else if (ret.returnCode == ReturnCode.Item200)
                    {
                        TransactionStatus transactionStatus = txn.statusLog.First();
                        accountTransaction.Status = transactionStatus.status.ToCurse();
                    }
                }
                else
                {
                    accountTransaction.Status = AccountTransactionStatus.Authorized;
                }

                // Save the transaction to the database
                accountTransaction.Save();

                // Log this to a file, for later debugging
                LogCheckout(accountTransaction.Status, checkoutTotal, userID, cobaltInstanceID, entityTypeID, entityID, accountID, paymentMethodID, paymentMethod, shoppingCart);

                // Save the cart, in case it is needed later
                shoppingCart.Save(userID, accountTransaction.ID, cobaltInstanceID, entityTypeID, entityID, paymentMethod, checkoutSubTotal, checkoutTotal, checkoutTax);

                // 4. If the payment failed or pending, return the failed payment status
                switch (accountTransaction.Status)
                {
                    case AccountTransactionStatus.Authorized:
                    case AccountTransactionStatus.Captured:
                        break;
                    case AccountTransactionStatus.Cancelled:
                    case AccountTransactionStatus.Failed:
                    case AccountTransactionStatus.AuthorizedForValidation:
                    case AccountTransactionStatus.AuthorizedPending:
                    case AccountTransactionStatus.Internal:
                    case AccountTransactionStatus.New:
                    case AccountTransactionStatus.Refunded:
                    case AccountTransactionStatus.Settled:
                        return new ServiceResponse<Checkout>(ServiceResponseStatus.PurchaseProducts_PaymentFailed);
                    case AccountTransactionStatus.Pending:
                    case AccountTransactionStatus.AuthorizationPending:
                        return new ServiceResponse<Checkout>(ServiceResponseStatus.PurchaseProducts_PaymentPending, checkout);
                    default:
                        throw new ArgumentOutOfRangeException("transactionStatus", "Unknown transaction status: {0}".FormatWith(accountTransaction.Status));
                }

                checkout = ProcessShoppingCart(shoppingCart, account);

                return new ServiceResponse<Checkout>(ServiceResponseStatus.Successful, checkout);
            }
            catch (Exception ex)
            {
                Logger.Error("Purchase Products Exception.", ex);
                return new ServiceResponse<Checkout>(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        private Checkout ProcessShoppingCart(ShoppingCart shoppingCart, AccountInformation account)
        {

            Checkout checkout = new Checkout();

            foreach (ShoppingCartItem item in shoppingCart.Items)
            {
                ProductOffering productOffering = ProductOffering.GetByID(item.ProductID);
                if (productOffering.IsSubscription)
                {
                    continue;
                }

                ProductPurchase purchase = new ProductPurchase();



                IEnumerable<ProductFulfillment> fulfillments =
                    ProductFulfillment.Fulfill(shoppingCart.UserID, shoppingCart.CobaltInstanceID, shoppingCart.EntityTypeID, shoppingCart.EntityID, productOffering.FulfillmentType, account, item);


                purchase.Price = item.GetPrice(shoppingCart.Currency);
                purchase.Quantity = item.Quantity;
                purchase.SubTotal = item.GetSubTotal(shoppingCart.Currency);
                purchase.Fulfillments = fulfillments;
                purchase.Status = ProductPurchaseStatus.Successful;
                purchase.ProductDescription = productOffering.Description;
                purchase.PlanDescription = ProductBillingPlan.GetByID(item.BillingPlanID).Description;

                checkout.PurchasedProducts.Add(purchase);
            }

            // 4. If any subscription products exists, perform the transaction
            if (shoppingCart.SubscriptionItems.Any())
            {
                foreach (ShoppingCartItem item in shoppingCart.SubscriptionItems)
                {
                    CreateSubscriptionResult result = account.CreateSubscription(shoppingCart.UserID, shoppingCart.CobaltInstanceID, shoppingCart.EntityTypeID, shoppingCart.EntityID, shoppingCart.TransactionID, item.ProductID, item.BillingPlanID, shoppingCart.PaymentMethod, item.SubscriptionStartDate, AccountTransactionType.PurchaseProduct, shoppingCart.Currency);

                    if (result.IsSuccessful)
                    {
                        checkout.PurchasedSubscriptions.Add(result.Subscription);
                    }
                }
            }

            checkout.SubTotal = shoppingCart.SubTotal;
            checkout.Tax = shoppingCart.Tax;
            checkout.Total = shoppingCart.Total;
            checkout.PaymentMethod = shoppingCart.PaymentMethod;
            checkout.Save(shoppingCart.TransactionID, account.ID);

            return checkout;
        }

        public ServiceResponse<Checkout> v2ChangeSubscriptionPlan(int userID, int cobaltInstanceID, int entityTypeID, int entityID, string accountID, string paymentMethodID, AccountPaymentMethod paymentMethod, CurrencyType currency, string subscriptionID, string newBillingPlanID, bool terminateExistingSub, int newSubCreditDays)
        {
            try
            {
                // 1. Make sure this account exists
                AccountInformation account = AccountInformation.GetByID(accountID);

                if (account == null)
                {
                    return new ServiceResponse<Checkout>(ServiceResponseStatus.AccountNotFound);
                }

                // 2. Make sure this payment method exists
                if (paymentMethod == null)
                {
                    paymentMethod = account.PaymentMethods.FirstOrDefault(p => p.ID == paymentMethodID);
                }
                else
                {
                    account.CreateOrUpdatePaymentMethod(paymentMethod);
                }

                if (paymentMethod == null)
                {
                    return new ServiceResponse<Checkout>(ServiceResponseStatus.ChangeSubscriptionPlan_PaymentMethodNotFound);
                }

                // 2.5 Make sure the payment type is available to the country specified
                if (!string.IsNullOrEmpty(paymentMethod.Address.Country))
                {
                    var country = CountryBillingOption.GetByName(paymentMethod.Address.Country);
                    if (country == null)
                    {
                        return new ServiceResponse<Checkout>(ServiceResponseStatus.ChangeSubscriptionPlan_UnknownCountry);
                    }

                    // validate the payment method
                    if (!country.ValidatePaymentType(paymentMethod.Type))
                    {
                        return new ServiceResponse<Checkout>(ServiceResponseStatus.ChangeSubscriptionPlan_InvalidCountryPaymentMethod);
                    }

                    // validate the currency
                    if (!country.ValidateCurrency(currency))
                    {
                        return new ServiceResponse<Checkout>(ServiceResponseStatus.ChangeSubscriptionPlan_InvalidCountryCurrency);
                    }
                }
                else
                {
                    return new ServiceResponse<Checkout>(ServiceResponseStatus.ChangeSubscriptionPlan_UnknownCountry);
                }

                // 3. Make sure the subscription exists
                Subscription existingSubscription = account.Subscriptions.FirstOrDefault(p => p.SubscriptionID == subscriptionID);

                if (existingSubscription == null)
                {
                    return new ServiceResponse<Checkout>(ServiceResponseStatus.ChangeSubscriptionPlan_SubscriptionNotFound);
                }

                // Cancel any pending subs for this product type
                IEnumerable<Subscription> pendingSubs = account.Subscriptions.Where(p => p.ProductID == existingSubscription.ProductID && p.DateStarted > DateTime.UtcNow && p != existingSubscription && p.DisplayStatus != DisplaySubscriptionStatus.Cancelled);
                foreach (Subscription pendingSub in pendingSubs)
                {
                    pendingSub.Cancel(true);
                }

                // Cancel the existing subscription:
                existingSubscription.Cancel(terminateExistingSub);

                // Set the new subscription to start as soon as the current one ends:
                DateTime effectiveDate = existingSubscription.NextBillingDate.HasValue ? existingSubscription.NextBillingDate.Value : existingSubscription.DateExpires;

                // if terminating the existing sub, the effectiveDate should be now, not the end of the currunt sub
                if (terminateExistingSub)
                {
                    effectiveDate = DateTime.UtcNow;
                }

                CreateSubscriptionResult result = account.CreateSubscription(userID, cobaltInstanceID, entityTypeID, entityID, null, existingSubscription.ProductID, newBillingPlanID, paymentMethod, effectiveDate, AccountTransactionType.ChangeSubscription, currency);
                if (result.Transaction == null)
                {
                    return new ServiceResponse<Checkout>(ServiceResponseStatus.ChangeSubscriptionPlan_PaymentFailed, "Failed to create new subscription.");
                }

                Checkout checkout = new Checkout()
                {
                    SubTotal = 0,
                    Tax = 0,
                    Total = 0,
                    TransactionID = result.Transaction.ID
                };

                bool willBeCharged = existingSubscription.DateExpires <= DateTime.UtcNow;

                // Credit the new subscription by the specified number of days, if applicabale
                if (newSubCreditDays > 0 && result.IsSuccessful)
                {
                    ServiceResponse creditResponse = CreditSubscription(userID, account.ID, result.Subscription.SubscriptionID, newSubCreditDays, "Prorated days from subscription {0}".FormatWith(existingSubscription.ID));

                    result.Subscription = Subscription.GetByID(result.Subscription.ID, true);
                    willBeCharged = true;
                }

                if (willBeCharged)
                {
                    // Create cart and update checkout object with amount being charged
                    ShoppingCart cart = new ShoppingCart() { Currency = currency };
                    cart.Items = new ShoppingCartItem[] { new ShoppingCartItem(existingSubscription.ProductID, newBillingPlanID, 1) };
                    ProductBillingPlan plan = ProductBillingPlan.GetByID(newBillingPlanID);
                    if (plan.HasCurrency(cart.Currency))
                    {
                        foreach (ShoppingCartItem item in cart.Items)
                        {
                            checkout.SubTotal += item.GetSubTotal(cart.Currency);
                        }
                    }
                    checkout.Total = checkout.SubTotal + checkout.Tax;
                    cart.Save(userID, result.Transaction.ID, cobaltInstanceID, entityTypeID, entityID, paymentMethod, checkout.SubTotal, checkout.Tax, checkout.Total);
                }

                checkout.PurchasedSubscriptions.Add(result.Subscription);

                if (!result.IsSuccessful)
                {
                    switch (result.Transaction.Status)
                    {
                        case AccountTransactionStatus.Cancelled:
                        case AccountTransactionStatus.Failed:
                        case AccountTransactionStatus.AuthorizedForValidation:
                        case AccountTransactionStatus.AuthorizedPending:
                        case AccountTransactionStatus.Internal:
                        case AccountTransactionStatus.New:
                        case AccountTransactionStatus.Refunded:
                        case AccountTransactionStatus.Settled:
                            return new ServiceResponse<Checkout>(ServiceResponseStatus.ChangeSubscriptionPlan_PaymentFailed);
                        default:
                            return new ServiceResponse<Checkout>(ServiceResponseStatus.UnknownException, checkout);
                    }
                }

                // If this is a PayPal purchase, divert the code path (Ugh!)
                if (paymentMethod.Type == AccountPaymentType.PayPal)
                {
                    // Save a proxy cart to be used when PayPal returns
                    ShoppingCart cart = new ShoppingCart();
                    cart.Items = new ShoppingCartItem[] { new ShoppingCartItem(existingSubscription.ProductID, newBillingPlanID, 1) };
                    cart.Save(userID, result.Transaction.ID, cobaltInstanceID, entityTypeID, entityID, paymentMethod, checkout.SubTotal, checkout.Tax, checkout.Total);
                    checkout.PayPalRedirectUrl = result.PayPalRedirectUrl;

                    return new ServiceResponse<Checkout>(ServiceResponseStatus.ChangeSubscriptionPlan_PaymentPending, checkout);
                }

                return new ServiceResponse<Checkout>(ServiceResponseStatus.Successful, checkout);
            }
            catch (Exception ex)
            {
                Logger.Error("Change Subscription Plan Exception.", ex);
                return new ServiceResponse<Checkout>(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }

        }

        public ServiceResponse<Checkout> ChangeSubscriptionPlan(int userID, int cobaltInstanceID, int entityTypeID, int entityID, string accountID, string paymentMethodID, AccountPaymentMethod paymentMethod, string subscriptionID, string newBillingPlanID, bool terminateExistingSub, int newSubCreditDays)
        {
            try
            {
                // 1. Make sure this account exists
                AccountInformation account = AccountInformation.GetByID(accountID);

                if (account == null)
                {
                    return new ServiceResponse<Checkout>(ServiceResponseStatus.AccountNotFound);
                }

                // 2. Make sure this payment method exists
                if (paymentMethod == null)
                {
                    paymentMethod = account.PaymentMethods.FirstOrDefault(p => p.ID == paymentMethodID);
                }
                else
                {
                    account.CreateOrUpdatePaymentMethod(paymentMethod);
                }

                if (paymentMethod == null)
                {
                    return new ServiceResponse<Checkout>(ServiceResponseStatus.ChangeSubscriptionPlan_PaymentMethodNotFound);
                }

                // 2.5 Make sure the payment type is available to the country specified
                if (!string.IsNullOrEmpty(paymentMethod.Address.Country))
                {
                    var country = CountryBillingOption.GetByName(paymentMethod.Address.Country);
                    if (country == null)
                    {
                        return new ServiceResponse<Checkout>(ServiceResponseStatus.ChangeSubscriptionPlan_UnknownCountry);
                    }

                    // validate the payment method
                    if (!country.ValidatePaymentType(paymentMethod.Type))
                    {
                        return new ServiceResponse<Checkout>(ServiceResponseStatus.ChangeSubscriptionPlan_InvalidCountryPaymentMethod);
                    }

                    // validate the currency
                    if (!country.ValidateCurrency(CurrencyType.USD))
                    {
                        return new ServiceResponse<Checkout>(ServiceResponseStatus.ChangeSubscriptionPlan_InvalidCountryCurrency);
                    }
                }
                else
                {
                    return new ServiceResponse<Checkout>(ServiceResponseStatus.ChangeSubscriptionPlan_UnknownCountry);
                }

                // 3. Make sure the subscription exists
                Subscription existingSubscription = account.Subscriptions.FirstOrDefault(p => p.SubscriptionID == subscriptionID);

                if (existingSubscription == null)
                {
                    return new ServiceResponse<Checkout>(ServiceResponseStatus.ChangeSubscriptionPlan_SubscriptionNotFound);
                }

                // Cancel any pending subs for this product type
                IEnumerable<Subscription> pendingSubs = account.Subscriptions.Where(p => p.ProductID == existingSubscription.ProductID && p.DateStarted > DateTime.UtcNow && p != existingSubscription && p.DisplayStatus != DisplaySubscriptionStatus.Cancelled);
                foreach (Subscription pendingSub in pendingSubs)
                {
                    pendingSub.Cancel(true);
                }

                // Cancel the existing subscription:
                existingSubscription.Cancel(terminateExistingSub);

                // Set the new subscription to start as soon as the current one ends:
                DateTime effectiveDate = existingSubscription.NextBillingDate.HasValue ? existingSubscription.NextBillingDate.Value : existingSubscription.DateExpires;

                // if terminating the existing sub, the effectiveDate should be now, not the end of the currunt sub
                if (terminateExistingSub)
                {
                    effectiveDate = DateTime.UtcNow;
                }

                CreateSubscriptionResult result = account.CreateSubscription(userID, cobaltInstanceID, entityTypeID, entityID, null, existingSubscription.ProductID, newBillingPlanID, paymentMethod, effectiveDate, AccountTransactionType.ChangeSubscription, CurrencyType.USD);
                if (result.Transaction == null)
                {
                    return new ServiceResponse<Checkout>(ServiceResponseStatus.ChangeSubscriptionPlan_PaymentFailed, "Failed to create new subscription.");
                }

                Checkout checkout = new Checkout()
                {
                    SubTotal = 0,
                    Tax = 0,
                    Total = 0,
                    TransactionID = result.Transaction.ID
                };

                bool willBeCharged = existingSubscription.DateExpires <= DateTime.UtcNow;

                // Credit the new subscription by the specified number of days, if applicabale
                if (newSubCreditDays > 0 && result.IsSuccessful)
                {
                    ServiceResponse creditResponse = CreditSubscription(userID, account.ID, result.Subscription.SubscriptionID, newSubCreditDays, "Prorated days from subscription {0}".FormatWith(existingSubscription.ID));

                    result.Subscription = Subscription.GetByID(result.Subscription.ID, true);
                    willBeCharged = true;
                }

                if (willBeCharged)
                {
                    // Create cart and update checkout object with amount being charged
                    ShoppingCart cart = new ShoppingCart() { Currency = CurrencyType.USD };
                    cart.Items = new ShoppingCartItem[] { new ShoppingCartItem(existingSubscription.ProductID, newBillingPlanID, 1) };
                    ProductBillingPlan plan = ProductBillingPlan.GetByID(newBillingPlanID);
                    if (plan.HasCurrency(cart.Currency))
                    {
                        foreach (ShoppingCartItem item in cart.Items)
                        {
                            checkout.SubTotal += item.GetSubTotal(cart.Currency);
                        }
                    }
                    checkout.Total = checkout.SubTotal + checkout.Tax;
                    cart.Save(userID, result.Transaction.ID, cobaltInstanceID, entityTypeID, entityID, paymentMethod, checkout.SubTotal, checkout.Tax, checkout.Total);
                }

                checkout.PurchasedSubscriptions.Add(result.Subscription);

                if (!result.IsSuccessful)
                {
                    switch (result.Transaction.Status)
                    {
                        case AccountTransactionStatus.Cancelled:
                        case AccountTransactionStatus.Failed:
                        case AccountTransactionStatus.AuthorizedForValidation:
                        case AccountTransactionStatus.AuthorizedPending:
                        case AccountTransactionStatus.Internal:
                        case AccountTransactionStatus.New:
                        case AccountTransactionStatus.Refunded:
                        case AccountTransactionStatus.Settled:
                            return new ServiceResponse<Checkout>(ServiceResponseStatus.ChangeSubscriptionPlan_PaymentFailed);
                        default:
                            return new ServiceResponse<Checkout>(ServiceResponseStatus.UnknownException, checkout);
                    }
                }

                // If this is a PayPal purchase, divert the code path (Ugh!)
                if (paymentMethod.Type == AccountPaymentType.PayPal)
                {
                    // Save a proxy cart to be used when PayPal returns
                    ShoppingCart cart = new ShoppingCart();
                    cart.Items = new ShoppingCartItem[] { new ShoppingCartItem(existingSubscription.ProductID, newBillingPlanID, 1) };
                    cart.Save(userID, result.Transaction.ID, cobaltInstanceID, entityTypeID, entityID, paymentMethod, checkout.SubTotal, checkout.Tax, checkout.Total);
                    checkout.PayPalRedirectUrl = result.PayPalRedirectUrl;

                    return new ServiceResponse<Checkout>(ServiceResponseStatus.ChangeSubscriptionPlan_PaymentPending, checkout);
                }

                return new ServiceResponse<Checkout>(ServiceResponseStatus.Successful, checkout);
            }
            catch (Exception ex)
            {
                Logger.Error("Change Subscription Plan Exception.", ex);
                return new ServiceResponse<Checkout>(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }

        }

        public ServiceResponse CancelSubscription(int userID, string accountID, string subscriptionID, string note)
        {
            return CancelSubscription(userID, accountID, subscriptionID, note, false);
        }

        public ServiceResponse TerminateSubscription(int userID, string accountID, string subscriptionID, string note)
        {
            return CancelSubscription(userID, accountID, subscriptionID, note, true);
        }

        public ServiceResponse CancelSubscription(int userID, string accountID, string subscriptionID, string note, bool disentitle)
        {
            try
            {
                // 1. Make sure this account exists
                AccountInformation account = AccountInformation.GetByID(accountID);

                if (account == null)
                {
                    return new ServiceResponse(ServiceResponseStatus.AccountNotFound);
                }

                // 2. Make sure the subscription exists
                Subscription existingSubscription = account.Subscriptions.FirstOrDefault(p => p.SubscriptionID == subscriptionID);

                if (existingSubscription == null)
                {
                    return new ServiceResponse(ServiceResponseStatus.CancelSubscription_SubscriptionNotFound);
                }

                if (existingSubscription.Status != SubscriptionStatus.Active)
                {
                    return new ServiceResponse(ServiceResponseStatus.CancelSubscription_SubscriptionNotActive);
                }

                // 3. Cancel it
                account.CancelSubscription(userID, subscriptionID, note, disentitle);

                return new ServiceResponse(ServiceResponseStatus.Successful);
            }
            catch (Exception ex)
            {
                Logger.Error("Cancel Subscription Exception.", ex);
                return new ServiceResponse(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        public ServiceResponse<Subscription> GetSubscriptionByID(string subscriptionID)
        {
            try
            {
                SubscriptionHistory history = SubscriptionHistory.GetBySubscriptionID(subscriptionID);
                Subscription sub = history.ToSubscription();

                return new ServiceResponse<Subscription>(ServiceResponseStatus.Successful, sub);
            }
            catch (Exception ex)
            {
                Logger.Error("Get Subscription Exception.", ex);
                return new ServiceResponse<Subscription>(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        /// <summary>
        /// Returns up to 100 recent subscriptions
        /// </summary>
        /// <param name="userID"></param>
        /// <returns></returns>
        public ServiceResponse<List<Subscription>> GetSubscriptionByUserID(int userID)
        {
            try
            {
                IEnumerable<SubscriptionHistory> histories = SubscriptionHistory.GetAllByUserID(userID);
                List<Subscription> subs = histories.Select(h => h.ToSubscription()).ToList();

                return new ServiceResponse<List<Subscription>>(ServiceResponseStatus.Successful, subs.Take(100).ToList());
            }
            catch (Exception ex)
            {
                Logger.Error("Get Subscription Plan By User Exception.", ex);
                return new ServiceResponse<List<Subscription>>(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }

        }

        public ServiceResponse<List<AccountTransaction>> GetTransactionsForUser(int userID, string userAccountID)
        {
            try
            {
                // First determine all account IDs related to this user:
                IEnumerable<SubscriptionHistory> subscriptions = SubscriptionHistory.GetAllByUserID(userID);
                IEnumerable<string> accountIDs = new List<string>() { userAccountID };
                accountIDs = accountIDs.Union(subscriptions.Select(p => p.EntitledAccountID).Distinct());

                List<AccountTransaction> accountTransactions = AccountTransaction.GetAllByAccountIDs(accountIDs);
                List<AccountTransaction> userTransactions = AccountTransaction.GetAllByUserIDAndType(userID, AccountTransactionType.PurchaseProduct);
                List<AccountTransaction> allTransactions = accountTransactions.Union(userTransactions).ToList();
                return new ServiceResponse<List<AccountTransaction>>(ServiceResponseStatus.Successful, allTransactions.Take(100).ToList());
            }
            catch (Exception ex)
            {
                Logger.Error("Get Transactions For User Exception.", ex);
                return new ServiceResponse<List<AccountTransaction>>(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }

        }

        public ServiceResponse CreditSubscription(int userID, string accountID, string subscriptionID, int days, string note)
        {
            try
            {
                // 1. Make sure this account exists
                AccountInformation account = AccountInformation.GetByID(accountID);

                if (account == null)
                {
                    return new ServiceResponse(ServiceResponseStatus.AccountNotFound);
                }

                // 2. Make sure the subscription exists
                Subscription existingSubscription = account.Subscriptions.FirstOrDefault(p => p.SubscriptionID == subscriptionID);

                if (existingSubscription == null)
                {
                    return new ServiceResponse(ServiceResponseStatus.CreditSubscription_SubscriptionNotFound);
                }

                // 3. Credit it
                account.CreditSubscription(userID, subscriptionID, days, AccountTransactionType.CreditSubscription, note);

                return new ServiceResponse(ServiceResponseStatus.Successful);
            }
            catch (Exception ex)
            {
                Logger.Error("Credit Subscription Exception.", ex);
                return new ServiceResponse(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        public ServiceResponse IssueRefund(int userID, string transactionID, decimal amount, string note)
        {

            try
            {

                // 1. Get the transaction
                AccountTransaction accountTransaction = AccountTransaction.GetByTransactionID(transactionID, false);

                if (accountTransaction == null)
                {
                    return new ServiceResponse(ServiceResponseStatus.IssueRefund_TransactionNotFound);
                }

                // 2. Do some range checks:
                if (accountTransaction.Status == AccountTransactionStatus.Refunded)
                {
                    return new ServiceResponse(ServiceResponseStatus.IssueRefund_AlreadyRefunded);
                }

                // 3. Do some range checks:
                if (amount <= 0 || amount > accountTransaction.Total)
                {
                    return new ServiceResponse(ServiceResponseStatus.IssueRefund_InvalidAmount);
                }


                // 4. Issue the refund
                accountTransaction.IssueRefund(userID, amount, note);


                return new ServiceResponse(ServiceResponseStatus.Successful);
            }
            catch (Exception ex)
            {
                Logger.Error("Issue Refund Exception.", ex);
                return new ServiceResponse(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }

        }

        #region Voice
        // TODO: can all this voice stuff go bye bye?
        /// <summary>
        /// Grants one or more types of tokens to an account
        /// </summary>
        /// <param name="accountID"></param>
        /// <param name="amounts"></param>
        /// <returns></returns>
        public ServiceResponse CreditTokens(string adminName, string accountID, IEnumerable<AccountTokenAmount> amounts)
        {
            // 1. Make sure that an amount exists, which is greater than 0.
            if (amounts == null || amounts.Any(p => p.Amount <= 0))
            {
                return new ServiceResponse(ServiceResponseStatus.CreditTokens_InvalidTokenAmount);
            }

            // 2. Make sure this account exists
            AccountInformation account = AccountInformation.GetByID(accountID);

            if (account == null)
            {
                return new ServiceResponse(ServiceResponseStatus.AccountNotFound);
            }

            if (account.CreditTokens(adminName, amounts))
            {
                return new ServiceResponse(ServiceResponseStatus.Successful);
            }
            return new ServiceResponse(ServiceResponseStatus.UnknownException);
        }

        /// <summary>
        /// Deducts one or more types of tokens from an account
        /// </summary>
        /// <param name="accountID"></param>
        /// <param name="amounts"></param>
        /// <returns></returns>
        public ServiceResponse DeductTokens(string adminName, string accountID, IEnumerable<AccountTokenAmount> amounts)
        {
            // 1. Make sure that an amount exists, which is greater than 0.
            if (amounts == null || amounts.Any(p => p.Amount <= 0))
            {
                return new ServiceResponse(ServiceResponseStatus.CreditTokens_InvalidTokenAmount);
            }

            // 2. Make sure this account exists
            AccountInformation account = AccountInformation.GetByID(accountID);

            if (account == null)
            {
                return new ServiceResponse(ServiceResponseStatus.AccountNotFound);
            }

            if (account.DeductTokens(adminName, amounts))
            {
                return new ServiceResponse(ServiceResponseStatus.Successful);
            }
            return new ServiceResponse(ServiceResponseStatus.UnknownException);
        }

        public ServiceResponse<AccountVoiceClient> GetAccountVoiceClient(string accountID)
        {
            try
            {
                AccountVoiceClient client = AccountVoiceClient.GetByAccountID(accountID, false);
                return new ServiceResponse<AccountVoiceClient>(ServiceResponseStatus.Successful, client);
            }
            catch (Exception ex)
            {
                Logger.Error("Get Account Voice Client Exception.", ex);
                return new ServiceResponse<AccountVoiceClient>(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        public ServiceResponse<AccountVoiceClient> AddAccountVoiceService(string accountID, Int64 clientID, Int64 serviceID, IPAddress ipAddress, int port)
        {
            try
            {
                if (clientID <= 0)
                {
                    throw new ArgumentException("clientID must be an unsigned integer, greater than 0.", "clientID");
                }

                if (serviceID <= 0)
                {
                    throw new ArgumentException("serviceID must be an unsigned integer, greater than 0.", "serviceID");
                }

                if (accountID.IsNullOrEmpty())
                {
                    throw new ArgumentException("accountID must be a valid billing account id.", "accountID");
                }

                if (ipAddress == null)
                {
                    throw new ArgumentException("ipAddress cannot be null.", "ipAddress");
                }

                if (port <= 0)
                {
                    throw new ArgumentException("port must be an unsigned integer, greater than 0.", "port");
                }

                AccountInformation account = AccountInformation.GetByID(accountID);

                if (account == null)
                {
                    return new ServiceResponse<AccountVoiceClient>(ServiceResponseStatus.AccountNotFound);
                }

                AccountVoiceClient client = AccountVoiceClient.GetByAccountID(accountID, false);
                if (client == null)
                {
                    client = new AccountVoiceClient()
                    {
                        AccountID = accountID,
                        ClientID = clientID
                    };
                    client.Save();
                }

                if (client.ClientID != clientID)
                {
                    return new ServiceResponse<AccountVoiceClient>(ServiceResponseStatus.AddAccountVoiceClient_ClientIDMismatch);
                }

                if (client.Services.Exists(p => p.ServiceID == serviceID))
                {
                    return new ServiceResponse<AccountVoiceClient>(ServiceResponseStatus.AddAccountVoiceClient_ServiceAlreadyExists);
                }

                client.AddService(serviceID, ipAddress, port);

                return new ServiceResponse<AccountVoiceClient>(ServiceResponseStatus.Successful, client);
            }
            catch (Exception ex)
            {
                Logger.Error("Add Account Voice Exception: {0}. Stack Trace: {1}", ex);
                return new ServiceResponse<AccountVoiceClient>(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        public ServiceResponse DeleteAccountVoiceService(string accountID, Int64 serviceID)
        {
            try
            {

                if (serviceID <= 0)
                {
                    throw new ArgumentException("serviceID must be an unsigned integer, greater than 0.", "serviceID");
                }

                if (accountID.IsNullOrEmpty())
                {
                    throw new ArgumentException("accountID must be a valid billing account id.", "accountID");
                }

                AccountInformation account = AccountInformation.GetByID(accountID);

                if (account == null)
                {
                    return new ServiceResponse(ServiceResponseStatus.AccountNotFound);
                }

                AccountVoiceClient client = AccountVoiceClient.GetByAccountID(accountID, false);
                if (client == null)
                {
                    return new ServiceResponse(ServiceResponseStatus.DeleteAccountVoiceService_ClientNotFound);
                }

                AccountVoiceService service = client.Services.FirstOrDefault(p => p.ServiceID == serviceID);
                if (service == null)
                {
                    return new ServiceResponse(ServiceResponseStatus.DeleteAccountVoiceService_ServiceNotFound);
                }

                service.Delete();
                client.Expire();

                return new ServiceResponse(ServiceResponseStatus.Successful);
            }
            catch (Exception ex)
            {
                Logger.Error("Delete Account Voice Exception.", ex);
                return new ServiceResponse(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        public ServiceResponse UpdateAccountVoiceService(string accountID, Int64 serviceID, IPAddress ipAddress, int port, AccountVoiceServiceStatus status, DateTime dateLastMoved)
        {
            try
            {

                if (serviceID <= 0)
                {
                    throw new ArgumentException("serviceID must be an unsigned integer, greater than 0.", "serviceID");
                }

                if (accountID.IsNullOrEmpty())
                {
                    throw new ArgumentException("accountID must be a valid billing account id.", "accountID");
                }

                AccountInformation account = AccountInformation.GetByID(accountID);

                if (account == null)
                {
                    return new ServiceResponse(ServiceResponseStatus.AccountNotFound);
                }

                AccountVoiceClient client = AccountVoiceClient.GetByAccountID(accountID, false);
                if (client == null)
                {
                    return new ServiceResponse(ServiceResponseStatus.UpdateAccountVoiceService_ClientNotFound);
                }

                AccountVoiceService service = client.Services.FirstOrDefault(p => p.ServiceID == serviceID);
                if (service == null)
                {
                    return new ServiceResponse(ServiceResponseStatus.UpdateAccountVoiceService_ServiceNotFound);
                }

                service.IPAddress = ipAddress;
                service.Port = port;
                service.Status = status;
                service.DateLastMoved = dateLastMoved;
                service.Save();

                return new ServiceResponse(ServiceResponseStatus.Successful);
            }
            catch (Exception ex)
            {
                Logger.Error("Update Account Voice Exception.", ex);
                return new ServiceResponse(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }

        public ServiceResponse<PagedRecords<AccountVoiceClient>> GetAccountVoiceClientsByPage(int pageNumber, int pageSize)
        {
            try
            {
                return new ServiceResponse<PagedRecords<AccountVoiceClient>>(ServiceResponseStatus.Successful, AccountVoiceClient.GetAllByPage(pageNumber, pageSize));
            }
            catch (Exception ex)
            {
                Logger.Error("Get Account Voice by Page Exception.", ex);
                return new ServiceResponse<PagedRecords<AccountVoiceClient>>(ServiceResponseStatus.UnknownException, ex.Message + " -- " + ex.StackTrace);
            }
        }
        #endregion

        #endregion

        #region Country Calls

        public ServiceResponse<CountryBillingOption[]> GetCountryBillingOptions()
        {
            try
            {
                var countries = CountryBillingOption.GetAllCountrys();
                return new ServiceResponse<CountryBillingOption[]>(ServiceResponseStatus.Successful, countries.ToArray());
            }
            catch (Exception exc)
            {
                return new ServiceResponse<CountryBillingOption[]>(ServiceResponseStatus.UnknownException, exc.Message);
            }
        }

        #endregion


        public ServiceResponse<string> HealthCheck()
        {
            return new ServiceResponse<string>(ServiceResponseStatus.Successful, "I'm alive!");
        }

        public void SyncAccountSubscriptions(string accountID)
        {

            BillingSync.Instance.SyncAccountSubscriptions(accountID);

        }

        public void SyncSuspectAccountSubscriptions()
        {

            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = "select AccountID from AccountEntitlement as a where a.EntitlementID = 'CP' and a.DateExpires > GETUTCDATE() and not exists(select top 1 1 from Subscription as b where b.AccountID = a.AccountID and b.ProductID = 'CP' and b.DateExpires >= GETUTCDATE()) ";

                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        string id = reader.GetString(0);
                        BillingSync.Instance.SyncAccountSubscriptions(id);
                    }
                }
            }
        }
    }
}
