﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Threading;
using Curse.Logging;
using Vindicia;
using Curse;
using Curse.Extensions;
using Curse.Billing.Data;
using System.Data.SqlClient;
using System.Data;
using System.Net;
using System.Configuration;
using Curse.Billing.Configuration;
using Curse.Billing.Extensions;
using Curse.Billing.Models;

namespace Curse.Billing
{
    public class BillingSync
    {
       
        #region Singleton Implementation

        private static readonly BillingSync _instance = new BillingSync();

        public static BillingSync Instance
        {
            get
            {
                return _instance;
            }
        }

        #endregion

        #region Fields

        private Thread _syncThread;
        private bool _isAlive = true;

        private DateTime _lastSubscriptionSyncTime;
        private DateTime _lastEntitlementSyncTime;
        private DateTime _lastDailyEntitlementSyncTime;
        private DateTime _lastTransactionSyncTime;

        #endregion

        #region Constructor

        public BillingSync()
        {
            Logger.Info("BillingSync - Initializing...");

            // Offset times control how far back the sync will go when starting up
            _lastSubscriptionSyncTime = DateTime.UtcNow.AddHours(GetOffsetTime("SubscriptionSyncInitialOffsetHours", 25));
            _lastEntitlementSyncTime = DateTime.UtcNow.AddHours(GetOffsetTime("EntitlementSyncInitialOffsetHours", 6));
            _lastDailyEntitlementSyncTime = DateTime.UtcNow.AddHours(GetOffsetTime("DailyEntitlementSyncInitialOffsetHours", 25));
            _lastTransactionSyncTime = DateTime.UtcNow.AddHours(GetOffsetTime("TransactionSyncInitialOffsetHours", 1));

            _syncThread = new Thread(SyncThread);
            _syncThread.Start();

            Logger.Info("BillingSync - Initialization Successful!");
        }

        private static int GetOffsetTime(string settingName, int defaultOffset)
        {
            string offsetStr = ConfigurationManager.AppSettings.Get(settingName);
            int offset = defaultOffset;
            int.TryParse(offsetStr, out offset);
            if (offset < 0)
            {
                throw new Exception(settingName + " must be >= 0 to prevent sync gaps");
            }
            return -offset;
        }
        
        #endregion
       
        public void Start()
        {        
            _isAlive = true;
        }

        public void Shutdown()
        {
            _isAlive = false;
        }

        // Every 10 mins
        private const int ThreadSleep = 1000 * 60 * 10;       

        public void SyncThread()
        {
            while (_isAlive)
            {

                DateTime syncStartTime;

                // Subscriptions
                try
                {
                    syncStartTime = DateTime.UtcNow;
                    this.SyncSubscriptions();
                    _lastSubscriptionSyncTime = syncStartTime;                  
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "SyncSubscriptions Exception! Details: {0}");
                }

                // Transactions
                try
                {
                    syncStartTime = DateTime.UtcNow;
                    this.SyncTransactions();
                    _lastTransactionSyncTime = syncStartTime;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "SyncTransactions Exception! Details: {0}");
                }

                // Entitlements
                try
                {
                    syncStartTime = DateTime.UtcNow;
                    this.SyncEntitlements();
                    _lastEntitlementSyncTime = syncStartTime;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "SyncEntitlements Exception! Details: {0}");
                }

#if DO_DAILY_SYNC
                if ((DateTime.UtcNow - _lastDailyEntitlementSyncTime).TotalDays > 1)
                {
                    // Entitlements
                    try
                    {
                        syncStartTime = DateTime.UtcNow;
                        this.SyncDailyEntitlements();
                        _lastDailyEntitlementSyncTime = syncStartTime;
                    }
                    catch (Exception ex)
                    {
                        Logger.Log("SyncDailyEntitlements Exception! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
                    }
                }

#endif
                Thread.Sleep(ThreadSleep);
            }
        }


        private void SyncEntitlements()
        {

            DateTime syncFrom = _lastEntitlementSyncTime.PrepareForVindicia().AddMinutes(-5);
            DateTime syncTo = DateTime.UtcNow.PrepareForVindicia();
            Logger.Info(string.Format("Syncing entitlements {0}-{1}...", syncFrom, syncTo));

            Entitlement[] updatedEntitlements = Retrieval.GetAllEntitlementsFromDate(syncFrom).OrderBy(e => e.active).ThenBy(e => e.endTimestamp).ToArray();

            if (updatedEntitlements.Length == 0)
            {
                Logger.Info("No entitlements have been updated.");
                return;
            }

            Logger.Info(String.Format("Syncing {0} entitlements...", updatedEntitlements.Length));
            
            SaveEntitlements(updatedEntitlements);
           
            Logger.Info("All entitlements synced");   
            
        }

        private static void SaveEntitlements(Entitlement[] updatedEntitlements, bool syncAll = false)
        {
            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                foreach (Entitlement entitlement in updatedEntitlements)
                {
                    bool toSync = false;
                    if (!syncAll)
                    {
                        Subscription[] subs = Subscription.GetAllByAccountID(entitlement.account.merchantAccountId).ToArray();
                        
                        foreach (var sub in subs)
                        {
                            if (sub.Status != SubscriptionStatus.Suspended && sub.Status != SubscriptionStatus.PendingCustomerAction)
                            {
                                toSync = true;
                            }
                        }
                    }
                    else
                    {
                        toSync = true;
                    }

                    if (toSync)
                    {
                        AccountInformation account = AccountInformation.GetByID(entitlement.account.merchantAccountId);
                        if (account != null)
                        {
                            AccountEntitlement accountEntitlement = AccountEntitlement.FromVindicia(entitlement);
                            account.UpdateEntitlement(accountEntitlement);
                        }
                    }
                }
            }
        }


#if DO_DAILY_SYNC
        private void SyncDailyEntitlements()
        {

            DateTime syncFrom = _lastDailyEntitlementSyncTime.AddDays(-1).AddMinutes(-5).PrepareForVindicia();

            Logger.Info(String.Format("Syncing daily entitlements from '{0}'...", syncFrom));

            Entitlement[] updatedEntitlements = Retrieval.GetAllEntitlementsFromDate(syncFrom).OrderBy(e => e.active).ThenBy(e => e.endTimestamp).ToArray();

            if (updatedEntitlements.Length == 0)
            {
                Logger.Debug("No daily entitlements have been updated.");
                return;
            }

            Logger.Info(String.Format("Syncing daily {0} entitlements...", updatedEntitlements.Length));

            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                foreach (Entitlement entitlement in updatedEntitlements)
                {

                    Subscription[] subs = Subscription.GetAllByAccountID(entitlement.account.merchantAccountId).ToArray();
                    bool toSync = false;
                    foreach (var sub in subs)
                    {
                        if (sub.Status != SubscriptionStatus.Suspended && sub.Status != SubscriptionStatus.PendingCustomerAction)
                        {
                            toSync = true;
                        }
                    }

                    if (toSync)
                    {
                        string accountID = entitlement.account.merchantAccountId;
                        AccountInformation account = AccountInformation.GetByID(accountID);
                        if (account != null)
                        {
                            AccountEntitlement accountEntitlement = AccountEntitlement.FromVindicia(entitlement);
                            account.UpdateEntitlement(accountEntitlement);
                        }
                    }
                }
            }

            Logger.Info("All daily entitlements synced.");

        }
#endif

        private void SyncTransactions()
        {            
            Logger.Info("Syncing transactions...");


            Transaction[] updatedTransactions = Retrieval.GetAllTransactionsFromDate(_lastTransactionSyncTime.PrepareForVindicia().AddMinutes(-5));

            if (updatedTransactions.Length == 0)
            {
                Logger.Info("No transactions have been updated.");
                return;
            }

            Logger.Info(String.Format("Syncing {0} transactions...", updatedTransactions.Length));

            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                foreach (Transaction updatedTransaction in updatedTransactions)
                {

                    // Get the transaction by ID first:
                    AccountTransaction accountTransaction = AccountTransaction.GetByTransactionID(updatedTransaction.merchantTransactionId, false);
                    if (accountTransaction != null)
                    {
                        if (updatedTransaction.statusLog.Length > 0)
                        {
                            AccountTransactionStatus status = updatedTransaction.statusLog[0].status.ToCurse();
                            if (accountTransaction.Status != status)
                            {
                                accountTransaction.Status = status;
                                accountTransaction.Save();
                            }
                        }
                    }
                    else
                    {
                        accountTransaction = AccountTransaction.FromVindicia(updatedTransaction);
                        accountTransaction.Save();
                    }                    
                }
            }

            Logger.Info("All transactions synced.");
        }

        private void SyncSubscriptions()
        {            
            Logger.Info("Syncing subscriptions...");

            AutoBill[] autoBills = Retrieval.GetAllAutoBillsFromDate(_lastSubscriptionSyncTime.PrepareForVindicia().AddMinutes(-5));

            if (autoBills.Length == 0)
            {
                Logger.Info("No subscriptions have been updated.");
                return;
            }

            Logger.Info(String.Format("Syncing {0} subscriptions...", autoBills.Length));

            SyncSubscriptions(autoBills);

            Logger.Info("All transactions synced.");
        }

        public void SyncAccountSubscriptions(string accountID)
        {
            Logger.Info("Syncing subscriptions...");

            var autoBills = Retrieval.GetAllAutoBillsByAccountID(accountID);
            var entitlements = Retrieval.GetAllEntitlementsByAccountID(accountID, false);
        

            Logger.Info(String.Format("Syncing {0} subscriptions...", autoBills.Length));

            SyncSubscriptions(autoBills);

            Logger.Info(String.Format("Syncing {0} entitlements...", entitlements.Length));

            SaveEntitlements(entitlements);

            Logger.Info("All transactions synced.");
        }

        private static void SyncSubscriptions(AutoBill[] autoBills)
        {
            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                foreach (AutoBill autoBill in autoBills)
                {
                    if (autoBill.merchantAutoBillId.IsNullOrEmpty())
                    {
                        continue;
                    }
                    // Get the subsciption by ID first:
                    Subscription subscription = Subscription.GetBySubscriptionID(autoBill.merchantAutoBillId, false);
                    Subscription updatedSubscription = Subscription.FromVindicia(autoBill);

                    if (subscription != null)
                    {
                        subscription.DateExpires = updatedSubscription.DateExpires;
                        subscription.NextBillAmount = updatedSubscription.NextBillAmount;
                        subscription.NextBillCurrency = updatedSubscription.NextBillCurrency;
                        subscription.NextBillingDate = updatedSubscription.NextBillingDate;
                        subscription.Status = updatedSubscription.Status;
                    }
                    else
                    {
                        subscription = updatedSubscription;
                    }

                    subscription.Save(conn, null);
                }
            }
        }


    }
}
