﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Runtime.Serialization;
using Vindicia;
using Curse.Extensions;
using Curse.Billing.Data;
using Curse.Billing.Extensions;
using System.Data.SqlClient;
using Curse.Billing.Configuration;

namespace Curse.Billing.Models
{
    [DataContract]
    public class AccountInformation
    {
        private static string[] _cacheExpirationKeys = { "AccountByID-{0}", "VindiciaAccountByID-{0}" };

        [DataMember]
        public string ID { get; set; }

        [DataMember]
        public string Name { get; set; }

        [DataMember]
        public string Company { get; set; }

        [DataMember]
        public string EmailAddress { get; set; }

        [DataMember]
        public AccountAddress Address { get; set; }

        [DataMember]
        public List<AccountPaymentMethod> PaymentMethods { get; set; }

        [DataMember]
        public List<Subscription> Subscriptions
        {
            get
            {
                return Subscription.GetAllByAccountID(this.ID);
            }
            set { }            
        }

        [DataMember]
        public List<AccountEntitlement> Entitlements
        {
            get
            {
                return AccountEntitlement.GetByAccountID(this.ID);
            }
            set { }
        }

        [DataMember]
        public CurrencyType PreferredCurrency { get; set; }

        [DataMember]
        public List<AccountTokenAmount> TokenBalances { get; private set; }

        public AccountInformation() { }

        public static AccountInformation FromVindicia(Account vindiciaAccount)
        {
            
            AccountInformation accountInformation = new AccountInformation()
            {
                ID = vindiciaAccount.merchantAccountId,
                Name = vindiciaAccount.name,
                Company = vindiciaAccount.company,
                EmailAddress = vindiciaAccount.emailAddress,
            };
            if (vindiciaAccount.shippingAddress != null)
            {
                accountInformation.Address = new AccountAddress(vindiciaAccount.shippingAddress);
            }

            // Payment Methods
            List<AccountPaymentMethod> paymentMethods = new List<AccountPaymentMethod>();
            if (vindiciaAccount.paymentMethods != null)
            {
                foreach (PaymentMethod method in vindiciaAccount.paymentMethods)
                {
                    paymentMethods.Add(AccountPaymentMethod.FromVindicia(method));
                }
            }
            accountInformation.PaymentMethods = paymentMethods;

            
            
            // Preferred Currency - Default to USD
            accountInformation.PreferredCurrency = CurrencyType.USD;           

            if (vindiciaAccount.nameValues != null)
            {
                NameValuePair prefCurrency = vindiciaAccount.nameValues.FirstOrDefault(p => p.name == "PreferredCurrency");
                if (prefCurrency != null)
                {
                    try
                    {
                        accountInformation.PreferredCurrency = (CurrencyType)Enum.Parse(typeof(CurrencyType), prefCurrency.value, true);
                    }
                    catch { }
                }
            }
            // Convert Token balances
            if (vindiciaAccount.tokenBalances != null && vindiciaAccount.tokenBalances.Any())
            {
                accountInformation.TokenBalances = vindiciaAccount.tokenBalances.Select(ta => AccountTokenAmount.FromVindicia(ta)).Distinct(new AccountTokenAmountIDComparer()).ToList();                    
            }
            else
            {
                accountInformation.TokenBalances = new List<AccountTokenAmount>();
            }

            return accountInformation;

        }       

        public Account ToVindiciaAccount()
        {            
            Account account = new Account()
            {
                merchantAccountId = this.ID,
                name = this.Name,
                company = this.Company,
                emailAddress = this.EmailAddress,
                shippingAddress = this.Address.ToVindicia()
            };

            // Set their preferred currency, default to USD
            if (this.PreferredCurrency == CurrencyType.Undefined)
            {
                this.PreferredCurrency = CurrencyType.USD;
            }

            account.nameValues = new NameValuePair[] { new NameValuePair() { name = "PreferredCurrency", value = this.PreferredCurrency.ToString() } };

            return account;
        }

        public static AccountInformation GetByID(string accountID)
        {
            return HttpRuntime.Cache.Get("AccountByID-{0}".FormatWith(accountID), () =>
            {
                Account a = Retrieval.GetAccountByID(accountID);

                if (a == null)
                {
                    return null;
                }
                else
                {          
                    //Retrieve the account info from Vindicia
                    AccountInformation accountInformation = AccountInformation.FromVindicia(a);
                    
                    return accountInformation;
                }
            });  
        }

        public void CreateOrUpdatePaymentMethod(AccountPaymentMethod paymentMethod)
        {
            if (paymentMethod.ID.IsNullOrEmpty())
            {
                paymentMethod.ID = "{0}-{1}".FormatWith(this.ID, DateTime.UtcNow.Ticks);
                paymentMethod.IsActive = true;
            }
           
            PaymentMethod vindiciaPaymentMethod = paymentMethod.ToVindicia();
            Account vindiciaAccount = this.ToVindiciaAccount();
            bool validated = false;
            Return ret = vindiciaAccount.updatePaymentMethod(vindiciaPaymentMethod, false, PaymentUpdateBehavior.Update, 0, out validated);

            if (ret.returnCode != ReturnCode.Item200)
            {
                throw new ArgumentException("The payment method could not be added: {0}".FormatWith(ret.returnString));
            }

            paymentMethod.Expire();
            this.Expire();            
        }

        public void UpdateEntitlement(AccountEntitlement newEntitlement)
        {
            AccountEntitlement existingEntitlement = this.Entitlements.FirstOrDefault(p => p.EntitlementID == newEntitlement.EntitlementID);

            if (existingEntitlement != null)
            {
                existingEntitlement.DateExpires = newEntitlement.DateExpires.ConvertFromVindicia();
                existingEntitlement.DateUpdated = DateTime.UtcNow;
                existingEntitlement.Active = newEntitlement.Active;
                existingEntitlement.Save();
                this.Expire();
                
            }
            else
            {
                this.Entitlements.Add(newEntitlement);
                newEntitlement.Save();
                this.Expire();
            }            
        }

        public void ResyncEntitlements(Entitlement[] updatedEntitlements)
        {
            foreach (Entitlement entitlement in updatedEntitlements)
            {
                AccountEntitlement accountEntitlement = AccountEntitlement.FromVindicia(entitlement);
                this.UpdateEntitlement(accountEntitlement);
            }  
        }

        public void ResyncEntitlements()
        {
            // Order so deactivations are first
            Entitlement[] updatedEntitlements = Retrieval.GetAllEntitlementsByAccountID(this.ID, false).OrderBy(e => e.active).ThenBy(e => e.endTimestamp).ToArray();
            ResyncEntitlements(updatedEntitlements);  
        }

        public void ResyncActiveEntitlementsOnly()
        {
            // Order so deactivations are first
            Entitlement[] updatedEntitlements = Retrieval.GetAllEntitlementsByAccountID(this.ID, true).OrderBy(e => e.endTimestamp).ToArray();
            ResyncEntitlements(updatedEntitlements);
        }

        public void Update()
        {
            bool created = false;          
            Account vindiciaAccount = this.ToVindiciaAccount();
            vindiciaAccount.update(out created);
            this.Expire();
        }
      
        public void Expire()
        {

            // Expire AccountInformation object cache
            foreach (string key in _cacheExpirationKeys)
            {
                HttpRuntime.Cache.Remove(key.FormatWith(this.ID));
            }

            // Expire AccountEntitlement object cache
            AccountEntitlement.ExpireByAccountID(this.ID);
        }

        public CreateSubscriptionResult CreateSubscription(int userID, int cobaltInstanceID, int entityTypeID, int entityID, int? transactionID, string productID, string billingPlanID, AccountPaymentMethod paymentMethod, DateTime? startDate, AccountTransactionType transactionType, CurrencyType currency)
        {
            ProductOffering product = ProductOffering.GetByID(productID);
            if (product == null) {
                throw new ArgumentException("Create Subscription failed. Product '{0}' does not exist.".FormatWith(productID), "product");
            }

            ProductBillingPlan billingPlan = ProductBillingPlan.GetByID(billingPlanID);
            if (billingPlan == null) {
                throw new ArgumentException("Create Subscription failed. Billing Plan '{0}' does not exist.".FormatWith(billingPlanID), "billingPlanID");
            }
            
            // If the plan is set to start in the future, get the delayed version of it
            if (startDate.HasValue) {
                billingPlan = ProductBillingPlan.GetByID(billingPlan.DelayedPlanID);

                if (billingPlan == null) {
                    throw new ArgumentException("Create Subscription failed. Billing Plan '{0}' has no delayed billing plan ID.".FormatWith(billingPlanID), "billingPlanID");
                }
            }           

            // Open a DB connection, and begin a transaction
            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection()) {
                SqlTransaction txn = conn.BeginTransaction();

                // Save the subscription
                Subscription subscription = new Subscription() {                                       
                    ProductID = product.ID,
                    BillingPlanID = billingPlan.ID,
                    PaymentMethodID = paymentMethod != null ? paymentMethod.ID : null,
                    AccountID = this.ID,
                    DateStarted = startDate.HasValue ? startDate.Value : DateTime.UtcNow,
                    DateExpires = DateTime.UtcNow,
                };
                if (paymentMethod != null)
                {
                    subscription.Status = paymentMethod.Type == AccountPaymentType.PayPal ? SubscriptionStatus.PendingCustomerAction : SubscriptionStatus.Active;
                }
                else
                {
                    subscription.Status =  SubscriptionStatus.Active;
                }
                 
                
                subscription.Save(conn, txn);
                
                // Create the Vindicia auto bill
                AutoBill ab = new AutoBill() {
                    account = new Account() { merchantAccountId = this.ID },
                    product = new Product() { merchantProductId = product.ID },
                    billingPlan = new BillingPlan { merchantBillingPlanId = billingPlan.ID },                    
                    merchantAutoBillId = subscription.SubscriptionID,
                    customerAutoBillName = billingPlan.Description,
                    currency = currency.ToString()
                };

                if (paymentMethod != null) {
                    ab.paymentMethod = paymentMethod.ToVindicia();
                }
                                                
                ab.startTimestamp = subscription.DateStarted.PrepareForVindicia();
                
                bool created = false;
                bool validatePayment = true;
                int minChargeBackProb = 50;
                TransactionStatus transactionStatus;
                DateTime firstBillDate;
                decimal firstBillAmount;
                string firstBillCurrency;
                int score;
                ScoreCode[] scoreCodes;

                string preventFirstBillingStr;
                bool preventFirstBilling = false;
                if (billingPlan.MerchantData.TryGetValue("PreventFirstBilling", out preventFirstBillingStr)) {
                    bool.TryParse(preventFirstBillingStr, out preventFirstBilling);
                }

                // If the first billing period has a 0 cost, do not validate the payment. 
                // Note - since this won't process a PayPal sub correctly without validating, we will continue to validate if PreventFirstBilling is set
                if (billingPlan.Periods.First().Prices.First().Amount == 0 && !preventFirstBilling) {
                    validatePayment = false;
                }

                // Disable payment validation for token subscriptions
                if (billingPlan.FirstPayPeriod.Try(pbp => pbp.Prices.First().Currency == CurrencyType.Token)) {
                    AccountTokenType tokenType = billingPlan.FirstPayPeriod.Prices.First().TokenType;
                    validatePayment = false;
                    minChargeBackProb = 100;
                    ab.currency = "_VT";    // This means tokens
                }

                bool successful = false;
                try {
                    Return ret = ab.update(DuplicateBehavior.Fail, true, validatePayment, minChargeBackProb, out created, out transactionStatus, out firstBillDate, out firstBillAmount, out firstBillCurrency, out score, out scoreCodes);

                    if (transactionStatus.payPalStatus != null) {                    
                        successful = ret.returnCode == ReturnCode.Item200 && transactionStatus.status == TransactionStatusType.AuthorizationPending;
                    }
                    else {
                        successful = ret.returnCode == ReturnCode.Item200;
                    }               
                }
                catch {
                    // The autobill failed, so rollback our transaction
                    txn.Rollback();

                    // Return null
                    return new CreateSubscriptionResult() {
                        IsSuccessful = false
                    };
                }

                if (successful) {
                    // Update the subscription with the autobill's data
                    subscription.UpdateFromVindicia(ab);
                    subscription.Save(conn, txn);                            
                    
                    // Commit the transaction
                    txn.Commit();

                    // Save the subscription history
                    SubscriptionHistory history = new SubscriptionHistory() {
                        SubscriptionID = subscription.SubscriptionID,
                        UserID = userID,
                        CobaltInstanceID = cobaltInstanceID,
                        EntitledAccountID = this.ID,
                        EntitledEntityTypeID = entityTypeID,
                        EntitledEntityID = entityID,
                        Timestamp = DateTime.UtcNow,
                        TransactionID = transactionID,
                        ProductID = product.ID,
                        BillingPlanID = billingPlan.ID,
                        PaymentMethodID = paymentMethod != null ? paymentMethod.ID : null,
                    };
                    history.Save(conn, null);
                    subscription.SetHistoryData();

                    AccountTransaction accountTransaction = null;
                    // Create transaction, if no transaction ID was passed as a parameter
                    if (!transactionID.HasValue) {
                        AccountTransactionStatus status = AccountTransactionStatus.New;

                        if (transactionStatus.payPalStatus != null) {
                            status = AccountTransactionStatus.Pending;
                        }

                        AccountTransactionItem item = new AccountTransactionItem() {
                            ProductID = product.ID,
                            Description = billingPlan.Description,
                            Price = 0,
                            Quantity = 1,
                            SubTotal = 0
                        };
                        accountTransaction = AccountTransaction.Create(userID, this.ID, paymentMethod, this.PreferredCurrency, 0, 0, 0, new AccountTransactionItem[] { item }, transactionType, status, null, transactionStatus.payPalStatus != null ? transactionStatus.payPalStatus.token : null);
                        accountTransaction.Save();
                        transactionID = accountTransaction.ID;
                    }
                    else {
                        accountTransaction = AccountTransaction.GetByID(transactionID.Value, false);
                    }

                    //only resync entitlements if the subscription came back as active. Otherwise, pending subscriptions
                    //will give user's immediate CP entitlements in Auth.
                    if (subscription.Status == SubscriptionStatus.Active)
                    {
                        // Resynchronize entitlements
                        this.ResyncEntitlements();    
                    }

                    // Expire the data cache
                    this.Expire();

                    // Return our version of the subscription object                    
                    return new CreateSubscriptionResult() {
                        IsSuccessful = true,
                        PayPalRedirectUrl = (transactionStatus.payPalStatus != null) ? transactionStatus.payPalStatus.redirectUrl : null,
                        Subscription = subscription,
                        Transaction = accountTransaction
                    };
                }
                else {
                    // The autobill failed, so rollback our transaction
                    txn.Rollback();

                    // Return null
                    return new CreateSubscriptionResult() {
                        IsSuccessful = false
                    };
                }
            }
        }

        public void CancelSubscription(int userID, string subscriptionID, string note, bool disentitle)
        {
            Subscription subscription = this.Subscriptions.FirstOrDefault(p => p.SubscriptionID == subscriptionID);
            subscription.Cancel(disentitle);

            ProductBillingPlan plan = ProductBillingPlan.GetByID(subscription.BillingPlanID);

            // Record the transaction
            AccountTransactionItem item = new AccountTransactionItem()
            {
                ProductID = subscription.ProductID,
                Description = plan.Description,
                Price = 0,
                Quantity = 1,
                SubTotal = 0
            };
            AccountTransaction accountTransaction = AccountTransaction.Create(userID, this.ID, null, this.PreferredCurrency, 0, 0, 0, new AccountTransactionItem[] { item }, AccountTransactionType.CancelSubscription, AccountTransactionStatus.Cancelled, note, null);
            accountTransaction.Save(); 
           
            // Add the cancelled date to the subscription

            // Expire the data cache
            this.Expire();
        }

        public bool CreditSubscription(int userID, string subscriptionID, int days, AccountTransactionType transactionType, string note)
        {

            Subscription subscription = this.Subscriptions.FirstOrDefault(p => p.SubscriptionID == subscriptionID);
            if (subscription == null)
            {
                throw new ArgumentException("Subscription '{0}' was not found.".FormatWith(subscriptionID), "subscriptionID");
            }

            bool wasCancelled = subscription.Status == SubscriptionStatus.Cancelled;

            // Create a Vindicia AutoBill object, given their unique ID 
            AutoBill ab = new AutoBill() { merchantAutoBillId = subscriptionID };
            DateTime nextBillingDate;
            Decimal nextBillingAmount;
            string nextBillingCurrency;

            // Attempt to delay auto billing
            Return sr = ab.delayBillingByDays(days, true, out nextBillingDate, out nextBillingAmount, out nextBillingCurrency);

            bool successful = (sr.returnCode == ReturnCode.Item200);

            // If it was successful, save an activity
            if (successful)
            {
                this.ToVindiciaAccount().SaveActivityNote("Subscription '{0}' credited with {1} days".FormatWith(subscriptionID, days));

                // Record a transaction
                AccountTransaction accountTransaction = AccountTransaction.Create(userID, this.ID, null, CurrencyType.USD, 0.00M, 0.00M, 0.00M, new AccountTransactionItem[] { }, transactionType, AccountTransactionStatus.Internal, note, null);

                // Re-cancel the subscription, if it was cancelled
                if (wasCancelled)
                {
                    ab.cancel(false, false);
                }

                subscription.UpdateFromVindicia(ab);
                subscription.Save();

                // Resynchronize entitlements
                this.ResyncEntitlements();

                this.Expire();
            }


            return successful;
        }
 
        public List<Coupon> GetPurchasedCoupons()
        {
            return Coupon.GetAllByPurchasedAccountID(this.ID);
        }

        public List<Coupon> GetClaimedCoupons()
        {
            return Coupon.GetAllByClaimedAccountID(this.ID);
        }

        public bool CreditTokens(string adminName, IEnumerable<AccountTokenAmount> tokenAmounts)
        {
            Account acct = new Account()
            {
                merchantAccountId = this.ID
            };
            TokenAmount[] amounts = tokenAmounts.Select(ta =>
                new TokenAmount()
                {
                    token = new Token()
                    {
                        merchantTokenId = ta.Type.ID
                    },
                    amount = ta.Amount
                }).ToArray();

            Return sr = acct.incrementTokens(ref amounts);
            bool successful = (sr.returnCode == ReturnCode.Item200);

            // If it was successful, save an activity
            if (successful)
            {
                this.ToVindiciaAccount().SaveActivityNote("Account credited with {0} tokens ({1})".FormatWith(amounts.Sum(a => a.amount), adminName));
                this.Expire();
            }

            return successful;
        }

        public bool DeductTokens(string adminName, IEnumerable<AccountTokenAmount> tokenAmounts)
        {
            Account acct = new Account()
            {
                merchantAccountId = this.ID
            };
            TokenAmount[] amounts = tokenAmounts.Select(ta =>
                new TokenAmount()
                {
                    token = new Token()
                    {
                        merchantTokenId = ta.Type.ID
                    },
                    amount = ta.Amount
                }).ToArray();

            Return sr = acct.decrementTokens(ref amounts);
            bool successful = (sr.returnCode == ReturnCode.Item200);

            // If it was successful, save an activity
            if (successful)
            {
                this.ToVindiciaAccount().SaveActivityNote("Account deducted {0} tokens ({1})".FormatWith(amounts.Sum(a => a.amount), adminName));
                this.Expire();
            }

            return successful;
        }
    }
}
