﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Runtime.Serialization;
using Vindicia;
using Curse.Extensions;
using Curse.Billing.Data;
using Curse.Billing.Extensions;


namespace Curse.Billing.Models
{
    [Serializable]
    [DataContract]
    public class AccountPaymentMethod
    {
        private const string CacheKey = "AccountPaymentMethodByID-{0}";

        [DataMember]
        public string ID
        {
            get;
            set;
        }

        [DataMember]
        public string Description
        {
            get;
            set;
        }

        [DataMember]
        public string VindiciaID
        {
            get;
            set;
        }

        [DataMember]
        public AccountPaymentType Type
        {
            get;
            set;
        }

        [DataMember]
        public string AccountHolderName
        {
            get;
            set;
        }

        [DataMember]
        public bool IsActive
        {
            get;
            set;
        }

        [DataMember]
        public AccountAddress Address
        {
            get;
            set;
        }

        [DataMember]
        public string AccountNumber
        {
            get;
            set;
        }

        [DataMember]
        public string BankSortCode
        {
            get;
            set;
        }

        [DataMember]
        public int ExpirationMonth
        {
            get;
            set;
        }

        [DataMember]
        public int ExpirationYear
        {
            get;
            set;
        }

        [DataMember]
        public string EmailAddress
        {
            get;
            set;
        }

        [DataMember]
        public string PayPalReturnUrl
        {
            get;
            set;
        }

        [DataMember]
        public string PayPalCancelUrl
        {
            get;
            set;
        }

        [DataMember]
        public AccountTokenType TokenType
        {
            get;
            set;
        }

        [DataMember]
        public int SortOrder
        {
            get;
            set;
        }

        [DataMember]
        public string CvnCode
        {
            get;
            set;
        }

        public static AccountPaymentMethod GetByID(string id)
        {
            return HttpRuntime.Cache.Get(false, CacheKey.FormatWith(id), () =>
            {
                PaymentMethod paymentMethod = new PaymentMethod();
                Return ret = paymentMethod.fetchByMerchantPaymentMethodId(id);
                if(ret.returnCode == ReturnCode.Item200)
                {
                    return FromVindicia(paymentMethod);
                }
                else
                {
                    return null;
                }
            });
        }

        public void Expire()
        {
            HttpRuntime.Cache.Expire(CacheKey.FormatWith(this.ID));
        }    

        public static AccountPaymentMethod FromVindicia(PaymentMethod paymentMethod)
        {
            AccountPaymentMethod method = new AccountPaymentMethod()
            {
                ID = paymentMethod.merchantPaymentMethodId,
                AccountHolderName = paymentMethod.accountHolderName,
                IsActive = paymentMethod.active,
                SortOrder = paymentMethod.sortOrder,                
                Description = paymentMethod.customerDescription,
            };

            if (paymentMethod.billingAddress != null)
            {
                method.Address = new AccountAddress(paymentMethod.billingAddress);
            }

            switch (paymentMethod.type)
            {
                // TODO: This needs work to set the AMEX/Maestro
                case PaymentMethodType.CreditCard:
                    method.Type = AccountPaymentType.CreditCard;
                    method.AccountNumber = paymentMethod.creditCard.account;                    
                    method.ExpirationYear = int.Parse(paymentMethod.creditCard.expirationDate.Substring(0, 4));
                    method.ExpirationMonth = int.Parse(paymentMethod.creditCard.expirationDate.Substring(4));
                    break;
                case PaymentMethodType.PayPal:
                    method.Type = AccountPaymentType.PayPal;                    
                    if (paymentMethod.paypal != null)
                    {
                        method.EmailAddress = paymentMethod.paypal.emailAddress;
                        method.PayPalCancelUrl = paymentMethod.paypal.cancelUrl;
                        method.PayPalReturnUrl = paymentMethod.paypal.returnUrl;
                    }                    
                    break;
                case PaymentMethodType.Token: //TODO: can all this voice stuff go bye bye?
                    method.Type = AccountPaymentType.Token;
                    method.TokenType = AccountTokenType.FromVindicia(paymentMethod.token);
                    break;
                case PaymentMethodType.DirectDebit:
                    method.Type = AccountPaymentType.DirectDebit;
                    method.AccountNumber = paymentMethod.directDebit.account;
                    method.BankSortCode = paymentMethod.directDebit.bankSortCode;
                    break;
                default:
                    throw new NotImplementedException("Unknown payment method: " + paymentMethod.type);
            }

            return method;
        }

        public PaymentMethod ToVindicia()
        {
            PaymentMethod paymentMethod = new PaymentMethod()
            {
                accountHolderName = this.AccountHolderName,
                active = this.IsActive,
                customerDescription = this.Description,
                merchantPaymentMethodId = this.ID,
                sortOrder = this.SortOrder,
                type = this.Type.ToVindicia(),                
            };

            if (this.Address != null) {
                paymentMethod.billingAddress = this.Address.ToVindicia();
            }

            switch (this.Type)
            {
                case AccountPaymentType.AMEX:
                case AccountPaymentType.Maestro:
                case AccountPaymentType.CreditCard:
                    paymentMethod.creditCard = new CreditCard()
                    {
                        account = this.AccountNumber,
                        expirationDate = this.ExpirationYear.ToString() + this.ExpirationMonth.ToString("00")
                    };

                    if (!this.CvnCode.IsNullOrEmpty())
                    {
                        paymentMethod.nameValues = new NameValuePair[]
                        {
                            new NameValuePair() { name = "CVN", value = this.CvnCode } 
                        };
                    }

                    break;

                case AccountPaymentType.PayPal:
                    paymentMethod.paypal = new PayPal()
                    {
                        emailAddress = this.EmailAddress,
                        returnUrl = this.PayPalReturnUrl,
                        cancelUrl = this.PayPalCancelUrl
                    };
                    break;
                case AccountPaymentType.Token:
                    paymentMethod.token = this.TokenType.ToVindicia();
                    paymentMethod.currency = "_VT";
                    break;
                case AccountPaymentType.DirectDebit:
                    paymentMethod.directDebit = new DirectDebit
                    {
                        account = this.AccountNumber,
                        bankSortCode = this.BankSortCode,
                        countryCode = this.Address.Country
                    };
                    break;
                default:
                    throw new NotImplementedException("Unknown payment type: {0}".FormatWith(this.Type));
            }

            return paymentMethod;
        }

    }
}
