﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Runtime.Serialization;
using System.Data.SqlClient;
using Curse.Extensions;
using Curse.Billing.Configuration;
using Curse.Billing.Extensions;
using System.Data;
using Vindicia;
using Curse.Billing.Data;

namespace Curse.Billing.Models
{

    [DataContract]
    public class AccountTransaction
    {
        private const string CacheKey = "AccountTransactionByID-{0}";

        public AccountTransaction()
        {
        }

        protected AccountTransaction(SqlDataReader reader)
        {
            this.ID = reader.GetInt32(0);
            this.UserID = reader.GetInt32(1);
            this.DateCreated = reader.GetDateTime(2);
            this.DateModified = reader.GetDateTime(3);
            this.AccountID = reader.GetString(4);
            this.SubTotal = reader.GetDecimal(5);
            this.Tax = reader.GetDecimal(6);
            this.Total = reader.GetDecimal(7);
            this.Status = (AccountTransactionStatus)reader.GetByte(8);
            if (reader[9] != System.DBNull.Value)
            {
                this.PaymentMethodID = reader.GetString(9);
            }

            this.Currency = (CurrencyType)reader.GetInt32(10);
            this.Type = (AccountTransactionType)reader.GetByte(11);
            this.TransactionID = reader.GetString(12);            
            if (reader[13] != System.DBNull.Value)
            {
                this.StatusUserID = reader.GetInt32(13);
            }
            if (reader[14] != System.DBNull.Value)
            {
                this.Note = reader.GetString(14);                
            }

            if (reader[15] != System.DBNull.Value)
            {
                this.PayPalToken = reader.GetString(15);
            }
            if (reader[16] != System.DBNull.Value)
            {
                this.PaymentMethodType = (AccountPaymentType)reader.GetByte(16);
            }
                        
            this.Items = AccountTransactionItem.GetAllByTransactionID(this.ID).ToArray();            
        }

        #region Properties

        [DataMember]
        public int ID
        {
            get;
            set;
        }

        [DataMember]
        public int UserID
        {
            get;
            set;
        }

        [DataMember]
        public DateTime DateCreated
        {
            get;
            set;
        }

        [DataMember]
        public DateTime DateModified
        {
            get;
            set;
        }

        [DataMember]
        public string AccountID
        {
            get;
            set;
        }

        [DataMember]
        public string PaymentMethodID
        {
            get;
            set;
        }

        [DataMember]
        public AccountPaymentType? PaymentMethodType
        {
            get;
            set;
        }


        [DataMember]
        public decimal SubTotal
        {
            get;
            set;
        }

        [DataMember]
        public decimal Tax
        {
            get;
            set;
        }

        [DataMember]
        public decimal Total
        {
            get;
            set;
        }

        [DataMember]
        public AccountTransactionStatus Status
        {
            get;
            set;
        }

        [DataMember]
        public AccountTransactionType Type
        {
            get;
            set;
        }

        [DataMember]
        public AccountTransactionItem[] Items
        {
            get;
            set;
        }

        [DataMember]
        public CurrencyType Currency
        {
            get;
            set;
        }

        [DataMember]
        public string TransactionID
        {
            get;
            set;
        }

        [DataMember]
        public int? StatusUserID
        {
            get;
            set;
        }

        [DataMember]
        public string Note
        {
            get;
            set;
        }

        public string PayPalToken
        {
            get;
            set;
        }


        [DataMember]
        public AccountPaymentMethod PaymentMethod
        {
            get
            {
                if (!this.AccountID.IsNullOrEmpty() && !this.PaymentMethodID.IsNullOrEmpty())
                {
                    return AccountInformation.GetByID(this.AccountID).Try(ai => ai.PaymentMethods.FirstOrDefault(p => p.ID == this.PaymentMethodID));
                }
                else
                {
                    return null;
                }
            }
            private set
            {
            }
        }

        #endregion

        public static AccountTransaction Create(int userID, string accountID, AccountPaymentMethod paymentMethod, CurrencyType currency, decimal subTotal, decimal tax, decimal total, AccountTransactionItem[] items, AccountTransactionType type, AccountTransactionStatus status, string note, string payPalToken)
        {

            // Create the transaction record
            AccountTransaction transaction = new AccountTransaction()
            {
                UserID = userID,                
                AccountID = accountID,                
                SubTotal = subTotal,
                Tax = tax,
                Total = total,
                Status = status,
                Items = items,
                Currency = currency,
                Type = type,
                Note = note,
                PayPalToken = payPalToken
            };
            if (paymentMethod != null)
            {
                transaction.PaymentMethodID = paymentMethod.ID;
                transaction.PaymentMethodType = paymentMethod.Type;
            }
            transaction.Save();
           
            return transaction;
        }

        public void Save()
        {
            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandType = System.Data.CommandType.StoredProcedure;
                cmd.CommandText = "spUpdateAccountTransaction";

                SqlParameter idParameter = null;
                idParameter = cmd.Parameters.Add("@ID", SqlDbType.Int);
                idParameter.Direction = ParameterDirection.InputOutput;
                if (this.ID > 0)
                {
                    idParameter.Value = this.ID;
                }

                SqlParameter transactionIDParameter = null;
                transactionIDParameter = cmd.Parameters.Add("@TransactionID", SqlDbType.VarChar);
                transactionIDParameter.Size = 64;
                transactionIDParameter.Direction = ParameterDirection.InputOutput;
                if (this.TransactionID != null)
                {
                    transactionIDParameter.Value = this.TransactionID;
                }
                else
                {
                    transactionIDParameter.Value = System.DBNull.Value;
                }

                SqlParameter param = null;

                param = cmd.Parameters.Add("@UserID", SqlDbType.Int);
                param.Value = (int)this.UserID;

                param = cmd.Parameters.Add("@AccountID", SqlDbType.VarChar);
                param.Size = 64;
                param.Value = this.AccountID;

                param = cmd.Parameters.Add("@DateCreated", SqlDbType.DateTime);
                if (this.DateCreated.Year >= 2000)
                {
                    param.Value = this.DateCreated;
                }
                else
                {
                    param.Value = System.DBNull.Value;
                }

                param = cmd.Parameters.Add("@PaymentMethodID", SqlDbType.VarChar);
                param.Size = 64;                
                if (this.PaymentMethodID != null)
                {
                    param.Value = this.PaymentMethodID;
                }
                else
                {
                    param.Value = System.DBNull.Value;
                }

                param = cmd.Parameters.Add("@SubTotal", SqlDbType.Money);                
                param.Value = this.SubTotal;

                param = cmd.Parameters.Add("@CurrencyID", SqlDbType.Int);
                param.Value = (int)this.Currency;

                param = cmd.Parameters.Add("@Tax", SqlDbType.Money);
                param.Value = this.Tax;

                param = cmd.Parameters.Add("@Total", SqlDbType.Money);
                param.Value = this.Total;

                param = cmd.Parameters.Add("@Status", SqlDbType.TinyInt);
                param.Value = (byte)this.Status;

                param = cmd.Parameters.Add("@Type", SqlDbType.TinyInt);
                param.Value = (byte)this.Type;

                param = cmd.Parameters.Add("@StatusUserID", SqlDbType.Int);
                if (this.StatusUserID.HasValue)
                {
                    param.Value = this.StatusUserID.Value;
                }
                else
                {
                    param.Value = System.DBNull.Value;
                }

                param = cmd.Parameters.Add("@Note", SqlDbType.NVarChar);
                param.Size = -1;
                if (this.Note == null)
                {
                    param.Value = System.DBNull.Value;
                }
                else
                {
                    param.Value = this.Note;                    
                }

                param = cmd.Parameters.Add("@PayPalToken", SqlDbType.VarChar);
                param.Size = 64;
                if (this.PayPalToken == null)
                {
                    param.Value = System.DBNull.Value;
                }
                else
                {
                    param.Value = this.PayPalToken;
                }

                param = cmd.Parameters.Add("@PaymentMethodType", SqlDbType.TinyInt);
                if (this.PaymentMethodType.HasValue)
                {
                    param.Value = (byte)this.PaymentMethodType;
                }
                else
                {
                    param.Value = System.DBNull.Value;
                }
                
                
                cmd.ExecuteNonQuery();
                
                if (this.ID == 0)
                {
                    this.ID = (int)idParameter.Value;
                    this.DateCreated = DateTime.UtcNow;
                    this.DateModified = DateTime.UtcNow;
                }

                if (this.TransactionID == null)
                {
                    this.TransactionID = (string)transactionIDParameter.Value;
                }

                // Create the transaction item record(s)
                if (this.Items != null)
                {
                    foreach (AccountTransactionItem item in this.Items)
                    {
                        item.TransactionID = this.ID;
                        item.Save();
                    }
                }
            }

            this.Expire();
        }

        public void FinalizePayPalCapture(string vid)
        {
            Transaction txn = new Transaction();
            Return r = txn.finalizePayPalAuth(vid, true);
            if (r.returnCode == ReturnCode.Item200)
            {
                this.Status = AccountTransactionStatus.Authorized;
                this.Save();
            }
            else
            {
                this.Status = AccountTransactionStatus.Failed;
                this.Save();
            }
        }
       
        public Transaction  ToVindicia()
        {
            AccountInformation accountInformation = AccountInformation.GetByID(this.AccountID);
            List<TransactionItem> transactionItems = new List<TransactionItem>();

            foreach (AccountTransactionItem item in this.Items)
            {
                transactionItems.Add(item.ToVindicia());
            }

            return new Transaction()
            {
                amount = this.Total,
                currency = this.Currency.ToString(),
                merchantTransactionId = this.TransactionID,
                timestamp = this.DateCreated.PrepareForVindicia(),
                account = new Account() { merchantAccountId = this.AccountID },
                shippingAddress = accountInformation.Address.ToVindicia(),
                transactionItems = transactionItems.ToArray(),
                sourcePaymentMethod = new PaymentMethod() { merchantPaymentMethodId = this.PaymentMethodID },
                note = this.Note
            };
        }

        public static AccountTransaction FromVindicia(Transaction transaction)
        {
            AccountTransaction accountTransaction = new AccountTransaction();
            accountTransaction.AccountID = transaction.account.merchantAccountId;

            if (transaction.currency == "_VT")
            {
                accountTransaction.Currency = CurrencyType.Token;
            }
            else
            {
                try
                {
                    accountTransaction.Currency = (CurrencyType)System.Enum.Parse(typeof(CurrencyType), transaction.currency);
                }
                catch
                {
                    accountTransaction.Currency = CurrencyType.USD;
                }
            }

            
            accountTransaction.DateCreated = transaction.timestamp.ConvertFromVindicia();
            accountTransaction.Note = transaction.note;
            
            accountTransaction.PaymentMethodID = transaction.sourcePaymentMethod.merchantPaymentMethodId;
            accountTransaction.PaymentMethodType = transaction.sourcePaymentMethod.type.ToCurse();
            accountTransaction.Type = AccountTransactionType.Automatic;

            if(transaction.statusLog.Length > 0)
            {
                accountTransaction.Status = transaction.statusLog[0].status.ToCurse();                
            }
            
            accountTransaction.SubTotal = transaction.amount;
            accountTransaction.Tax = 0;
            accountTransaction.Total = transaction.amount;
            accountTransaction.TransactionID = transaction.merchantTransactionId;            
            accountTransaction.UserID = 0;

            List<AccountTransactionItem> accountTransactionItems = new List<AccountTransactionItem>();
            // Items
            foreach (TransactionItem transactionItem in transaction.transactionItems)
            {
                accountTransactionItems.Add(AccountTransactionItem.FromVindicia(transactionItem));
            }
            accountTransaction.Items = accountTransactionItems.ToArray();

            return accountTransaction;
        }

        private void Expire()
        {
            HttpRuntime.Cache.Expire(CacheKey.FormatWith(this.ID));
        }       

        public static AccountTransaction GetByVid(string vid)
        {
            Transaction txn = Retrieval.GetTransactionByVid(vid);
            if (txn == null)
            {
                return null;
            }
            else
            {
                return GetByTransactionID(txn.merchantTransactionId, false);
            }
        }

        public static AccountTransaction GetByID(int id, bool bypassCache)
        {
            return HttpRuntime.Cache.Get(bypassCache, CacheKey.FormatWith(id), () =>
            {
                return GetFromDatabaseByID(id);
            });
        }

        public static AccountTransaction GetByTransactionID(string transactionID, bool bypassCache)
        {
            int? id = GetIDFromDatabaseByTransactionID(transactionID);
            if (!id.HasValue)
            {
                return null;
            }
            else
            {
                return GetByID(id.Value, bypassCache);
            }
        }

        private static AccountTransaction GetFromDatabaseByID(int id)
        {
            AccountTransaction accountTransaction = null;            
            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = "select * from AccountTransaction where ID = @ID";
                SqlParameter idParameter = cmd.Parameters.Add("@ID", SqlDbType.Int);
                idParameter.Value = id;

                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                    if (reader.Read())
                    {
                        accountTransaction = new AccountTransaction(reader);
                    }
                }
            }

            return accountTransaction;
        }

        public static AccountTransaction GetByPayPalToken(string token)
        {
            int? id = GetIDFromDatabaseByPayPalToken(token);
            if (!id.HasValue)
            {
                return null;
            }
            else
            {
                return GetByID(id.Value, false);
            }
        }

        private static int? GetIDFromDatabaseByPayPalToken(string token)
        {            
            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = "select * from AccountTransaction where PayPalToken = @PayPalToken";
                SqlParameter idParameter = cmd.Parameters.Add("@PayPalToken", SqlDbType.VarChar);
                idParameter.Size = 64;
                idParameter.Value = token;

                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                    if (reader.Read())
                    {
                        return reader.GetInt32(0);
                    }
                }
            }

            return null;
        }

        public static List<AccountTransaction> GetAllByUserID(int userID)
        {
            List<AccountTransaction> accountTransactions = new List<AccountTransaction>();
            List<int> ids = GetAllIDsFromDatabaseByUserID(userID);
            foreach (int id in ids)
            {
                AccountTransaction accountTransaction = GetByID(id, false);
                if (accountTransaction != null)
                {
                    accountTransactions.Add(accountTransaction);
                }
            }

            return accountTransactions;
        }

        public static List<AccountTransaction> GetAllByUserIDAndType(int userID, AccountTransactionType type)
        {
            List<AccountTransaction> accountTransactions = new List<AccountTransaction>();
            List<int> ids = GetAllIDsFromDatabaseByUserIDAndType(userID, type);
            foreach (int id in ids)
            {
                AccountTransaction accountTransaction = GetByID(id, false);
                if (accountTransaction != null)
                {
                    accountTransactions.Add(accountTransaction);
                }
            }

            return accountTransactions;
        }

        public static List<AccountTransaction> GetAllByAccountIDs(IEnumerable<string> accountIDs)
        {
            List<AccountTransaction> accountTransactions = new List<AccountTransaction>();
            List<int> ids = GetAllIDsFromDatabaseByAccountIDs(accountIDs);
            foreach (int id in ids)
            {
                AccountTransaction accountTransaction = GetByID(id, false);
                if (accountTransaction != null)
                {
                    accountTransactions.Add(accountTransaction);
                }
            }

            return accountTransactions;
        }   

        private static List<int> GetAllIDsFromDatabaseByUserID(int userID)
        {
            List<int> ids = new List<int>();
            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = "select ID from AccountTransaction where UserID = @UserID";
                SqlParameter accountIDParameter = cmd.Parameters.Add("@UserID", System.Data.SqlDbType.Int);
                accountIDParameter.Value = userID;

                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        ids.Add(reader.GetInt32(0));
                    }
                }
            }

            return ids;
        }

        private static List<int> GetAllIDsFromDatabaseByUserIDAndType(int userID, AccountTransactionType type)
        {
            List<int> ids = new List<int>();
            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = "select ID from AccountTransaction where UserID = @UserID and Type = @Type" ;
                
                SqlParameter accountIDParameter = cmd.Parameters.Add("@UserID", System.Data.SqlDbType.Int);
                accountIDParameter.Value = userID;

                SqlParameter typeParameter = cmd.Parameters.Add("@Type", System.Data.SqlDbType.TinyInt);
                typeParameter.Value = (byte)type;

                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        ids.Add(reader.GetInt32(0));
                    }
                }
            }

            return ids;
        }

        private static int? GetIDFromDatabaseByTransactionID(string transactionID)
        {
            
            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = "select ID from AccountTransaction where TransactionID = @TransactionID";
                SqlParameter transactionIDParameter = cmd.Parameters.Add("@TransactionID", System.Data.SqlDbType.VarChar);
                transactionIDParameter.Size = 64;
                transactionIDParameter.Value = transactionID;

                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                    if (reader.Read())
                    {
                        return reader.GetInt32(0);
                    }
                }
            }

            return null;
        }

        private static List<int> GetAllIDsFromDatabaseByAccountIDs(IEnumerable<string> accountIDs)
        {
            List<int> ids = new List<int>();
            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = "select distinct(ID) from AccountTransaction where AccountID in('" + string.Join("','", accountIDs.ToArray()) + "')";
                
                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        ids.Add(reader.GetInt32(0));
                    }
                }
            }

            return ids;
        }

        public void IssueRefund(int userID, decimal refundAmount, string note)
        {

            if (this.Total == 0)
            {
                throw new Exception("A transaction with no total cannot be refunded.");
            }

            if (refundAmount > this.Total)
            {
                throw new ArgumentException("The refund amount specified is greater than the transaction total", "refundAmount");
            }

            Transaction txn = new Transaction() { merchantTransactionId = this.TransactionID };
            Refund refund = new Refund()
            {
                transaction = txn,
                timestamp = DateTime.UtcNow,
                amount = refundAmount,
                currency = this.Currency.ToString(),
                note = note,
                tokenAction = RefundTokenAction.CancelZeroBalance,
                timestampSpecified = true,
                tokenActionSpecified = true,
                merchantRefundId = "CURSE-REF-" + this.ID 
            };
            Refund[] refunds = new Refund[] { refund };
            Refund refundFactory = new Refund();
            Return ret = refundFactory.perform(ref refunds);

            if (ret.returnCode == ReturnCode.Item200)
            {
                this.Status = AccountTransactionStatus.Refunded;
                this.DateCreated = DateTime.UtcNow;
                this.StatusUserID = userID;
                this.Note = note;
                this.Save();
            }
            else             
            {                
                throw new Exception(ret.returnCode + ": " + ret.returnString);
            }
            
        }
    }
}
