﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Runtime.Serialization;
using System.Data.SqlClient;
using Curse.Extensions;
using Curse.Billing.Configuration;
using Curse.Billing.Extensions;
using System.Data;
using System.Net;
//TODO: can all this voice stuff go bye bye?
namespace Curse.Billing.Models
{
    [DataContract]
    public class AccountVoiceClient
    {

        private const string CacheKey = "AccountVoiceClient-{0}";

        public AccountVoiceClient()
        {
        }

        protected AccountVoiceClient(SqlDataReader reader)
        {
            this.ID = reader.GetInt32(0);
            this.AccountID = reader.GetString(1);
            this.ClientID = reader.GetInt64(2);
            this.DateCreated = reader.GetDateTime(3);
            this.DateModified = reader.GetDateTime(3);
        }

        [DataMember]
        public int ID
        {
            get;
            set;
        }

        [DataMember]
        public string AccountID
        {
            get;
            set;
        }

        [DataMember]
        public Int64 ClientID
        {
            get;
            set;
        }

        [DataMember]
        public DateTime DateCreated
        {
            get;
            set;
        }

        [DataMember]
        public DateTime DateModified
        {
            get;
            set;
        }

        [DataMember]
        public List<AccountVoiceService> Services
        {
            get
            {
                return AccountVoiceService.GetAllByClientID(this.ID);
            }
            private set
            {
            }
        }

        public void Save()
        {
            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandType = System.Data.CommandType.StoredProcedure;
                cmd.CommandText = "spUpdateAccountVoiceClient";

                SqlParameter idParameter = null;
                idParameter = cmd.Parameters.Add("@ID", SqlDbType.Int);
                idParameter.Direction = ParameterDirection.InputOutput;
                if (this.ID > 0)
                {
                    idParameter.Value = this.ID;
                }              

                SqlParameter param = null;

                param = cmd.Parameters.Add("@ClientID", SqlDbType.BigInt);
                param.Value = this.ClientID;

                param = cmd.Parameters.Add("@AccountID", SqlDbType.VarChar);
                param.Size = 64;
                param.Value = this.AccountID;
                         
                cmd.ExecuteNonQuery();

                if (this.ID == 0)
                {
                    this.ID = (int)idParameter.Value;
                    this.DateCreated = DateTime.UtcNow;
                }

                this.DateModified = DateTime.UtcNow;
                               
            }
            this.Expire();      
        }

        public void Expire()
        {
            HttpRuntime.Cache.Expire(CacheKey.FormatWith(this.ID));
        }

        public void AddService(Int64 serviceID, IPAddress ipAddress, int port)
        {
            if (this.ID == 0)
            {
                throw new InvalidOperationException("AccountVoiceClient must be saved before adding a service!");
            }

            AccountVoiceService service = new AccountVoiceService()
            {
                AccountVoiceClientID = this.ID,
                ServiceID = serviceID,                
                IPAddress = ipAddress,
                Port = port,
                DateLastMoved = DateTime.UtcNow.AddDays(-1)
            };
            service.Save();            
        }

        public static AccountVoiceClient GetByID(int id, bool bypassCache)
        {
            return HttpRuntime.Cache.Get(bypassCache, CacheKey.FormatWith(id), () =>
            {
                return GetFromDatabaseByID(id);
            });
        }

        private static AccountVoiceClient GetFromDatabaseByID(int id)
        {
            AccountVoiceClient accountTransaction = null;
            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = "select * from AccountVoiceClient where ID = @ID";
                SqlParameter idParameter = cmd.Parameters.Add("@ID", SqlDbType.Int);
                idParameter.Value = id;

                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                    if (reader.Read())
                    {
                        accountTransaction = new AccountVoiceClient(reader);
                    }
                }
            }

            return accountTransaction;
        }

        public static AccountVoiceClient GetByAccountID(string accountID, bool bypassCache)
        {
            int? id = GetIDFromDatabaseByAccountID(accountID);
            if (!id.HasValue)
            {
                return null;
            }
            else
            {
                return GetByID(id.Value, bypassCache);
            }
        }

        private static int? GetIDFromDatabaseByAccountID(string transactionID)
        {

            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = "select ID from AccountVoiceClient where AccountID = @AccountID";
                SqlParameter accountIDParameter = cmd.Parameters.Add("@AccountID", System.Data.SqlDbType.VarChar);
                accountIDParameter.Size = 64;
                accountIDParameter.Value = transactionID;

                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                    if (reader.Read())
                    {
                        return reader.GetInt32(0);
                    }
                }
            }

            return null;
        }

        public static PagedRecords<AccountVoiceClient> GetAllByPage(int pageNumber, int pageSize)
        {
            if (pageNumber < 1)
            {
                throw new ArgumentException("pageNumber must be 1 or greater", "pageNumber");
            }


            int pageIndex = pageNumber - 1;
            int startingRow = (pageSize * pageIndex) + 1;
            int endingRow = (startingRow + pageSize) - 1;

            PagedRecords<AccountVoiceClient> result = new PagedRecords<AccountVoiceClient>();
            result.Records = new List<AccountVoiceClient>();

            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {

                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = "select count(0) from AccountVoiceClient";
                result.RecordCount = (int)cmd.ExecuteScalar();

                cmd = conn.CreateCommand();
                cmd.CommandText = "select [ID] from(select [ID], ROW_NUMBER() OVER(ORDER BY [ID]) as [RowNum] from AccountVoiceClient) as List where [RowNum] between @StartingRow and @EndingRow";
                SqlParameter startingRowParameter = cmd.Parameters.Add("@StartingRow", SqlDbType.Int);
                startingRowParameter.Value = startingRow;
                SqlParameter endingRowParameter = cmd.Parameters.Add("@EndingRow", SqlDbType.Int);
                endingRowParameter.Value = endingRow;

                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        int id = reader.GetInt32(0);
                        AccountVoiceClient accountVoiceService = GetByID(id, false);
                        if (accountVoiceService == null)
                        {
                            throw new Exception("Unable to get account voice service: " + id.ToString());
                        }
                        result.Records.Add(accountVoiceService);
                    }
                }
            }

            return result;
        }
    }
}
