﻿using System;
using System.Linq;
using System.Runtime.Serialization;
using System.Collections.Generic;
using System.Web;
using Curse.Extensions;
using Curse.Billing.Extensions;
using Curse.Billing.Configuration;
using System.Data.SqlClient;
using System.Web.Caching;

namespace Curse.Billing.Models
{
    public class CountryBillingOption
    {
        private const string CacheKey = "CountryByCode-{0}";
        
        [DataMember]
        public int CountryID { get; set; }

        [DataMember]
        public string CountryName { get; set; }

        [DataMember]
        public string CountryCode { get; set; }

        [DataMember]
        public List<AccountPaymentType> PaymentTypes { get; set; }

        [DataMember]
        public List<Currency> Currencies { get; set; }

        public CountryBillingOption() { }
        public CountryBillingOption(SqlDataReader reader)
        {
            // Init from a reader
            this.CountryID = reader.GetInt32(reader.GetOrdinal("ID"));
            this.CountryName = reader.GetString(reader.GetOrdinal("Name"));
            this.CountryCode = reader.GetString(reader.GetOrdinal("Code"));
        }

        public void PopulatePaymentAndCurrency(SqlConnection conn)
        {
            var cmd = new SqlCommand("SELECT PaymentTypeID FROM CountryPaymentType WHERE CountryID = @CountryID", conn);
            cmd.Parameters.Add("CountryID", System.Data.SqlDbType.Int).Value = this.CountryID;

            PaymentTypes = new List<AccountPaymentType>();
            using (var reader = cmd.ExecuteReader())
            {
                while (reader.Read())
                {
                    PaymentTypes.Add((AccountPaymentType)reader.GetInt32(reader.GetOrdinal("PaymentTypeID")));
                }
            }

            Currencies = new List<Currency>();
            cmd.CommandText = "SELECT * FROM Currency WHERE ID IN (SELECT CurrencyID FROM CountryCurrency WHERE CountryID = @CountryID)";
            using (var reader = cmd.ExecuteReader())
            {
                while (reader.Read())
                {
                    Currencies.Add(new Currency(reader));
                }
            }
        }

        #region Caching
        public void Save()
        {
            HttpRuntime.Cache.Insert(CacheKey.FormatWith(this.CountryName), this, null, DateTime.Now.AddHours(1), Cache.NoSlidingExpiration);
        }
        public void Expire()
        {
            HttpRuntime.Cache.Expire(CacheKey.FormatWith(this.CountryName));
        }
        public static CountryBillingOption GetByName(string countryCode)
        {
            return HttpRuntime.Cache.Get(CacheKey.FormatWith(countryCode), () =>
            {
                return GetFromDatabaseByCountryCode(countryCode);
            }); 
        }
        public static CountryBillingOption[] GetAllCountrys()
        {
            return HttpRuntime.Cache.Get("AllCountries", () =>
            {
                return GetAllCountriesFromTheDatabase();
            });
        }
        private static CountryBillingOption GetFromDatabaseByCountryCode(string countryCode)
        {
            using (var conn = DatabaseConfiguration.GetBillingConnection())
            {
                var cmd = new SqlCommand("SELECT * FROM Country WHERE Code = @CountryCode", conn);
                cmd.Parameters.Add("CountryCode", System.Data.SqlDbType.VarChar).Value = countryCode;

                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                    if (reader.Read())
                    {
                        var country = new CountryBillingOption(reader);
                        country.PopulatePaymentAndCurrency(conn);
                        country.Save();
                        return country;
                    }
                    else return null;
                }
            }
        }
        private static CountryBillingOption[] GetAllCountriesFromTheDatabase()
        {
            var countryList = new List<CountryBillingOption>();
            using (var conn = DatabaseConfiguration.GetBillingConnection())
            {
                var cmd = new SqlCommand("SELECT * FROM Country", conn);

                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        var country = new CountryBillingOption(reader);
                        country.PopulatePaymentAndCurrency(conn);
                        country.Save();
                        countryList.Add(country);
                    }
                }
            }

            HttpRuntime.Cache.Insert("AllCountries", countryList, null, DateTime.Now.AddHours(1), Cache.NoSlidingExpiration);
            return countryList.ToArray();
        }
        #endregion

        #region Validation Methods
        public bool ValidatePaymentType(AccountPaymentType paymentType)
        {
            return this.PaymentTypes.Contains(paymentType);
        }
        public bool ValidateCurrency(CurrencyType currency)
        {
            return (Currencies.FirstOrDefault(p => p.CurrencyID == (int)currency) != null);
        }
        #endregion
    }
}
