﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Runtime.Serialization;
using System.Data;
using System.Data.SqlClient;
using Curse.Logging;
using Vindicia;

using Curse.Extensions;
using Curse.Billing.Configuration;
using Curse.Billing.Extensions;
using Curse.Billing.VindiciaExtensions;
using Curse.Billing.Data;

namespace Curse.Billing.Models
{
    [DataContract]
    public class Coupon
    {

        #region Cache Keys

        private const string CouponCodesByPurchasedAccountCacheKey = "CouponCodesByPurchasedAccountID-{0}";
        private const string CouponCodesByClaimedAccountCacheKey = "CouponCodesByClaimedAccountID-{0}";

        #endregion


        #region Properties

        [DataMember]
        public int ID
        {
            get;
            set;
        }

        [DataMember]
        public string Code
        {
            get;
            set;
        }

        [DataMember]
        public int CouponTypeID
        {
            get;
            set;
        }

        [DataMember]
        public DateTime DateCreated
        {
            get;
            set;
        }

        [DataMember]
        public string ReferenceBillingPlanID
        {
            get;
            set;
        }

        [DataMember]
        public string PurchasedAccountID
        {
            get;
            set;
        }

        [DataMember]
        public string ClaimedAccountID
        {
            get;
            set;
        }

        [DataMember]
        public DateTime? ClaimedDate
        {
            get;
            set;
        }
        
        [DataMember]
        public bool IsClaimed
        {
            get
            {
                return ClaimedDate.HasValue;
            }
            set { }
        }

        #endregion        

        public Coupon() { }

        public Coupon(SqlDataReader reader)
        {
            ID = reader.GetInt32(0);                        
            ReferenceBillingPlanID = reader.GetString(1);
            DateCreated = reader.GetDateTime(2);
            Code = reader.GetString(3);
            CouponTypeID = reader.GetInt32(4);
            
            if (reader["PurchasedAccountID"] != System.DBNull.Value)
            {
                PurchasedAccountID = reader.GetString(5);
            }

            // These columns will be going away from the main table - REE 
            //if (reader["ClaimedAccountID"] != System.DBNull.Value)
            //{
            //    ClaimedAccountID = reader.GetString(6);
            //}            

            //if (reader["ClaimedDate"] != System.DBNull.Value)
            //{
            //    ClaimedDate = reader.GetDateTime(7);
            //}
        }//construct from DataReader

        #region Cache Access

        public static List<Coupon> GetAllByPurchasedAccountID(string accountID)
        {
            IEnumerable<string> codes = HttpRuntime.Cache.Get(CouponCodesByPurchasedAccountCacheKey.FormatWith(accountID), () =>
            {
                return GetAllFromDatabaseByPurchasedAccountID(accountID);
                

            });

            return GetAllByCodes(codes);
        }

        public static List<Coupon> GetAllByClaimedAccountID(string accountID)
        {
            IEnumerable<string> codes = HttpRuntime.Cache.Get(CouponCodesByClaimedAccountCacheKey.FormatWith(accountID), () =>
            {
                return GetAllFromDatabaseByClaimedAccountID(accountID);
            });

            return GetAllByCodes(codes);
        }

        public static List<Coupon> GetAllByCodes(IEnumerable<string> codes)
        {
            List<Coupon> coupons = new List<Coupon>();
            foreach (string code in codes)
            {
                Coupon coupon = GetByCode(code);
                if (coupon != null)
                {
                    coupons.Add(coupon);
                }
            }

            return coupons;
        }     

        public static Coupon GetByCode(string code, bool bypassCache)
        {
            return HttpRuntime.Cache.Get(bypassCache, "CouponByCode-{0}".FormatWith(code), () =>
            {
                return GetFromDatabaseByCode(code);
            });                       
        }

        public static Coupon GetByCode(string code)
        {
            return GetByCode(code, false);
        }

        #endregion

        #region Database Access

        private static List<string> GetAllFromDatabaseByPurchasedAccountID(string accountID)
        {

            List<string> codes = new List<string>();

            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = "select Code from Coupon where PurchasedAccountID = @AccountID";
                SqlParameter accountIDParameter = cmd.Parameters.Add("@AccountID", SqlDbType.VarChar);
                accountIDParameter.Size = 64;
                accountIDParameter.Value = accountID;

                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        codes.Add(reader.GetString(0));
                    }
                }
            }

            return codes;
        }

        private static List<string> GetAllFromDatabaseByClaimedAccountID(string accountID)
        {

            List<string> codes = new List<string>();

            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = "select Code from Coupon where ID IN (Select CouponID From CouponUsage Where ClaimedAccountID = @AccountID)";
                SqlParameter accountIDParameter = cmd.Parameters.Add("@AccountID", SqlDbType.VarChar);
                accountIDParameter.Size = 64;
                accountIDParameter.Value = accountID;

                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        codes.Add(reader.GetString(0));
                    }
                }
            }
            return codes;
        }

        private static Coupon GetFromDatabaseByCode(string code)
        {
            Coupon coupon = null;
            List<AccountEntitlement> entitlements = new List<AccountEntitlement>();
            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = "select * from Coupon where Code = @Code";
                SqlParameter accountIDParameter = cmd.Parameters.Add("@Code", SqlDbType.VarChar);
                accountIDParameter.Size = 64;
                accountIDParameter.Value = code;

                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                    if (reader.Read())
                    {
                        coupon = new Coupon(reader);
                    }
                }
            }            

            return coupon;
        }

        #endregion

        public static Coupon Create(string referenceBillingPlanID)
        {
            return Create(null, referenceBillingPlanID);
        }

        public static Coupon Create(AccountInformation accountInformation, string referenceBillingPlanID)
        {
            string couponCode = null;
           
            // 1. Execute the stored procedure to create the coupon
            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = "spCreateCoupon";
                cmd.CommandType = CommandType.StoredProcedure;
                SqlParameter param = null;

                if (accountInformation != null)
                {
                    param = cmd.Parameters.Add("@PurchasedAccountID", SqlDbType.VarChar);
                    param.Value = accountInformation.ID;
                    param.Size = 64;
                }

                param = cmd.Parameters.Add("@ReferenceBillingPlanID", SqlDbType.VarChar);
                param.Value = referenceBillingPlanID;
                param.Size = 64;

                SqlParameter codeParameter = cmd.Parameters.Add("@Code", SqlDbType.VarChar);
                codeParameter.Direction = ParameterDirection.Output;
                codeParameter.Size = 19;

                cmd.ExecuteNonQuery();

                couponCode = (string)codeParameter.Value;


                // Expire the coupon cache
                if (accountInformation != null)
                {
                    HttpRuntime.Cache.Expire(CouponCodesByPurchasedAccountCacheKey, accountInformation.ID);
                }
               
            }

            // 2. Return the new coupon
            return Coupon.GetByCode(couponCode, true);
            
        }

        public ServiceResponseStatus Validate(string accountID)
        {
            try
            {
                using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
                {
                    SqlCommand cmd = conn.CreateCommand();
                    cmd.CommandText = "spValidateCoupon";
                    cmd.CommandType = CommandType.StoredProcedure;

                    cmd.Parameters.Add("@Code", SqlDbType.VarChar, 64).Value = this.Code;
                    cmd.Parameters.Add("@CouponTypeID", SqlDbType.Int).Value = this.CouponTypeID;
                    cmd.Parameters.Add("@AccountID", SqlDbType.VarChar, 64).Value = accountID;
                    cmd.Parameters.Add("@ClaimedDate", SqlDbType.DateTime); cmd.Parameters["@ClaimedDate"].Direction = ParameterDirection.Output;

                    // If the code is not valid it will throw an error from the database
                    cmd.ExecuteNonQuery();

                    //Set the claimed date and account information since we're no longer setting that from the constructor
                    if (cmd.Parameters["@ClaimedDate"].Value != DBNull.Value)
                    {
                        this.ClaimedDate = (DateTime)cmd.Parameters["@ClaimedDate"].Value;
                        this.ClaimedAccountID = accountID;
                    }//if Claimed Date is not null

                    // Still return a value based on whether it's claimed
                    return (this.IsClaimed) ? ServiceResponseStatus.ClaimCoupon_CodeAlreadyClaimed : ServiceResponseStatus.Successful;
                }//end using conn
            }//try
            catch (SqlException exc) {
                if (exc.Message.ToLower().Contains("expired"))
                    return ServiceResponseStatus.ClaimCoupon_CodeExpired;
                else if (exc.Message.Contains(this.Code))
                    return ServiceResponseStatus.ClaimCoupon_CodeAlreadyClaimed;
                else throw;
            }//catch
        }//Validate

        public ServiceResponseStatus Claim(int userID, int cobaltInstanceID, int entityTypeID, int entityID, AccountInformation account)
        {

            // Coupon Code Already Claimed
            //if (this.IsClaimed)
            //{
            //    return ServiceResponseStatus.ClaimCoupon_CodeAlreadyClaimed;
            //}

            ServiceResponseStatus validateStatus = Validate(account.ID);
            // Coupon Code Already Claimed
            if (validateStatus != ServiceResponseStatus.Successful)
            {
                return validateStatus;
            }

            // Get the billing plan that this coupon references
            ProductBillingPlan plan = ProductBillingPlan.GetByID(this.ReferenceBillingPlanID);            

            // If the plan could not be found, log the issue.
            if (plan == null || plan.GrantedBillingPlanID.IsNullOrEmpty() || plan.GrantedProductID.IsNullOrEmpty())
            {
                Logger.Error("The reference billing plan ID: '{0}', could not be retrieved.", this.ReferenceBillingPlanID);
                return ServiceResponseStatus.KnownException;
            }

            int planDurationDays = plan.FirstPayPeriod.Days;
                     
            
            // See if this account already has a subscription which is related to this product
            AutoBill[] existingSubscriptions = Retrieval.GetAllAutoBillsByAccountID(account.ID);
            AutoBill mostRecentSubscription = existingSubscriptions.Where(p => p.product.merchantProductId == plan.GrantedProductID && p.endTimestamp > DateTime.UtcNow).OrderByDescending(p => p.startTimestamp).FirstOrDefault();

            ServiceResponseStatus status = ServiceResponseStatus.Successful;
            bool successful = false;

            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                SqlTransaction txn = conn.BeginTransaction();

                SqlCommand cmd = conn.CreateCommand();
                cmd.Transaction = txn;
                cmd.CommandText = "spClaimCoupon";
                cmd.CommandType = CommandType.StoredProcedure;

                SqlParameter param = null;                                

                param = cmd.Parameters.Add("@AccountID", SqlDbType.VarChar);
                param.Size = 64;
                param.Value = account.ID;

                param = cmd.Parameters.Add("@Code", SqlDbType.VarChar);
                param.Size = 64;
                param.Value = this.Code;
                cmd.ExecuteNonQuery();

                
                // If an active subscription is found, simply delay billing
                if (mostRecentSubscription != null)
                {
                    successful = account.CreditSubscription(userID, mostRecentSubscription.merchantAutoBillId, planDurationDays, AccountTransactionType.ClaimCoupon, null);
                }
                // Otherwise, create a new subscription
                else
                {
                    CreateSubscriptionResult result = account.CreateSubscription(userID, cobaltInstanceID, entityTypeID, entityID, null, plan.GrantedProductID, plan.GrantedBillingPlanID, null, null, AccountTransactionType.ClaimCoupon, CurrencyType.USD);
                    Subscription newSubscription = result.Subscription;
                    successful = result.IsSuccessful;
                }

                if (successful)
                {
                    // Commit the transaction
                    txn.Commit();

                    // Refresh the coupon from the database
                    Coupon.GetByCode(this.Code, true);

                    // Expire the coupon code cache
                    HttpRuntime.Cache.Expire(CouponCodesByClaimedAccountCacheKey, account.ID);

                    // Expire the account cache
                    account.Expire();

                    // Make a note on the account
                    account.ToVindiciaAccount().SaveActivityNote("Claimed coupon {0}".FormatWith(this.Code));                                    
                }
                else
                {
                    // Update the status
                    status = ServiceResponseStatus.ClaimCoupon_UnableToCreditSubscription;

                    // Rollback the transaction
                    txn.Rollback();
                }                               
            }            

            return status;
        }
    }//Coupon Class
}//Curse.Billing.Models
