﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Runtime.Serialization;
using Curse.Billing.Data;
using Curse.Billing.Extensions;
using Curse.Extensions;
using Vindicia;

namespace Curse.Billing.Models
{

    [DataContract]
    public class ProductBillingPlan
    {
        [DataMember]
        public string ID
        {
            get;
            set;
        }

        [DataMember]
        public string Description
        {
            get;
            set;
        }

        [DataMember]
        public ProductBillingPeriod FirstPayPeriod
        {
            get;
            set;
        }

        public string GrantedProductID
        {
            get;
            set;
        }

        public string GrantedBillingPlanID
        {
            get;
            set;
        }

        public string DelayedPlanID
        {
            get;
            set;
        }

        public IEnumerable<ProductBillingPeriod> Periods
        {
            get;
            set;
        }

        [DataMember]
        public Dictionary<string, string> MerchantData
        {
            get;
            set;
        }

        [DataMember]
        public List<string> EntitlementIDs
        {
            get;
            set;
        }       

        public bool HasCurrency(CurrencyType currency)
        {
            return this.FirstPayPeriod.Try(pbp => pbp.Prices.Exists(p => p.Currency == currency));
        }

        public ProductPrice GetPrice(CurrencyType currency)
        {
            ProductPrice price = this.FirstPayPeriod.Prices.FirstOrDefault(p => p.Currency == currency);
            if (price == null)
            {
                if (currency == CurrencyType.USD)
                {
                    throw new Exception("No price is set in USD for billing plan: " + ID.ToString());
                }
                else
                {
                    return GetPrice(CurrencyType.USD);
                }
            }
            else
            {
                return price;
            }
        }

        public ProductBillingPlan() { }

        public ProductBillingPlan GetDelayedPlan()
        {
            return ProductBillingPlan.GetByID(this.DelayedPlanID);
        }
      
        public static List<ProductBillingPlan> GetAllByProductID(string productID)
        {

            string[] ids = HttpRuntime.Cache.Get("BillingPlanIDsByProductID-{0}".FormatWith(productID), () =>
            {
                List<string> planIDs = new List<string>();
                Product product = new Product() { merchantProductId = productID };                
                BillingPlan[] plans = product.GetBillingPlans();
                foreach (BillingPlan plan in plans)
                {
                    planIDs.Add(plan.merchantBillingPlanId);
                }
                return planIDs.ToArray();
                
            });
            
            List<ProductBillingPlan> list = new List<ProductBillingPlan>();

            foreach (string id in ids)
            {
                ProductBillingPlan plan = GetByID(id);
                if (plan != null)
                {
                    list.Add(plan);
                }
            }

            return list;
            
        }
        
        public static ProductBillingPlan GetByID(string id)
        {
            return HttpRuntime.Cache.Get("BillingPlanByID-{0}".FormatWith(id), () =>
            {
                BillingPlan plan = new BillingPlan();
                Return r = plan.fetchByMerchantBillingPlanId(id);
                if (r.returnCode == ReturnCode.Item200)
                {
                    return plan.ToCurse();
                }
                else
                {
                    return null;
                }
            });
            
        }

    }
}
