﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Runtime.Serialization;
using Curse.Extensions;
using Curse.Billing.Extensions;

namespace Curse.Billing.Models
{
    [DataContract]
    public class ProductFulfillment
    {

        public delegate IEnumerable<ProductFulfillment> ProductFulfilmentDelegate(int userID, int cobaltInstanceID, int entityTypeID, int entityID, AccountInformation account, ShoppingCartItem item);
        private static Dictionary<ProductFulfillmentType, ProductFulfilmentDelegate> _fulfillmentDelegates = new Dictionary<ProductFulfillmentType, ProductFulfilmentDelegate>();
        
        [DataMember]
        public ProductFulfillmentType Type
        {
            get;
            set;
        }

        [DataMember]
        public string Data
        {
            get;
            set;
        }        

        static ProductFulfillment()
        {
            _fulfillmentDelegates.Add(ProductFulfillmentType.Coupon, FulfillCoupon);
            _fulfillmentDelegates.Add(ProductFulfillmentType.TokenGrant, FulfillTokenGrant);
        }

        public static IEnumerable<ProductFulfillment> Fulfill(int userID, int cobaltInstanceID, int entityTypeID, int entityID, ProductFulfillmentType type, AccountInformation account, ShoppingCartItem item)
        {          
            if (type == ProductFulfillmentType.None || !_fulfillmentDelegates.ContainsKey(type))
            {
                return null;
            }
            else
            {
                return _fulfillmentDelegates[type](userID, cobaltInstanceID, entityTypeID, entityID, account, item);
            }
        }

        private static IEnumerable<ProductFulfillment> FulfillCoupon(int userID, int cobaltInstanceID, int entityTypeID, int entityID, AccountInformation account, ShoppingCartItem item)
        {
            List<ProductFulfillment> fulfillments = new List<ProductFulfillment>();
            for (int i = 0; i < item.Quantity; i++)
            {
                Coupon coupon = Coupon.Create(account, item.BillingPlanID);

                fulfillments.Add(new ProductFulfillment()
                {
                    Data = coupon.Code,
                    Type = ProductFulfillmentType.Coupon
                });
            }
            return fulfillments;
        }
        private static IEnumerable<ProductFulfillment> FulfillTokenGrant(int userID, int cobaltInstanceID, int entityTypeID, int entityID, AccountInformation account, ShoppingCartItem item)
        {
            List<ProductFulfillment> fulfillments = new List<ProductFulfillment>();

            if (item == null) {
                throw new ArgumentNullException("item");
            }
            ProductOffering product = ProductOffering.GetByID(item.ProductID);
            if (product == null) {
                throw new ArgumentException("Could not get Product with ID ({0})".FormatWith(item.ProductID));
            }
            if (product.TokensGranted == null || !product.TokensGranted.Any()) {
                throw new Exception("Product {0} does not have any token grants set".FormatWith(product.ID));
            }
            
            string data = "Tokens granted:";
            List<AccountTokenAmount> tokensGranted = new List<AccountTokenAmount>();
            foreach (AccountTokenAmount tokenAmount in product.TokensGranted) {
                int effectiveAmount = tokenAmount.Amount * item.Quantity;
                data += " ({0} {1})".FormatWith(tokenAmount.Amount * item.Quantity, tokenAmount.Type.ID);
                tokensGranted.Add(new AccountTokenAmount(tokenAmount.Type, effectiveAmount));
                
                // If on an existing token plan with no tokens, delay billing to prevent tokens from dissapearing
                if (account.TokenBalances.SingleOrDefault(ata => ata.Type.ID == tokenAmount.Type.ID).Try(tb => tb.Amount) == 0) {
                    // Look for the token plan
                    foreach (Subscription activeSub in account.Subscriptions.Where(s => s.DisplayStatus == DisplaySubscriptionStatus.Active)) {
                        ProductBillingPlan pbp = ProductBillingPlan.GetByID(activeSub.BillingPlanID);
                        ProductPrice firstPrice = pbp.FirstPayPeriod.Prices.First();
                        // This subscription uses the token purchased, reset its billing time until now
                        if (firstPrice.Currency == CurrencyType.Token && firstPrice.TokenType.ID == tokenAmount.Type.ID) {
                            activeSub.DelayBillingToDate(DateTime.UtcNow);
                            break;
                        }
                    }
                }
            }

            // Credit tokens to the account
            account.CreditTokens(null, tokensGranted);

            // Switch to token plan if no active subscriptions            
            ProductBillingPlan grantingBillingPlan = product.AvailableBillingPlans.SingleOrDefault(bp => bp.ID == product.DefaultBillingPlanID);
            ProductOffering grantedProduct = ProductOffering.GetByID(grantingBillingPlan.GrantedProductID);
            ProductBillingPlan grantedBillingPlan = ProductBillingPlan.GetByID(grantingBillingPlan.GrantedBillingPlanID);
            IEnumerable<string> accountEntitlements = account.Entitlements.Where(ae => ae.Active).Select(ae => ae.EntitlementID);
            
            if (grantedBillingPlan != null && grantedBillingPlan.EntitlementIDs.Except(accountEntitlements).Any()) {
                AccountTokenType tokenType = grantedBillingPlan.FirstPayPeriod.Prices.First().TokenType;
                AccountPaymentMethod tokenMethod = new AccountPaymentMethod() {
                    TokenType = tokenType,
                    Type = AccountPaymentType.Token
                };
                account.CreateOrUpdatePaymentMethod(tokenMethod);
                CreateSubscriptionResult result = account.CreateSubscription(userID, cobaltInstanceID, entityTypeID, entityID, null, grantingBillingPlan.GrantedProductID, grantingBillingPlan.GrantedBillingPlanID, tokenMethod, null, AccountTransactionType.PurchaseProduct, CurrencyType.USD);
                Subscription sub = result.Subscription;
            }
            fulfillments.Add(new ProductFulfillment() {
                Data = data,
                Type = ProductFulfillmentType.TokenGrant
            });
            return fulfillments;
        }
    }
}
