﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Runtime.Serialization;
using Curse.Logging;
using Vindicia;
using Curse.Billing.Extensions;
using Curse.Extensions;
using Curse.Billing.Data;
using System.Diagnostics;

namespace Curse.Billing.Models
{
    [DataContract]
    public class ProductOffering
    {

        #region Properties

        [DataMember]
        public string ID
        {
            get;
            set;
        }

        [DataMember]
        public string Description
        {
            get;
            set;
        }

        [DataMember]
        public string DefaultBillingPlanID
        {
            get;
            set;
        }

        [DataMember]
        public bool IsSubscription
        {
            get;
            set;
        }

        [DataMember]
        public ProductFulfillmentType FulfillmentType
        {
            get;
            set;
        }

        [DataMember]
        public List<ProductBillingPlan> AvailableBillingPlans
        {
            get
            {
                return ProductBillingPlan.GetAllByProductID(this.ID);
            }
            set { }
        }

        [DataMember]
        public List<AccountTokenAmount> TokensGranted
        {
            get;
            set;
        }

        [DataMember]
        public List<string> EntitlementIDs
        {
            get;
            set;                
        }

        [DataMember]
        public string ProductGroupID
        {
            get;
            private set;
        }

        #endregion

        public ProductOffering() { }

        public ProductOffering(Product vindiciaProduct)
        {
            this.ID = vindiciaProduct.merchantProductId;
            this.Description = vindiciaProduct.description;

            
            if (vindiciaProduct.defaultBillingPlan != null)
            {
                this.DefaultBillingPlanID = vindiciaProduct.defaultBillingPlan.merchantBillingPlanId;
            }

            // Product Group ID, if any
            if(vindiciaProduct.nameValues.Any(p => p.name == "ProductGroupID"))
            {
                this.ProductGroupID = vindiciaProduct.nameValues.FirstOrDefault(p => p.name == "ProductGroupID").value;
            }

            // Whether or not this is a subscription or a one-time purchase
            this.IsSubscription = vindiciaProduct.nameValues.Any(p => p.name == "IsSubscription" && p.value == "true");

            // Determine the fulfillment type, defaulting to 'None'
            FulfillmentType = ProductFulfillmentType.None;
            if (vindiciaProduct.nameValues.Any(p => p.name == "FulfillmentType"))
            {
                string fulfillmentType = vindiciaProduct.nameValues.FirstOrDefault(p => p.name == "FulfillmentType").value;
                if (Enum.IsDefined(typeof(ProductFulfillmentType), fulfillmentType))
                {
                    FulfillmentType = (ProductFulfillmentType)Enum.Parse(typeof(ProductFulfillmentType), fulfillmentType, true);
                }
            }

            // Fulfillmenttype is Token Grant, populate what type of tokens
            if (FulfillmentType == ProductFulfillmentType.TokenGrant)
            {
                NameValuePair tokenTypeData = vindiciaProduct.nameValues.SingleOrDefault(p => p.name == "TokenName");
                if (tokenTypeData == null) {
                    throw new Exception("{0} has a FulfillmentType of TokenGrant but does not define TokenName".FormatWith(this.ID));
                }
                NameValuePair tokenAmountData = vindiciaProduct.nameValues.SingleOrDefault(p => p.name == "TokenAmount");
                if (tokenTypeData == null) {
                    throw new Exception("{0} has a FulfillmentType of TokenGrant but does not define TokenAmount".FormatWith(this.ID));
                }
                int tokenAmount = int.Parse(tokenAmountData.value);
                AccountTokenType tokenType = AccountTokenType.FetchByID(tokenTypeData.value);
                List<AccountTokenAmount> tokenAmounts = new List<AccountTokenAmount>();
                tokenAmounts.Add(new AccountTokenAmount(tokenType, tokenAmount));
                this.TokensGranted = tokenAmounts;
            }

            if (vindiciaProduct.merchantEntitlementIds != null && vindiciaProduct.merchantEntitlementIds.Any())
            {
                EntitlementIDs = vindiciaProduct.merchantEntitlementIds.Select(me => me.id).ToList();
            }
            else
            {
                EntitlementIDs = new List<string>();
            }
        }

        public static ProductOffering GetByID(string id)
        {
            return ProductOffering.GetAll().FirstOrDefault(p => p.ID == id);
        }

        public static ProductOffering[] GetByProductGroupID(string groupID)
        {
            return ProductOffering.GetAll().Where(p => p.ProductGroupID == groupID).ToArray();
        }

        public static ProductOffering[] GetAll(bool activeOnly = true)
        {
            return HttpRuntime.Cache.Get("ProductOfferings-ActiveOnline:" + activeOnly, () =>
            {
                Logger.Debug("Getting all product offerings from Vindicia");
                
                var products = Retrieval.GetAllProducts();

                Logger.Debug("Vindicia returned " + products.Length + " products.");

                if (activeOnly)
                {
                    products = products.Where(p => p.status == ProductStatus.Active).ToArray();
                }

                Logger.Debug("Vindicia returned " + products.Length + " active products.");

                var productOfferings = new List<ProductOffering>();

                foreach (var p in products)
                {
                    productOfferings.Add(new ProductOffering(p));
                }

                return productOfferings.ToArray();
            });

        }
        
        public void TestIt()
        {
            Debug.WriteLine("TestIt Called!");
        }
    }
}
