﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Runtime.Serialization;
using Curse.Extensions;
using System.Web.Caching;
using System.Data.SqlClient;
using Curse.Billing.Configuration;
using System.Xml.Serialization;
using Curse.Logging;
using Newtonsoft.Json;

namespace Curse.Billing.Models
{
    [DataContract]
    public class ShoppingCart
    {
        private const string CacheKey = "ShoppingCartByID-{0}";

        [DataMember]
        public ShoppingCartItem[] Items
        {
            get;
            set;
        }

        [DataMember]
        public CurrencyType Currency
        {
            get;
            set;
        }

        [DataMember]
        public string UrlReferrer
        {
            get;
            set;
        }

        public int TransactionID
        {
            get;
            private set;
        }        

        public int UserID
        {
            get;
            private set;
        }

        public int CobaltInstanceID
        {
            get;
            private set;
        }

        public int  EntityTypeID
        {
            get;
            private set;
        }

        public int EntityID
        {
            get;
            private set;
        }

        public AccountPaymentMethod PaymentMethod
        {
            get;
            private set;
        }

        public decimal SubTotal
        {
            get;
            private set;
        }

        public decimal Total
        {
            get;
            private set;
        }

        public decimal Tax
        {
            get;
            private set;
        }
       
        
        public IEnumerable<ShoppingCartItem> Validate()
        {            

            foreach (ShoppingCartItem item in this.Items)
            {

                if (item.ProductID.IsNullOrEmpty() || item.BillingPlanID.IsNullOrEmpty() || item.Quantity < 1)
                {
                    yield return item;
                }
                
                // 1. Retrieve the product offering
                ProductOffering product = ProductOffering.GetByID(item.ProductID);

                if (product == null)
                {
                    yield return item;
                }

                // 2. Retrieve the billing plan offering
                ProductBillingPlan plan = ProductBillingPlan.GetByID(item.BillingPlanID);

                if (plan == null)
                {
                    yield return item;
                }
                
                // 3. Check for items without native-currency pricing
                if (!plan.HasCurrency(CurrencyType.Token) && !plan.HasCurrency(this.Currency))
                {
                    yield return item;
                }

                // 4. Check for duplicate subscripion items
                if (product.IsSubscription && item.Quantity > 1)
                {
                    yield return item;
                }

                // 5. Check for DelayBillingDays for a non-subscription item
                if (item.SubscriptionStartDate.HasValue && !product.IsSubscription)
                {
                    yield return item;
                }

                // 6. Check for duplicate subscriptions, of different billing plans
                if (product.IsSubscription && this.Items.Count(p => p.ProductID == item.ProductID) > 1)
                {
                    yield return item;
                }

            }
            
        }

        public void Save(int userID, int transactionID, int cobaltInstanceID, int entityTypeID, int entityID, AccountPaymentMethod paymentMethod, decimal subTotal, decimal total, decimal tax)
        {
            this.UserID = userID;
            this.TransactionID = transactionID;
            this.CobaltInstanceID = cobaltInstanceID;
            this.EntityTypeID = entityTypeID;
            this.EntityID = entityID;
            this.PaymentMethod = paymentMethod;
            this.SubTotal = subTotal;
            this.Tax = tax;
            this.Total = total;

            HttpRuntime.Cache.Insert(CacheKey.FormatWith(transactionID), this, null, DateTime.UtcNow.AddHours(1), Cache.NoSlidingExpiration);
            
        }

        public static ShoppingCart GetByID(int transactionID)
        {

            var shoppingCart = HttpRuntime.Cache.Get(CacheKey.FormatWith(transactionID));
            if (shoppingCart != null)
            {
                return shoppingCart as ShoppingCart;
            }
            
            return null;
        }

        public IEnumerable<ShoppingCartItem> SubscriptionItems
        {
            get
            {
                foreach (ShoppingCartItem item in this.Items)
                {
                    ProductOffering productOffering = ProductOffering.GetByID(item.ProductID);

                    if (productOffering == null)
                    {
                        throw new ArgumentException("Unknown product ID: '{0}'".FormatWith(item.ProductID));
                    }

                    // If the subscription start date is in the past null it out
                    if (item.SubscriptionStartDate.HasValue && item.SubscriptionStartDate < DateTime.UtcNow)
                    {
                        item.SubscriptionStartDate = null;
                    }

                    if (productOffering.IsSubscription)
                    {
                        yield return item;
                    }
                }
            }
        }
    }
}
