﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Runtime.Serialization;
using System.Data.SqlClient;
using Curse.Extensions;
using Curse.Billing.Configuration;
using Curse.Billing.Extensions;
using Curse.Billing.VindiciaExtensions;
using System.Data;
using Curse.Logging;
using Vindicia;
using Curse.Billing.Data;

namespace Curse.Billing.Models
{
    [DataContract]
    public class Subscription
    {
        private const string CacheKey = "SubscriptionByID-{0}";

        public Subscription() { }

        public Subscription(SqlDataReader reader)
        {
            this.ID = reader.GetInt32(0);
            this.SubscriptionID = reader.GetString(1);
            this.AccountID = reader.GetString(2);
            this.ProductID = reader.GetString(3);
            this.BillingPlanID = reader.GetString(4);
            if (!reader.IsDBNull(5))
            {
                this.PaymentMethodID = reader.GetString(5);
            }
            this.DateStarted = reader.GetDateTime(6);
            this.DateExpires = reader.GetDateTime(7);
            this.DateUpdated = reader.GetDateTime(8);
            
            if (reader[9] != System.DBNull.Value)
            {
                this.NextBillingDate = reader.GetDateTime(9);
            }

            if (reader[10] != System.DBNull.Value)
            {
                this.NextBillAmount = reader.GetDecimal(10);
            }

            if (reader[11] != System.DBNull.Value)
            {
                this.NextBillCurrency = reader.GetString(11);
            }

            this.Status = (SubscriptionStatus)reader.GetByte(12);

            if (reader[13] != DBNull.Value) {
                this.DateCancelled = reader.GetDateTime(13);
            }

            SetHistoryData();
        }

        internal void SetHistoryData()
        {
            // Set SubscriptionHistory data
            SubscriptionHistory history = SubscriptionHistory.GetBySubscriptionID(this.SubscriptionID);
            if (history != null)
            {
                this.UserID = history.UserID;
                this.CobaltInstanceID = history.CobaltInstanceID;
                this.EntitledAccountID = history.EntitledAccountID;
                this.EntitledEntityTypeID = history.EntitledEntityTypeID;
                this.EntitledEntityID = history.EntitledEntityID;
            }
        }
        public int ID
        {
            get;
            set;
        }

        [DataMember]
        public string SubscriptionID
        {
            get;
            set;
        }

        [DataMember]
        public string AccountID
        {
            get;
            set;
        }

        [DataMember]
        public int UserID { get; private set; }

        [DataMember]
        public int CobaltInstanceID { get; private set; }

        [DataMember]
        public string CobaltInstanceName
        {
            get
            {
                if (CobaltInstanceID > 0)
                {
                    return CobaltInstance.GetByID(CobaltInstanceID).Name;
                }
                else
                {
                    return null;
                }
            }
            private set { }
        }

        [DataMember]
        public string EntitledAccountID { get; private set; }

        [DataMember]
        public int EntitledEntityTypeID { get; private set; }

        [DataMember]
        public int EntitledEntityID { get; private set; }

        [DataMember]
        public string ProductDescription
        {
            get
            {
                if (!this.ProductID.IsNullOrEmpty())
                {
                    return ProductOffering.GetByID(this.ProductID).Try(p => p.Description);
                }
                else
                {
                    return null;
                }
            }
            private set
            {
            }
        }

        [DataMember]
        public string PlanDescription
        {
            get
            {
                if (!this.BillingPlanID.IsNullOrEmpty())
                {
                    return ProductBillingPlan.GetByID(this.BillingPlanID).Try(p => p.Description);
                }
                else
                {
                    return null;
                }
            }
            private set
            {
            }
        }
        
        [DataMember]
        public string ProductID
        {
            get;
            set;
        }
        
        [DataMember]
        public string BillingPlanID
        {
            get;
            set;
        }

        [DataMember]
        public bool HasNextBilling
        {
            get
            {
                return NextBillingDate.HasValue && NextBillAmount > 0;
            }
            private set { }         
        }

        [DataMember]
        public DateTime? NextBillingDate
        {
            get;
            set;
        }

        [DataMember]
        public decimal? NextBillAmount
        {
            get;
            set;
        }

        [DataMember]
        public string NextBillCurrency
        {
            get;
            set;
        }

        [DataMember]
        public DateTime DateStarted
        {
            get;
            set;
        }

        [DataMember]
        public DateTime DateExpires
        {
            get;
            set;
        }

        [DataMember]
        public DateTime DateUpdated
        {
            get;
            set;
        }

        [DataMember]
        public DateTime DateCancelled { get; set; }
        
        private ProductBillingPlan Plan
        {
            get
            {
                if (!this.BillingPlanID.IsNullOrEmpty())
                {
                    return ProductBillingPlan.GetByID(this.BillingPlanID);
                }
                else
                {
                    return null;
                }
            }            
        }

        [DataMember]
        public DateTime EndOfTerm
        {
            get
            {
                return CalculateEndOfTerm();
            }
            set {}            
        }

        private DateTime CalculateEndOfTerm()
        {
            if (Plan == null)
            {
                return DateExpires;
            }
            if (HasNextBilling)
            {
                if (!Plan.HasCurrency(CurrencyType.Token))
                {
                    return NextBillingDate.Value;
                }
                else
                {
                    AccountInformation account = AccountInformation.GetByID(this.AccountID);
                    ProductPrice tokenCost = Plan.GetPrice(CurrencyType.Token);
                    AccountTokenAmount tokenBalance = account.TokenBalances.FirstOrDefault(p => p.Type.ID == tokenCost.TokenType.ID);
                    if (tokenBalance == null || tokenBalance.Amount == 0)
                    {
                        return NextBillingDate.Value;
                    }

                    int tokenBalanaceAmount = tokenBalance.Amount;
                    decimal periods = tokenBalanaceAmount / tokenCost.Amount;
                    int planDays = Plan.FirstPayPeriod.Days;
                    decimal availableDays = periods * planDays;
                    return (HasNextBilling ? NextBillingDate.Value : DateTime.UtcNow).AddDays((double)availableDays);
                }
            }
            else
            {
                return DateExpires;
            }
        }

        [DataMember]
        public string PaymentMethodID
        {
            get;
            set;
        }

        [DataMember]
        public AccountPaymentMethod PaymentMethod
        {
            get
            {
                if (!PaymentMethodID.IsNullOrEmpty())
                {
                    return AccountPaymentMethod.GetByID(PaymentMethodID);
                }
                else
                {
                    return null;
                }
            }
            private set { }
        }  

        [DataMember]
        public SubscriptionStatus Status
        {
            get;
            set;
        }

        [DataMember]
        public DisplaySubscriptionStatus DisplayStatus
        {
            get
            {
                if (Status == SubscriptionStatus.Active && DateStarted <= DateTime.UtcNow)
                {
                    return DisplaySubscriptionStatus.Active;
                }

                if (Status == SubscriptionStatus.Active && DateStarted > DateTime.UtcNow || Status == SubscriptionStatus.PendingCustomerAction)
                {
                    return DisplaySubscriptionStatus.Pending;
                }

                if (Status == SubscriptionStatus.Suspended)
                {
                    return DisplaySubscriptionStatus.Cancelled;
                }

                if (Status != SubscriptionStatus.Active && (DateExpires - DateStarted).TotalDays < 1)
                {
                    return DisplaySubscriptionStatus.Cancelled;
                }

                if (Status != SubscriptionStatus.Active && DateStarted > DateTime.UtcNow && !HasNextBilling)
                {
                    return DisplaySubscriptionStatus.Cancelled;
                }

                if (Status != SubscriptionStatus.Active && DateExpires > DateTime.UtcNow)
                {
                    return DisplaySubscriptionStatus.Expiring;
                }

                return DisplaySubscriptionStatus.Expired;
            }
            private set
            {
            }
        }
        
        public static Subscription FromVindicia(AutoBill autoBill)
        {                        
            Subscription sub = new Subscription();
            sub.UpdateFromVindicia(autoBill);            
            return sub;
        }

        public static Subscription GetSubscription(string id)
        {
            return Subscription.GetBySubscriptionID(id, true);
        }

        public void UpdateFromVindicia(AutoBill autoBill)
        {
            this.SubscriptionID = autoBill.merchantAutoBillId;
            this.AccountID = autoBill.account.merchantAccountId;
            this.BillingPlanID = autoBill.billingPlan.merchantBillingPlanId;
            this.ProductID = autoBill.product.merchantProductId;            
            this.DateStarted = autoBill.startTimestamp.ConvertFromVindicia();
            this.DateExpires = autoBill.endTimestamp.ConvertFromVindicia();
            this.Status = autoBill.status.ToCurse();
            
            
            if (autoBill.paymentMethod != null)
            {
                this.PaymentMethodID = autoBill.paymentMethod.merchantPaymentMethodId;
            }
            else
            {
                this.PaymentMethodID = null;
            }

            bool hasFutureBilling = false;
            if (autoBill.futureRebills != null && autoBill.futureRebills.Length > 0)
            {
                Transaction nextBilling = null;


                // For token based subscriptions, use the first billing,
                // otherwise use the first billing that is at least one day in the future.
                if (autoBill.currency == "_VT")
                {
                    nextBilling = autoBill.futureRebills.FirstOrDefault();
                }
                else
                {
                    nextBilling = autoBill.futureRebills.FirstOrDefault(t => t.timestamp > this.DateStarted.AddHours(24));
                }
                
                if (nextBilling != null)
                {
                    this.NextBillAmount = nextBilling.amount;
                    this.NextBillCurrency = nextBilling.currency;
                    this.NextBillingDate = nextBilling.timestamp;
                    hasFutureBilling = true;
                }
            }

            if (!hasFutureBilling)
            {
                this.NextBillingDate = null;
                this.NextBillAmount = null;
                this.NextBillCurrency = null;
            }
        }

        public void Cancel()
        {
            Cancel(false);
        }

        public void Cancel(bool disentitle)
        {
            AutoBill ab = new AutoBill() { merchantAutoBillId = this.SubscriptionID };

            // If the auto bill does not even start until the future, force it to cancel, despite the minimum term commitment;
            if (this.DateStarted > DateTime.UtcNow.AddDays(1))
            {
                ab.cancel(false, true);

            }
            else
            {
                ab.cancel(disentitle, false); 
            }
            this.UpdateFromVindicia(ab);
            this.DateCancelled = DateTime.UtcNow;
            this.Save();            
        }

        public static void FinalizePayPalAuthorization(string vid)
        {
            AutoBill ab = new AutoBill();
            TransactionStatus transactionStatus;
            Return r = ab.finalizePayPalAuth(vid, true, out transactionStatus);
            
            if (r.returnCode != ReturnCode.Item200)
            {
                Logger.Debug("Finalize Paypal Authorization -- unsuccessful return code from Vindicia:", r.returnCode.ToString());
                throw new Exception("Unable to finalize paypal subscription authorization!");   
            }

            Logger.Debug("Finalize Paypal Authorization -- getting sub");
            Subscription sub = Subscription.GetBySubscriptionID(ab.merchantAutoBillId, false);
            Logger.Debug("Finalize Paypal Authorization -- updating sub:", sub);
            sub.UpdateFromVindicia(ab);
            Logger.Debug("Finalize Paypal Authorization -- saving sub");
            sub.Save();
            Logger.Debug("Finalize Paypal Authorization complete.");
        }

        /// <summary>
        /// Delays the next billing of the subscription until the provided date
        /// </summary>
        /// <param name="userID"></param>
        /// <param name="nextBilldate"></param>
        /// <param name="transactionType"></param>
        /// <param name="note"></param>
        /// <returns></returns>
        public bool DelayBillingToDate(DateTime nextBillDate)
        {
            if (this.Status != SubscriptionStatus.Active)
            {
                throw new InvalidOperationException("DelayBillingToDate may only be called on Active subscriptions");
            }

            if (nextBillDate < DateTime.UtcNow)
            {
                throw new ArgumentException("nextBillDate cannot be in the past", "nextBillDate");
            }
            
            // Create a Vindicia AutoBill object, given their unique ID 
            AutoBill ab = new AutoBill() { merchantAutoBillId = this.SubscriptionID };
            DateTime nextBillingDate;
            Decimal nextBillingAmount;
            string nextBillingCurrency;

            // Attempt to delay auto billing
            Return sr = ab.delayBillingToDate(nextBillDate.PrepareForVindicia(), true, out nextBillingDate, out nextBillingAmount, out nextBillingCurrency);

            bool successful = (sr.returnCode == ReturnCode.Item200);

            // If it was successful, save an activity
            if (successful)
            {
                this.Expire();
            }


            return successful;
        }


        public void Save()
        {
            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                this.Save(conn, null);
            }
        }
        
        public void Save(SqlConnection conn, SqlTransaction txn)
        {            
            SqlCommand cmd = conn.CreateCommand();
            if (txn != null)
            {
                cmd.Transaction = txn;
            }
            cmd.CommandType = System.Data.CommandType.StoredProcedure;
            cmd.CommandText = "spUpdateSubscription";

            SqlParameter idParameter = null;
            idParameter = cmd.Parameters.Add("@ID", SqlDbType.Int);
            idParameter.Direction = ParameterDirection.InputOutput;
            if (this.ID > 0)
            {
                idParameter.Value = this.ID;
            }

            SqlParameter subscriptionIDParameter = null;
            subscriptionIDParameter = cmd.Parameters.Add("@SubscriptionID", SqlDbType.VarChar);
            subscriptionIDParameter.Direction = ParameterDirection.InputOutput;
            subscriptionIDParameter.Size = 64;
            if (this.SubscriptionID != null)
            {
                subscriptionIDParameter.Value = this.SubscriptionID;
            }
            else
            {
                subscriptionIDParameter.Value = System.DBNull.Value;
            }
           
            SqlParameter param = null;                

            param = cmd.Parameters.Add("@AccountID", SqlDbType.VarChar);
            param.Size = 64;
            param.Value = this.AccountID;

            param = cmd.Parameters.Add("@ProductID", SqlDbType.VarChar);
            param.Size = 64;
            param.Value = this.ProductID;

            param = cmd.Parameters.Add("@BillingPlanID", SqlDbType.VarChar);
            param.Size = 64;
            param.Value = this.BillingPlanID;

            param = cmd.Parameters.Add("@PaymentMethodID", SqlDbType.VarChar);
            param.Size = 64;
            if (this.PaymentMethodID != null)
            {
                param.Value = this.PaymentMethodID;
            }
            else
            {
                param.Value = System.DBNull.Value;
            }

            param = cmd.Parameters.Add("@DateStarted", SqlDbType.DateTime);
            param.Value = this.DateStarted;

            param = cmd.Parameters.Add("@DateExpires", SqlDbType.DateTime);
            if (this.DateExpires.Year < 2000)
            {
                this.DateExpires = this.DateStarted;
            }
            param.Value = this.DateExpires;

            param = cmd.Parameters.Add("@NextBillDate", SqlDbType.DateTime);
            if (this.NextBillingDate.HasValue)
            {
                param.Value = this.NextBillingDate.Value;
            }
            else
            {
                param.Value = System.DBNull.Value;
            }

            param = cmd.Parameters.Add("@NextBillAmount", SqlDbType.Money);
            if (this.NextBillAmount.HasValue)
            {
                param.Value = this.NextBillAmount.Value;
            }
            else
            {
                param.Value = System.DBNull.Value;
            }

            param = cmd.Parameters.Add("@NextBillCurrency", SqlDbType.VarChar);
            param.Size = 64;
            if (this.NextBillCurrency != null)
            {
                param.Value = this.NextBillCurrency;
            }
            else
            {
                param.Value = System.DBNull.Value;
            }

            param = cmd.Parameters.Add("@Status", SqlDbType.TinyInt);
            param.Value = (byte)this.Status;

            param = cmd.Parameters.Add("@DateCancelled", SqlDbType.DateTime);
            if (this.DateCancelled != null && this.DateCancelled != default(DateTime)) {
                param.Value = this.DateCancelled;
            }
            else { param.Value = DBNull.Value; }
                            

            cmd.ExecuteNonQuery();

            if (this.ID == 0)
            {
                this.ID = (int)idParameter.Value;
                this.SubscriptionID = (string)subscriptionIDParameter.Value;
                this.DateUpdated = DateTime.UtcNow;
            }                            

            this.Expire();
        }

        public void Expire()
        {
            HttpRuntime.Cache.Expire(CacheKey.FormatWith(this.ID));
        }       

        #region Data Access

        public static List<Subscription> GetAllByAccountID(string accountID)
        {
            List<Subscription> accountTransactions = new List<Subscription>();
            List<int> ids = GetAllIDsFromDatabaseByAccountID(accountID);
            foreach (int id in ids)
            {
                Subscription accountTransaction = GetByID(id, false);
                if (accountTransaction != null)
                {
                    accountTransactions.Add(accountTransaction);
                }
            }

            return accountTransactions;
        }

        public static Subscription GetByID(int id, bool bypassCache)
        {
            return HttpRuntime.Cache.Get(bypassCache, CacheKey.FormatWith(id), () =>
            {
                return GetFromDatabaseByID(id);
            });
        }

        public static Subscription GetBySubscriptionID(string subscriptionID, bool bypassCache)
        {
            int? id = GetIDFromDatabaseBySubscriptionID(subscriptionID);
            if (!id.HasValue)
            {
                return null;
            }
            else
            {
                return GetByID(id.Value, bypassCache);
            }
        }

        private static Subscription GetFromDatabaseByID(int id)
        {
            Subscription subscription = null;
            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = "select * from Subscription where ID = @ID";
                SqlParameter idParameter = cmd.Parameters.Add("@ID", SqlDbType.Int);
                idParameter.Value = id;

                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                    if (reader.Read())
                    {
                        subscription = new Subscription(reader);
                    }
                }
            }

            return subscription;
        }

        private static List<int> GetAllIDsFromDatabaseByAccountID(string accountID)
        {
            List<int> accountIDs = new List<int>();
            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = "select ID from Subscription where AccountID = @AccountID";
                SqlParameter transactionIDParameter = cmd.Parameters.Add("@AccountID", System.Data.SqlDbType.VarChar);
                transactionIDParameter.Size = 64;
                transactionIDParameter.Value = accountID;

                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        accountIDs.Add(reader.GetInt32(0));
                    }
                }
            }

            return accountIDs;
        }

        private static int? GetIDFromDatabaseBySubscriptionID(string subscriptionID)
        {

            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = "select ID from Subscription where SubscriptionID = @SubscriptionID";
                SqlParameter transactionIDParameter = cmd.Parameters.Add("@SubscriptionID", System.Data.SqlDbType.VarChar);
                transactionIDParameter.Size = 64;
                transactionIDParameter.Value = subscriptionID;

                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                    if (reader.Read())
                    {
                        return reader.GetInt32(0);
                    }
                }
            }

            return null;
        }

        #endregion


    }
}
