﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Runtime.Serialization;
using Curse.Extensions;
using Curse.Billing.Extensions;
using System.Data.SqlClient;
using Curse.Billing.Configuration;
using System.Data;

namespace Curse.Billing.Models
{
    [DataContract]
    public class SubscriptionHistory
    {
        #region Properties

        [DataMember]
        public int ID { get; set; }

        [DataMember]
        public int UserID { get; set; }

        [DataMember]
        public int CobaltInstanceID { get; set; }

        [DataMember]
        public string CobaltInstanceName
        {
            get
            {
                return CobaltInstance.GetByID(CobaltInstanceID).Name;
            }
            private set {}
        }

        [DataMember]
        public string EntitledAccountID { get; set; }

        [DataMember]
        public int EntitledEntityTypeID { get; set; }

        [DataMember]
        public int EntitledEntityID { get; set; }

        [DataMember]
        public DateTime Timestamp { get; set; }

        [DataMember]
        public int? TransactionID { get; set; }

        [DataMember]
        public string SubscriptionID { get; set; }

        [DataMember]
        public string ProductID { get; set; }

        [DataMember]
        public string BillingPlanID { get; set; }

        [DataMember]
        public string PaymentMethodID { get; set; }
        
        #endregion

        public SubscriptionHistory() { }

        public SubscriptionHistory(SqlDataReader reader)
        {
            ID = (int) reader["ID"];
            UserID = (int)reader["UserID"];
            CobaltInstanceID = (int)reader["CobaltInstance"];
            EntitledAccountID = (string)reader["EntitledAccountID"];
            EntitledEntityTypeID = (int)reader["EntitledEntityTypeID"];
            EntitledEntityID = (int)reader["EntitledEntityID"];
            Timestamp = (DateTime)reader["Timestamp"];

            if (reader["TransactionID"] != System.DBNull.Value)
            {
                TransactionID = (int)reader["TransactionID"];
            }

            SubscriptionID = (string)reader["SubscriptionID"];
            ProductID = (string)reader["ProductID"];
            BillingPlanID = (string)reader["BillingPlanID"];
            PaymentMethodID = (string)reader["PaymentMethodID"];
        }

        public Subscription ToSubscription()
        {
            AccountInformation account = AccountInformation.GetByID(EntitledAccountID);
            return account.Subscriptions.Single(s => s.SubscriptionID == SubscriptionID);
        }

        #region Cache Access

        public static List<SubscriptionHistory> GetAllByUserID(int userID)
        {
            IEnumerable<int> ids = GetAllIDsFromDatabaseByUserID(userID);
            return GetAllByIDs(ids);
        }
        public static List<SubscriptionHistory> GetAllByUserID(int userID, int maxRecords)
        {
            IEnumerable<int> ids = GetAllIDsFromDatabaseByUserID(userID, maxRecords);
            return GetAllByIDs(ids);
        }
        
        public static List<SubscriptionHistory> GetAllByIDs(IEnumerable<int> ids)
        {
            List<SubscriptionHistory> instances = new List<SubscriptionHistory>();
            foreach (int id in ids)
            {
                SubscriptionHistory SubscriptionHistory = GetByID(id);
                if (SubscriptionHistory != null)
                {
                    instances.Add(SubscriptionHistory);
                }
            }

            return instances;
        }

        public static SubscriptionHistory GetBySubscriptionID(string subscriptionID)
        {
            return GetBySubscriptionID(subscriptionID, false);
        }

        public static SubscriptionHistory GetBySubscriptionID(string subscriptionID, bool bypassCache)
        {
            return HttpRuntime.Cache.Get(bypassCache, "SubscriptionHistoryBySubscriptionID-{0}".FormatWith(subscriptionID), () =>
            {
                return GetFromDatabaseBySubscriptionID(subscriptionID);
            });
        }

        public static SubscriptionHistory GetByID(int id)
        {
            return GetByID(id, false);
        }
        public static SubscriptionHistory GetByID(int id, bool bypassCache)
        {
            return HttpRuntime.Cache.Get(bypassCache, "SubscriptionHistoryByID-{0}".FormatWith(id), () =>
            {
                return GetFromDatabaseByID(id);
            });                       
        }

        #endregion

        #region Database Access

        private static SubscriptionHistory GetFromDatabaseByID(int id)
        {
            SubscriptionHistory instance = null;
            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = "select * from SubscriptionHistory where ID = @ID";
                SqlParameter accountIDParameter = cmd.Parameters.Add("@ID", SqlDbType.Int);
                accountIDParameter.Value = id;

                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                    if (reader.Read())
                    {
                        instance = new SubscriptionHistory(reader);
                    }
                }
            }

            return instance;
        }

        private static IEnumerable<int> GetAllIDsFromDatabaseByUserID(int userID)
        {
            List<int> ids = new List<int>();

            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = "SELECT id FROM SubscriptionHistory WHERE UserID = @UserID";
                SqlParameter accountIDParameter = cmd.Parameters.Add("@UserID", SqlDbType.Int);
                accountIDParameter.Value = userID;

                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        ids.Add(reader.GetInt32(0));
                    }
                }
            }

            return ids;
        }

       
        private static IEnumerable<int> GetAllIDsFromDatabaseByUserID(int userID, int maxRecords)
        {
            List<int> ids = new List<int>();

            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                SqlCommand cmd = conn.CreateCommand();
                // You can't parameterize a top in sqlserver...
                cmd.CommandText = "SELECT TOP " + maxRecords + " id FROM SubscriptionHistory WHERE UserID = @UserID ORDER BY Timestamp DESC";
                SqlParameter accountIDParameter = cmd.Parameters.Add("@UserID", SqlDbType.Int);
                accountIDParameter.Value = userID;

                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        ids.Add(reader.GetInt32(0));
                    }
                }
            }

            return ids;
        }

        private static SubscriptionHistory GetFromDatabaseBySubscriptionID(string subscriptionID)
        {
            SubscriptionHistory instance = null;
            using (SqlConnection conn = DatabaseConfiguration.GetBillingConnection())
            {
                SqlCommand cmd = conn.CreateCommand();
                cmd.CommandText = "select * from SubscriptionHistory where SubscriptionID = @SubscriptionID";
                SqlParameter subIDParameter = cmd.Parameters.Add("@SubscriptionID", SqlDbType.VarChar);
                subIDParameter.Size = 64;
                subIDParameter.Value = subscriptionID;

                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                    if (reader.Read())
                    {
                        instance = new SubscriptionHistory(reader);
                    }
                }
            }

            return instance;
        }

        public void Save(SqlConnection conn, SqlTransaction txn)
        {
            if (this.ID > 0)
            {
                throw new Exception("May not call Save() on an existing history record");
            }

            SqlCommand cmd = conn.CreateCommand();
            if (txn != null)
            {
                cmd.Transaction = txn;
            }
            cmd.CommandType = System.Data.CommandType.StoredProcedure;
            cmd.CommandText = "spCreateSubscriptionHistory";            

            SqlParameter idParameter = null;
            idParameter = cmd.Parameters.Add("@ID", SqlDbType.Int);
            idParameter.Direction = ParameterDirection.Output;

            
            SqlParameter param = null;

            param = cmd.Parameters.Add("@SubscriptionID", SqlDbType.VarChar);
            param.Size = 64;
            param.Value = this.SubscriptionID;

            param = cmd.Parameters.Add("@UserID", SqlDbType.Int);
            param.Value = this.UserID;

            param = cmd.Parameters.Add("@CobaltInstance", SqlDbType.Int);
            param.Value = this.CobaltInstanceID;

            param = cmd.Parameters.Add("@EntitledAccountID", SqlDbType.VarChar);
            param.Size = 64;
            param.Value = this.EntitledAccountID;

            param = cmd.Parameters.Add("@EntitledEntityTypeID", SqlDbType.Int);
            param.Value = this.EntitledEntityTypeID;

            param = cmd.Parameters.Add("@EntitledEntityID", SqlDbType.Int);
            param.Value = this.EntitledEntityID;

            param = cmd.Parameters.Add("@Timestamp", SqlDbType.DateTime);
            param.Value = this.Timestamp;

            param = cmd.Parameters.Add("@TransactionID", SqlDbType.Int);
            if (this.TransactionID.HasValue)
            {
                param.Value = this.TransactionID.Value;
            }
            else
            {
                param.Value = System.DBNull.Value;
            }

            param = cmd.Parameters.Add("@ProductID", SqlDbType.VarChar);
            param.Size = 64;
            param.Value = this.ProductID;

            param = cmd.Parameters.Add("@BillingPlanID", SqlDbType.VarChar);
            param.Size = 64;
            param.Value = this.BillingPlanID;

            param = cmd.Parameters.Add("@PaymentMethodID", SqlDbType.VarChar);
            param.Size = 64;

            if (this.PaymentMethodID == null)
            {
                param.Value = System.DBNull.Value;
            }
            else
            {
                param.Value = this.PaymentMethodID;
            }
                        
            cmd.ExecuteNonQuery();            

            if (this.ID == 0)
            {
                this.ID = (int)idParameter.Value;                
            }
        }

       
        #endregion
    }
}
