﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using Vindicia;
using Curse.Billing.Models;
using Curse.Extensions;

namespace Curse.Billing.Extensions
{
    public static class BillingPlanExtensions
    {
        public static ProductBillingPlan ToCurse(this BillingPlan value)
        {

            ProductBillingPlan plan = new ProductBillingPlan()
            {
                ID = value.merchantBillingPlanId,
                Description = value.description,
                Periods = value.ToCursePeriods(),                
            };

            if (value.nameValues != null)
            {
                NameValuePair nvp = null;


                // Get the product that this coupon grants
                nvp = value.nameValues.FirstOrDefault(p => p.name == "GrantedProduct");
                if (nvp != null)
                {
                    plan.GrantedProductID = nvp.value;
                }

                // Get the billing plan that this coupon grants
                nvp = value.nameValues.FirstOrDefault(p => p.name == "GrantedBillingPlan");
                if (nvp != null)
                {
                    plan.GrantedBillingPlanID = nvp.value;
                }

                // Get the billing plan that this coupon grants
                nvp = value.nameValues.FirstOrDefault(p => p.name == "DelayedBillingPlan");
                if (nvp != null)
                {
                    plan.DelayedPlanID = nvp.value;
                }

                // Make all merchant data accessible
                plan.MerchantData = value.nameValues.ToDictionary(n => n.name, n => n.value);

                // Prevent First billing should be used when we actually want a zero dollar first pay period,
                // such as in an promotion
                string preventFirstBilling = bool.FalseString;
                if (plan.MerchantData.TryGetValue("PreventFirstBilling", out preventFirstBilling))
                {
                    if (bool.Parse(preventFirstBilling))
                    {
                        // Set the first pay period
                        plan.FirstPayPeriod = plan.Periods.FirstOrDefault();
                    }
                }
            }
            else
            {
                plan.MerchantData = new Dictionary<string, string>();
            }

            if(plan.FirstPayPeriod == null)
            {
                // Set the first pay period
                plan.FirstPayPeriod = plan.Periods.FirstOrDefault(p => p.Prices.Exists(q => q.Amount > 0));
            }

            if (value.merchantEntitlementIds != null && value.merchantEntitlementIds.Any())
            {
                plan.EntitlementIDs = value.merchantEntitlementIds.Select(me => me.id).OrderBy(s => s).ToList();
            }
            else
            {
                plan.EntitlementIDs = new List<string>();
            }

            return plan;
        }

        // Converts a Vindicia BillingPlanPeriod to a Curse ProductBillingPeriod
        public static ProductBillingPeriod ToCurse(this BillingPlanPeriod value)
        {

            ProductBillingPeriod period = new ProductBillingPeriod()
            {
                Type = value.type.ToCurse(),
                Quantity = value.quantity,
                Days = value.GetPlanDurationDays(),
                Cycles = value.cycles,
                Prices = value.ToCursePrices()
            };

            return period;
        }

        // Converts an array of Vindicia BillingPlanPeriods to Curse ProductBillingPeriods
        public static List<ProductBillingPeriod> ToCursePeriods(this BillingPlan value)
        {
            List<ProductBillingPeriod> periods = new List<ProductBillingPeriod>();
            foreach (BillingPlanPeriod period in value.periods)
            {
                periods.Add(period.ToCurse());
            }
            return periods;
        }

        // Converts an array of Vindicia BillingPlanPeriods to Curse ProductBillingPeriods
        public static List<ProductPrice> ToCursePrices(this BillingPlanPeriod value)
        {
            List<ProductPrice> prices = new List<ProductPrice>();

            foreach (BillingPlanPrice price in value.prices)
            {
                prices.Add(price.ToCurse());
            }

            return prices;
        }

        public static ProductPrice ToCurse(this BillingPlanPrice value)
        {

            CurrencyType currencyType = CurrencyType.USD;

            if (value.tokenAmount != null)
            {
                return new ProductPrice()
                {
                    Amount = value.tokenAmount.amount,
                    TokenType = AccountTokenType.FromVindicia(value.tokenAmount.token),
                    Currency = CurrencyType.Token
                };
            }

            try
            {
                currencyType = (CurrencyType)Enum.Parse(typeof(CurrencyType), value.currency, true);
            }
            catch { }
            
            return new ProductPrice()
            {
                Amount = value.amount,
                Currency = currencyType
            };
        }

        public static int GetPlanDurationDays(this BillingPlanPeriod value)
        {

            switch (value.type)
            {
                case BillingPeriodType.Day:
                    return value.quantity;

                case BillingPeriodType.Month:
                    return value.quantity * 31;

                case BillingPeriodType.Week:
                    return value.quantity * 7;

                case BillingPeriodType.Year:
                    return value.quantity * 365;

                default:
                    throw new Exception("Unknown Billing Period Type!");

            }

        }

        public static ProductBillingPeriodType ToCurse(this BillingPeriodType value)
        {
            switch (value)
            {
                case BillingPeriodType.Day:
                    return ProductBillingPeriodType.Day;

                case BillingPeriodType.Month:
                    return ProductBillingPeriodType.Month;

                case BillingPeriodType.Week:
                    return ProductBillingPeriodType.Week;

                case BillingPeriodType.Year:
                    return ProductBillingPeriodType.Year;

                default:
                    throw new Exception("Unknown Billing Period Type!");

            }
        }
    }
}
