﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Net.Security;
using System.Text;
using System.ServiceModel;
using System.Collections.Concurrent;
using System.Diagnostics;
using System.ServiceModel.Description;
using System.Threading;
using System.Web;
using Curse.Logging;

namespace Curse.Caching
{
    public class CacheCluster : IDisposable
    {
        public static readonly CacheCluster Instance = new CacheCluster();

        private ServiceHost _serviceHost;
        private const string ServiceBindingUri = "net.tcp://{0}:{1}/cache-service";

        private int _id;
        private string _hostName;
        private int _portNumber;
        private DateTime _lastBroadCast = DateTime.UtcNow;

        private readonly ConcurrentQueue<string> _pendingKeys = new ConcurrentQueue<string>();
        private readonly ConcurrentDictionary<long, ClusterMemberConnection> _clusterMembers = new ConcurrentDictionary<long, ClusterMemberConnection>();
        private bool _started = false;


        private class ClusterMemberConnection : IDisposable
        {
            public long ID { get; private set; }
            public string IpAddress { get; private set; }
            public int PortNumber { get; private set; }
            private CacheServiceClient _cacheServiceClient { get; set; }
            
            private readonly object _syncRoot = new object();

            public CacheServiceClient Client
            {
               get
               {                   
                    return _cacheServiceClient;
                   
               }
            }

            public bool EnsureConnectivity()
            {
                lock (_syncRoot)
                {

                    try
                    {
                        if (_cacheServiceClient == null)
                        {
                            _cacheServiceClient = CreateNewClient();
                            _cacheServiceClient.Open();
                        }
                        else if (_cacheServiceClient.State != CommunicationState.Opened)
                        {
                            if (_cacheServiceClient.State == CommunicationState.Faulted)
                            {
                                _cacheServiceClient.Abort();
                            }
                            else if (_cacheServiceClient.State != CommunicationState.Closed)
                            {
                                _cacheServiceClient.Close();
                            }

                            _cacheServiceClient = CreateNewClient();
                            _cacheServiceClient.Open();
                        }

                        return true;
                    }
                    catch (CommunicationException)
                    {
                        return false;
                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex, "Unable to establish a management service connection to: " + IpAddress);
                        return false;
                    }
                }
            }

            public int FailureCount { get; private set; }

            public ClusterMemberConnection(long id, string ipAddress, int portNumber)
            {
                ID = id;
                IpAddress = ipAddress;
                PortNumber = portNumber;                
            }

            private CacheServiceClient CreateNewClient()
            {
                var uri = new Uri(string.Format(ServiceBindingUri, IpAddress, PortNumber));
                var binding = new NetTcpBinding(SecurityMode.None)
                {
                    OpenTimeout = TimeSpan.FromSeconds(5),
                    SendTimeout = TimeSpan.FromSeconds(5),
                    ReceiveTimeout = TimeSpan.FromHours(1)
                };

                binding.SendTimeout = TimeSpan.FromMinutes(30);
                binding.Security.Message.ClientCredentialType = MessageCredentialType.None;
                binding.Security.Transport.ClientCredentialType = TcpClientCredentialType.None;

                var endpoint = new EndpointAddress(uri);
                return new CacheServiceClient(binding, endpoint);
            }

            public void TrackFailure()
            {
                ++FailureCount;
            }

            public void Dispose()
            {
                if (_cacheServiceClient != null)
                {
                    var conn = _cacheServiceClient;
                    _cacheServiceClient = null;
                    try
                    {
                        conn.Close();
                    }
                    catch (Exception ex)
                    {
                        Logger.Warn(ex, "Failed to disconnect from: " + IpAddress);
                    }

                }
            }
        }

        public void RegisterMember(int id, string hostName, int port)
        {
            if (_clusterMembers.ContainsKey(id))
            {
                return;
            }

#if DEBUG
            if (hostName.ToLower().Contains("-live"))
            {
                Logger.Warn("Skipping attempt by a live server to register itself to the cluster!");
                return;
            }
#else
            if (!hostName.ToLower().Contains("-live"))
            {
                Logger.Warn("Skipping attempt by a dev server to register itself to the cluster!");
                return;
            }
#endif

            Logger.Info("Discovered a new cluster member at " + hostName + ":" + port);            
            var member = new ClusterMemberConnection(id, hostName, port);
            _clusterMembers.TryAdd(id, member);
        }

        public void Start()
        {
            if (_started)
            {
                throw new Exception("Already started!");
            }
            _started = true;

            foreach (var clusterMember in ClusterManager.Instance.Members.Values)
            {
                if (clusterMember == ClusterManager.Instance.Self)
                {
                    continue;
                }
                RegisterMember(clusterMember.ID, clusterMember.HostName, clusterMember.Port);
            }
            _id = ClusterManager.Instance.Self.ID;
            _hostName = ClusterManager.Instance.Self.HostName;
            _portNumber = ClusterManager.Instance.Self.Port;

            var baseAddress = new Uri(string.Format(ServiceBindingUri, _hostName, _portNumber));
            var host = new ServiceHost(typeof(CacheService), baseAddress);
            var binding = new NetTcpBinding(SecurityMode.None, false)
            {
                MaxConnections = Int32.MaxValue,
                PortSharingEnabled = false,
                MaxReceivedMessageSize = Int32.MaxValue,
                ReaderQuotas = {MaxArrayLength = Int32.MaxValue},
                TransactionFlow = false,
                ListenBacklog = 1000
            };

            binding.Security.Message.ClientCredentialType = MessageCredentialType.None;
            binding.Security.Transport.ClientCredentialType = TcpClientCredentialType.None;
            binding.Security.Transport.ProtectionLevel = ProtectionLevel.None;            

            binding.OpenTimeout = TimeSpan.FromSeconds(5);
            binding.SendTimeout = TimeSpan.FromSeconds(5);
            binding.ReceiveTimeout = TimeSpan.FromHours(1);
            binding.SendTimeout = TimeSpan.FromMinutes(30);


            host.AddServiceEndpoint(typeof(ICacheService), binding, string.Empty);
            
            // Service Throttling
            var stb = new ServiceThrottlingBehavior();
            host.Description.Behaviors.Remove<ServiceThrottlingBehavior>();
            stb.MaxConcurrentCalls = Int32.MaxValue;
            stb.MaxConcurrentInstances = Int32.MaxValue;
            stb.MaxConcurrentSessions = Int32.MaxValue;
            host.Description.Behaviors.Add(stb);

            // Service Meta Data
            var serviceMetadataBehavior = new ServiceMetadataBehavior();
            serviceMetadataBehavior.HttpGetEnabled = false;
            host.Description.Behaviors.Remove<ServiceMetadataBehavior>();
            host.Description.Behaviors.Add(serviceMetadataBehavior);

            var attempts = 0;
            while (true)
            {
                try
                {
                    host.Open();
                    Logger.Info("Cache cluster host opened!");
                    break;
                }
                catch (Exception ex)
                {
                    if (++attempts >= 20)
                    {
                        Logger.Error(ex, "Cache cluster host could not be opened! This server will not receive cache invalidations.");
                        throw;
                    }

                    Logger.Warn(ex, "Failed to open cache cluster host. Trying again in 500ms...");
                    Thread.Sleep(500);
                }                
            }
            
            _serviceHost = host;

            new Thread(ProcessPendingKeys) { IsBackground = true }.Start();
        }

        public void Dispose()
        {
            _started = false;

            if (_serviceHost != null)
            {
                try
                {
                    _serviceHost.Close(TimeSpan.FromSeconds(5));
                    _serviceHost = null;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to dispose cache cluster!");
                }
            }
        }

        public void TransmitInvalidationKey(string key)
        {
            if (!_started || _clusterMembers.Count == 0)
            {
                return;
            }

            _pendingKeys.Enqueue(key);
        }

        public void TransmitInvalidationKeys(string[] keys)
        {
            if (!_started || _clusterMembers.Count == 0)
            {
                return;
            }

            foreach (string key in keys)
            {
                TransmitInvalidationKey(key);
            }
        }

        private void ProcessPendingKeys()
        {
            while (true)
            {
                Thread.Sleep(1);

                try
                {
                    if ((DateTime.UtcNow - _lastBroadCast) > TimeSpan.FromSeconds(30))
                    {
                        Broadcast();
                    }
                }
                catch (ThreadAbortException)
                {
                    break;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Unhandled exception during broadcast!");
                }

                try
                {
                    InternalProcessPendingKeys();
                }
                catch (ThreadAbortException)
                {
                    break;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Unhandled exception processing pending keys!");                    
                }
            }
        }
        
        private void Broadcast()
        {
            _lastBroadCast = DateTime.UtcNow;

            // Send a transmission to each cluster member
            foreach (var kvp in _clusterMembers)
            {
                var conn = kvp.Value;
                
                try
                {
                    if (!conn.EnsureConnectivity())
                    {
                        conn.TrackFailure();
                        continue;
                    }

                    conn.Client.RegisterSelf(_id, _hostName, _portNumber);
                }
                catch (Exception ex)
                {
                    Logger.Error("Failed to broadcast to cluster member" + ex);                    
                    conn.TrackFailure();                    
                }
            }
        }

        private void InternalProcessPendingKeys()
        {
            if (!_started)
            {
                return;
            }

            if (_pendingKeys.Count == 0)
            {
                return;
            }

            var keys = new List<string>();
            for (var i = 0; i < 100; i++)
            {
                string key = null;
                if (!_pendingKeys.TryDequeue(out key))
                {
                    break;
                }
                keys.Add(key);
            }

            if (keys.Count == 0)
            {
                return;
            }

            // Send a transmission to each cluster member
            foreach (var kvp in _clusterMembers)
            {
                var conn = kvp.Value;

                
                try
                {
                    if (!conn.EnsureConnectivity())
                    {
                        conn.TrackFailure();
                        continue;
                    }

                    conn.Client.ExpireCacheKeys(keys.ToArray());
                }
                catch (Exception ex)
                {
                    conn.TrackFailure();
                    Logger.Warn(ex, "Failed to transmit cache key expirations to host!", new { conn.ID, conn.IpAddress, conn.PortNumber});
                }
            }

            var failedConnections = _clusterMembers.Values.Where(p => p.FailureCount > 10).ToArray();

            foreach (var conn in failedConnections)
            {
                Logger.Error("Removing a dead cluster member experiencing too many failures.", new {conn.ID, conn.IpAddress, conn.PortNumber, conn.FailureCount});
                ClusterMemberConnection removedConn = null;
                if (_clusterMembers.TryRemove(conn.ID, out removedConn))
                {
                    removedConn.Dispose();
                }
            }
        }

        public void TestClients(string[] keys)
        {
            // Send a transmission to each cluster member
            foreach (var kvp in _clusterMembers)
            {
                try
                {
                    if (!kvp.Value.EnsureConnectivity())
                    {
                        continue;
                    }
                    kvp.Value.Client.ExpireCacheKeys(keys);
                }
                catch (Exception)
                {
                    Debug.WriteLine("Failed to transmit message to: " + kvp.Key.ToString());
                }
            }
        }

        public void ReceiveInvalidationKey(string key)
        {
            CacheManager.Remove(key);
        }

        public void ReceiveInvalidationKey(string[] keys)
        {
            if (!_started)
            {
                return;
            }

            foreach (string key in keys)
            {
                CacheManager.Remove(key);
            }
        }
    }
}
