﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Data.SqlClient;
using System.Net;
using Curse.Logging;

namespace Curse.Caching
{

    public class ClusterManager
    {
        public static readonly ClusterManager Instance = new ClusterManager();
        public Dictionary<int, ClusterMember> Members { get; private set; }
        public ClusterMember Self { get; private set; }
        private string _connectionString = null;

        public ClusterManager()
        {
            Members = new Dictionary<int, ClusterMember>();
        }

        public void Initialize(string applicationName, SqlConnection conn, bool enableLogging = false)
        {            
            if (!System.Net.NetworkInformation.NetworkInterface.GetIsNetworkAvailable())
            {
                throw new InvalidOperationException("This host cannot participate in a cluster, as it has no network available.");
            }
            
            _connectionString = conn.ConnectionString;
            
            var host = Dns.GetHostEntry(Dns.GetHostName());

            Logger.Info("Retrieved host name from DNS: " + host.HostName);

            using (var cmd = conn.CreateCommand())
            {
                cmd.CommandText = "select * from ClusterMember";
                using (var reader = cmd.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        var member = new ClusterMember(reader);
                        Members.Add(member.ID, member);
                    }
                }
            }

            int? maxPort = Members.Any() ? (int?)Members.Values.Max(p => p.Port) : null;

            Self = Members.Values.FirstOrDefault(p => p.ApplicationIdentity.Equals(applicationName, StringComparison.InvariantCultureIgnoreCase) && p.HostName == host.HostName); 
            if (Self == null)
            {
                var member = new ClusterMember
                {
                    ApplicationIdentity = applicationName,
                    HostName = host.HostName,
                    Port = maxPort.HasValue ? maxPort.Value + 1 : 8080
                };
                member.Save(conn);
                Members.Add(member.ID, member);
                Self = member;
            }

            Logger.Info("Registered with cluster!", new { Self.Port, Self.ApplicationIdentity });
        }

        public void AddMember(int id)
        {
            if (Members.ContainsKey(id))
            {
                return;                
            }

            using (var conn = new SqlConnection(_connectionString))
            {
                conn.Open();
                using (var cmd = conn.CreateCommand())
                {
                    cmd.CommandText = "select * from ClusterMember where @ID = ID";
                    cmd.Parameters.AddWithValue("@ID", id);

                    using (var reader = cmd.ExecuteReader())
                    {
                        while (reader.Read())
                        {
                            var member = new ClusterMember(reader);
                            Members.Add(member.ID, member);
                        }
                    }
                }
            }            
        }
    }
}
