﻿using System;
using System.Collections.Concurrent;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using Curse.Logging;
using Curse.Radon.DownloadTracker.Configuration;
using System.Data.SqlClient;
using System.Net;
using System.Net.Sockets;
using System.Data;
using System.Threading;

namespace Curse.Radon.DownloadTracker
{
    public class DownloadTracker : IHttpModule
    {
        static readonly LogCategory Logger = new LogCategory("DownloadTracker");

        static DownloadTracker()
        {
            new Thread(SaveToDatabaseThread)
            {
                IsBackground = true,
                Name = "DownloadTracker"
            }.Start();

            Logger.Info("Download tracker configuration mode: " + DownloadTrackerConfiguration.ActiveConfiguration);            

            RestTimeSeconds = DownloadTrackerConfiguration.Database.WriteRestTimeSeconds;
            DatabaseConnectionString = DownloadTrackerConfiguration.ConnectionString;
            RedirectUrl = DownloadTrackerConfiguration.Database.RedirectUrl;

            Logger.Info("Download tracker configuration loaded!", new { RestTimeSeconds, DatabaseConnectionString, RedirectUrl });
        }

        private static readonly ConcurrentDictionary<string, TrackedDownload> DownloadedFileIDs = new ConcurrentDictionary<string, TrackedDownload>();        
        private const string TableName = "[dbo].[FileDownload]";
        private static readonly int RestTimeSeconds = 10;
        private static readonly string DatabaseConnectionString;
        private static readonly string RedirectUrl;

        void TrackFileDownload(object sender, EventArgs e)
        {
            try
            {                

                var app = (HttpApplication)sender;
                var context = app.Context;

                var headerKeys = context.Request.Headers.AllKeys;

                // Requester is likely a download manager.
                if (headerKeys.Contains("Range") || headerKeys.Contains("Want-Digest"))
                {
                    return;
                }                

                var filePath = context.Request.FilePath;

                if (!filePath.Contains("files"))
                {
                    return;
                }

                // Get the FileID
                var parts = filePath.Split('/');

                if (parts.Length != 5)
                {
                    return;
                }

                var highIDString = parts[2];
                var lowIDString = parts[3];
                var highID = -1;
                var lowID = -1;
                var fileID = -1;

                if (int.TryParse(highIDString, out highID) && int.TryParse(lowIDString, out lowID))
                {
                    fileID = (highID * 1000) + lowID;
                }

                var address = GetClientIPAddress(context.Request);

                int? packFileID = null;
                if (context.Request.Headers["x-elerium-packfileid"] != null)
                {
                    var packValue = context.Request.Headers["x-elerium-packfileid"];
                    int temp;
                    Int32.TryParse(packValue, out temp);
                    packFileID = temp;
                }

                if (fileID <= 0)
                {
                    return;
                }

                var key = fileID + "-" + address;



                if (!DownloadedFileIDs.ContainsKey(key))
                {
#if DEBUG
                    Logger.Trace("Tracking download: " + context.Request.Url);
#endif

                    var referrer = string.Empty;

                    try
                    {
                        referrer = context.Request.UrlReferrer.NullSafeToString(4000);
                    }
                    catch (Exception ex)
                    {
                        Logger.Warn(ex, "Failed to get referrer.");
                    }

                    var userAgent = string.Empty;

                    try
                    {
                        userAgent = context.Request.UserAgent.NullSafeToString(4000);
                    }
                    catch (Exception ex )
                    {
                        Logger.Warn(ex, "Failed to get user agent.");
                    }
                    
                    DownloadedFileIDs.TryAdd(key, new TrackedDownload(address, fileID, referrer, userAgent, packFileID));
                }

                context.Response.Redirect(RedirectUrl + filePath, false);
                context.ApplicationInstance.CompleteRequest();
            }
            catch (Exception ex)
            {
                Logger.Error(ex, "Failed to track download!");
            }
        }

        private static void SaveToDatabaseThread()
        {
            while (true)
            {
                try
                {
                    Thread.Sleep(TimeSpan.FromSeconds(RestTimeSeconds));
                    SaveToDatabase();
                }
                catch (ThreadAbortException)
                {
                    return;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Failed to save to database!");
                }
            }
        }

        private static Int64 _historicalSavedCount = 0;
        private static Int64 _recentSavedCount = 0;
        private static DateTime _lastLoggedCount = DateTime.MinValue;
        private static bool _isFirstSave = false;

        static void SaveToDatabase()
        {
            if (DownloadedFileIDs.Count == 0)
            {
                return;
            }

            // Create a of copy of the dictionary
            var projectFileDownloads = new Dictionary<string, TrackedDownload>(DownloadedFileIDs);

            // Clear  the existing one
            DownloadedFileIDs.Clear();
            
            using (var conn = new SqlConnection(DatabaseConnectionString))
            {
                conn.Open();                

                // Create a data table for bulk copying
                using (var table = new DataTable())
                {

                    using (var command = new SqlCommand(string.Format("SELECT TOP 1 * FROM {0}", TableName), conn))
                    {
                        using (var reader = command.ExecuteReader(CommandBehavior.SchemaOnly))
                        {
                            table.Load(reader);
                        }
                    }

                    // Populate DataTable.
                    table.BeginLoadData();


                    foreach (var t in projectFileDownloads.Values)
                    {
                        var row = table.NewRow();
                        row["IPAddress"] = t.IPAddress.GetAddressBytes();
                        row["ProjectFileID"] = t.FileID;
                        row["DateDownloaded"] = t.DateDownloaded;
                        row["Referrer"] = t.Referrer;
                        row["UserAgent"] = t.UserAgent;
                        row["Status"] = 1;
                        if (t.PackFileID.HasValue)
                        {
                            row["PackFileID"] = t.PackFileID.Value;
                        }

                        table.Rows.Add(row);
                    }

                    table.EndLoadData();

                    // Bulk Copy into the table
                    try
                    {
                        using (var bulk = new SqlBulkCopy(conn, SqlBulkCopyOptions.Default, null))
                        {
                            bulk.BatchSize = 5000;
                            bulk.BulkCopyTimeout = 5000;
                            bulk.DestinationTableName = TableName;
                            bulk.WriteToServer(table);
                        }
                    }
                    catch (Exception ex)
                    {
                        Logger.Error(ex, "Failed to bulk copy to database!");
                    }
                }                
            }

            _historicalSavedCount += projectFileDownloads.Count;
            _recentSavedCount += projectFileDownloads.Count;

            if (_isFirstSave || DateTime.UtcNow - _lastLoggedCount > TimeSpan.FromMinutes(5))
            {
                Logger.Info("Recent Saved: " + _recentSavedCount.ToString("##,###") + ", Historical Saved: " + _historicalSavedCount.ToString("##,###"));
                _isFirstSave = false;
                _recentSavedCount = 0;
                _lastLoggedCount = DateTime.UtcNow;
            }
        }

        private IPAddress GetClientIPAddress(HttpRequest request)
        {
            var ipString = string.Empty;
            if (request.Headers != null)
            {
                ipString = request.Headers["X-Forwarded-For"];
            }
            if (string.IsNullOrEmpty(ipString))
            {
                ipString = request.UserHostAddress;
            }

            if (string.IsNullOrEmpty(ipString))
            {
                return null;
            }

            if (ipString.Contains(","))
            {
                var ipArray = ipString.Split(',');
                ipString = ipArray[ipArray.Length - 1].Trim();
            }

            IPAddress ipAddress;
            if (IPAddress.TryParse(ipString, out ipAddress))
            {
                var family = ipAddress.AddressFamily;
                if (family == AddressFamily.InterNetwork || family == AddressFamily.InterNetworkV6)
                {
                    var bytes = ipAddress.GetAddressBytes();
                    if (bytes != null && bytes.Length > 0)
                    {
                        return ipAddress;
                    }
                }
            }
            return null;
        }

        #region IHttpModule Implementation

        public String ModuleName
        {
            get { return "DownloadTracker"; }
        }

        public void Dispose()
        {
            
        }
        

        public void Init(HttpApplication application)
        {            
            application.BeginRequest += TrackFileDownload;            
        }

        #endregion
    }
}
