using System;
using System.Diagnostics;
using System.IO;
using System.Text;

namespace Curse.Contactology
{
    /**
     * Enum supporting common logging event levels
     *
     * @author Michael Comperda
     */
    public enum ELogLevel
    {
        Info,
        Access,
        Warning,
        Error,
        Debug
    }

    /**
     * Static class for thread safe generic file based logging
     *
     * @author Michael Comperda
     */
    public static class Logger
    {
        /**
         * Write-only property assigns the highest log level to report
         * Should be called during startup/reloading routines
         * Should be called at least once to change default Debug level
         *
         * @param  value highest ELogLevel to report for future log events
         */
        public static ELogLevel SetLogLevel
        {
            set
            {
                sLogLevel = value;
            }
        }
        
        
        /**
         * Write-only property assigns the directory which log files will be created
         * Thread safe, will wait for locks before setting a new file
         * Should be called during startup/reloading routines
         * Must be called at least once to generate a log file
         *
         * @param  value path including final directory seperator character to where logs are stored
         */
        public static String SetLogPath
        {
            set
            {                
                sLogPath = value;
                CreateNewLogFile();
            }
        }

        public static int SetAutoGrowthMegabytes
        {
            set
            {
                sAutoGrowBytes = value * 1048576;
                sAutoGrowLogFile = true;
            }
        }

        /**
         * Core logging method
         * Thread safe and will Assert if SetLogPath was not called
         *
         * @param  pLevel  level of the event to be logged
         * @param  pHost   host in ip form, of the log event generator, may be null
         * @param  pFormat format of the message to display
         * @params pArgs   optional parameters for a formatted message
         */
        public static void Log(ELogLevel pLevel,
                               String pHost,
                               String pFormat,
                               params Object[] pArgs)
        {
            if (pLevel <= sLogLevel)
            {

                GrowLogFile();

                String header = String.Format("{0} {1}{2}] ",
                                              DateTime.UtcNow.ToString("yyyy-MM-dd HH:mm:ss"),
                                              pLevel,
                                              pHost != null ? " " + pHost : "");
                lock (sLock)
                {                                        
                    using (StreamWriter writer = File.AppendText(sLogFileFullPath))
                    {
                        writer.WriteLine(header + pFormat, pArgs);
                    }
                }
            }
        }

        
        /**
         * Higher level warning log event producer for user driven ignored actions
         *
         * @param  pUserId user id of the user for whom this action has been ignored
         * @param  pHost   host in ip form, of the log event generator, may be null
         * @param  pAction action header for identification of what is being ignored
         * @param  pReason format of the reason this action was ignored
         * @params pArgs   optional parameters for a formatted message
         */
        public static void IgnoredAction(Int32 pUserId,
                                         String pHost,
                                         String pAction,
                                         String pReason,
                                         params Object[] pArgs)
        {
            String header = String.Format("Ignored {0}: User {1}, Reason: ",
                                          pAction,
                                          pUserId);
            Log(ELogLevel.Warning,
                pHost,
                header + pReason,
                pArgs);
        }

        public static void DumpHex(Byte[] pBuffer, Int32 pStart, Int32 pLength)
        {
            if (sLogLevel >= ELogLevel.Debug)
            {
                string data = BitConverter.ToString(pBuffer, pStart, pLength).Replace("-", " ");

                lock (sLock)
                {
                    using (StreamWriter writer = File.AppendText(sLogFileFullPath))
                    {
                        for (Int32 offset = 0; offset < data.Length; offset += (16 * 3))
                        {
                            writer.WriteLine(data.Substring(offset, Math.Min(16 * 3, data.Length - offset)));
                        }
                    }
                }
            }
        }

        /**
         * Private static member data
         */
        private static ELogLevel sLogLevel = ELogLevel.Debug;
        private static String sLogFileFullPath = null;
        private static Object sLock = new Object();
        private static string sLogPath = null;
        private static bool sAutoGrowLogFile = false;
        private static int sAutoGrowBytes = 100000;

        private static void CreateNewLogFile()
        {
            lock (sLock)
            {
                sLogFileFullPath = GetLogFilename();
                if (!Directory.Exists(sLogPath))
                {
                    Directory.CreateDirectory(sLogPath);
                }
                using (StreamWriter writer = File.CreateText(sLogFileFullPath))
                {
                }
            }

            Log(ELogLevel.Debug,
                null,
                "Log Opened");
        }

        private static string GetLogFilename()
        {
            return String.Format("{0}{1}.txt", sLogPath, DateTime.UtcNow.ToString("yyyy-MM-dd-HHmmss"));
        }

        private static void GrowLogFile()
        {
            if (!sAutoGrowLogFile)
            {
                return;
            }
            
            FileInfo finfo = null;
            lock (sLock)
            {
                finfo = new FileInfo(sLogFileFullPath);
            }

            if (finfo.Length >= sAutoGrowBytes)
            {
                CreateNewLogFile();
            }
            
        }
    }
}
