﻿using System;
using System.Web;
using System.Configuration;
using Curse.DownloadSecurity.Throttling;
using System.Collections.Generic;
using Curse.DownloadSecurity.Tokens;
using Curse.DownloadSecurity.Exceptions;

namespace Curse.DownloadSecurity
{
    class SecureDownloadModule : IHttpModule
    {
        enum AuthEncoding
        {
            v1 = 1,
            v2,
            v3
        }

        internal class AuthenticationData
        {
            readonly HttpContext _context;
            private readonly UInt32 _requestHashCode;
            private UInt32 _hashCodeCurrent;
            UInt32 _hashCodeFuture = 0;
            readonly int _hashComponent = 0;
            readonly bool _isAuthenticated;

            AuthEncoding AuthEncoding
            {
                get; 
                set;
            }

            UInt32 HashCodeCurrent
            {
                get
                {
                    if (_hashCodeCurrent != 0) return _hashCodeCurrent;

                    if (AuthEncoding == AuthEncoding.v1)
                    {
                        _hashCodeCurrent = SecurityToken.GetHashCode(_context.Request.UserHostAddress, _hashComponent, DateTime.UtcNow);
                    }
                    else
                    {
                        _hashCodeCurrent = SecurityToken.GetHashCode(_hashComponent, DateTime.UtcNow);
                    }

                    return _hashCodeCurrent;
                }
            }

            UInt32 HashCodeFuture
            {
                get
                {
                    if (_hashCodeFuture != 0) return _hashCodeFuture;

                    if (AuthEncoding == AuthEncoding.v1)
                    {
                        _hashCodeFuture = SecurityToken.GetHashCode(_context.Request.UserHostAddress, _hashComponent, DateTime.UtcNow.AddMinutes(-1));
                    }
                    else
                    {
                        return SecurityToken.GetHashCode(_hashComponent, DateTime.UtcNow.AddMinutes(-1));
                    }

                    return _hashCodeFuture;
                }
            }

            Boolean IsAuthenticated
            {
                get
                {
                    return _isAuthenticated;
                }
            }

            public AuthenticationData(HttpContext context)
            {
                _context = context;
                if (_context == null)
                {
                    throw new ArgumentNullException("context");
                }

                //Determine the encoding type
                if (!string.IsNullOrEmpty(context.Request["v3"]))
                {
                    AuthEncoding = AuthEncoding.v3;
                }
                else if (!string.IsNullOrEmpty(context.Request["v2"]))
                {
                    AuthEncoding = AuthEncoding.v2;
                }
                else if (!string.IsNullOrEmpty(context.Request["v"]))
                {
                    AuthEncoding = AuthEncoding.v1;
                }
                else
                {
                    throw new VNotPresentException();
                }

                // Get the file path
                var filePath = context.Request.FilePath;
                if (filePath == "/robots.txt" || filePath == "/pingdom.aspx" || filePath == "/")
                {
                    throw new InvalidFilePathException();
                }

                // Parse the path data based on the encoding type
                var filePathParts = filePath.Split('/');
                switch (AuthEncoding)
                {
                    case AuthEncoding.v1:
                        // Get the Addon ID
                        if (!Int32.TryParse(filePathParts[filePathParts.Length - 3], out _hashComponent))
                        {
                            throw new ParseFailedException("Unable to parse the Addon Id");
                        }

                        // Get the request Hash Code
                        if (!UInt32.TryParse(context.Request["v"], out _requestHashCode))
                        {
                            throw new ParseFailedException("Unable to parse the requested hash code");
                        }
                        break;
                    case AuthEncoding.v2:
                        // Get the Addon ID
                        if (!Int32.TryParse(filePathParts[filePathParts.Length - 3], out _hashComponent))
                        {
                            throw new ParseFailedException("Unable to parse the Addon Id");
                        }

                        // Get the request Hash Code
                        if (!UInt32.TryParse(context.Request["v2"], out _requestHashCode))
                        {
                            throw new ParseFailedException("Unable to parse the requested hash code");
                        }
                        break;
                    case AuthEncoding.v3:
                        // Get the fileId
                        int high, low = 0;
                        if (!Int32.TryParse(filePathParts[2], out high))
                        {
                            throw new ParseFailedException("Unable to parse the file Id high");
                        }

                        // Get my low
                        if (!Int32.TryParse(filePathParts[3], out low))
                        {
                            throw new ParseFailedException("Unable to parse the file Id low.");
                        }

                        //Set the hash component
                        _hashComponent = (high * 1000) + low;                        

                        // Get the request Hash Code
                        if (!UInt32.TryParse(context.Request["v3"], out _requestHashCode))
                        {
                            throw new ParseFailedException("Unable to parse the requested hash code");
                        }

                        break;
                }

                if (_requestHashCode != HashCodeCurrent && _requestHashCode != HashCodeFuture)
                {
                    _isAuthenticated = false;
                    throw new HashCodeMismatchException(string.Format("v no match, expected {0} or {1} received {2}", HashCodeCurrent, HashCodeFuture, _requestHashCode));
                }
                else
                {
                    _isAuthenticated = true;
                }
            }
        }

        private HttpApplication _currentApplication;
        private string _redirectUrl = null;

        /// <summary>
        /// Added to make debugging on the remote server easier by throwing exceptions instead of simple redirects
        /// </summary>
        private bool _throwExceptionsOnError = false;

        public void AuthenticateDownload(Object source, EventArgs e)
        {
            var context = _currentApplication.Context;
            try
            {
                var authetication = new AuthenticationData(context);
            }
            catch (Exception)
            {
                if (_throwExceptionsOnError)
                {
                    throw;
                }

                context.Response.Redirect(_redirectUrl, false);
                HttpContext.Current.ApplicationInstance.CompleteRequest();                
            }
        }
       
        /// <summary>
        /// Context-BeginRequest handler for configuring BitThrottling settings
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        void ConfigureThrottling(object sender, EventArgs e)
        {
            if (!_throttlingEnabled)
            {
                return;
            }

            var app = (HttpApplication)sender;
            var context = app.Context;

            var filePath = context.Request.FilePath;

            if (filePath == "/robots.txt" || filePath == "/pingdom.aspx" || filePath == "/") return;

            Dictionary<UInt32, int> profiles = null;
            UInt32 profileHash = 0;
            ThrottlingSettings throttlingSettings = null;
            var throttleID = 0;

            if (!string.IsNullOrEmpty(context.Request["p2"]))
            {
                profiles = BitRateThrottling.GetProfileHash(_throttlingProfiles);
                profileHash = UInt32.Parse(context.Request["p2"]);
            }
            else if (!string.IsNullOrEmpty(context.Request["p"]))
            {
                profiles = BitRateThrottling.GetProfileHash(context.Request.UserHostAddress, _throttlingProfiles);
                profileHash = UInt32.Parse(context.Request["p"]);
            }

            // Defaults to 0
            if (profiles != null && profiles.ContainsKey(profileHash))
            {
                throttleID = profiles[profileHash];
            }

            var configElement = _throttlingProfilesByID[throttleID];

            throttlingSettings = ThrottlingFactory.Instance.CreateThrottlingSettings(
                    app.Context.Request.PhysicalPath,
                    throttleID,
                    configElement.MinimumRate,
                    configElement.MaximumRate,
                    configElement.Duration);

            if (throttlingSettings == null)
            {
                throw new Exception("Unable to match settings");
            }

            BitRateThrottling.SendBitRateThrottlingServerVariables(context, throttlingSettings);
        }

        #region IHttpModule Members

        private static ThrottlingConfigurationHandler _throttlingConfiguration;
        private static ThrottlingProfileConfigCollection _throttlingProfiles;
        private static Dictionary<int, ThrottlingProfileConfigElement> _throttlingProfilesByID;
        private static bool _throttlingEnabled;

        public void Init(HttpApplication context)
        {
            _currentApplication = context;
            _redirectUrl = ConfigurationManager.AppSettings["SecureDownloadModule.RedirectUrl"];
            _throwExceptionsOnError = bool.Parse(ConfigurationManager.AppSettings["SecureDownloadModule.ThrowExceptions"]);
            _throttlingConfiguration = ThrottlingConfigurationHandler.GetConfig();
            _throttlingEnabled = _throttlingConfiguration.Throttle;
            _throttlingProfiles = _throttlingConfiguration.Profiles;

            _throttlingProfilesByID = new Dictionary<int, ThrottlingProfileConfigElement>();
            foreach (ThrottlingProfileConfigElement elem in _throttlingProfiles)
            {
                _throttlingProfilesByID[elem.ID] = elem;
            }

            var requireAuthenticatedAddonDownloads = bool.Parse(ConfigurationManager.AppSettings["SecureDownloadModule.RequireAuthenticatedAddonDownloads"]);

            if (requireAuthenticatedAddonDownloads)
            {
                context.BeginRequest += AuthenticateDownload;
            }

            // Configuration of enabling Throttling is done inside the event handler after retrieving custom config section
            context.BeginRequest += ConfigureThrottling;
        }

        public void Dispose()
        {
            _currentApplication = null;
        }

        #endregion
    }
}
