﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using System.Runtime.InteropServices;

namespace Curse.DownloadSecurity.Throttling
{    
    public class ThrottlingFactory
    {        
        #region Singleton Implementation

        private static readonly ThrottlingFactory _instance = new ThrottlingFactory();
        private Dictionary<string, ThrottlingSettings> _throttleRateCache = new Dictionary<string, ThrottlingSettings>();

        private ThrottlingFactory() { }
        
        public static ThrottlingFactory Instance
        {
            get
            {
                return _instance;
            }
        }

        #endregion

        

        /// <summary>
        /// Create a throttling settings object that tries to clamp the download duration to a specified time, within 
        /// the minimum and maximum rate bounds
        /// </summary>
        /// <param name="pFilePath">The location of the file on disk</param>
        /// <param name="pMinRate">
        ///     The minimum allowed rate in kbps.  If the rate would be lower in order to transmit the specified file in the specified
        ///     duration, this rate is used instead.  A value of 0 means no minimum will be used.
        /// </param>
        /// <param name="pMaxRate">
        ///     The minimum allowed rate in kbps.  If the rate would be lower in order to transmit the specified file in the specified
        ///     duration, this rate is used instead.  A value of 0 means no minimum will be used.
        /// </param>
        /// <param name="pDuration">The intended duration of the download, in seconds</param>
        /// <returns></returns>
        public ThrottlingSettings CreateThrottlingSettings(string filePath, int configID, long minRate, long maxRate, int duration)
        {
           
            ThrottlingSettings settings = null;
            string cacheKey = filePath.ToUpperInvariant() + "-" + configID.ToString();

            if (_throttleRateCache.TryGetValue(cacheKey, out settings))
            {                
                return settings;
            }

            if (File.Exists(filePath))
            {
                
                FileInfo info = new FileInfo(filePath);

                long fileLength = info.Length;

                long kilobits = (fileLength * 8) / 1024;

                long rate = kilobits;

                if (duration > 0)
                {
                    rate /= duration;
                }

                if (minRate > 0 && rate < minRate)
                {
                    rate = minRate;
                }

                if (maxRate > 0 && rate > maxRate)
                {
                    rate = maxRate;
                }                

                settings = new ThrottlingSettings(rate);
            }
            else
            {
                settings = new ThrottlingSettings();
            }            

            _throttleRateCache[cacheKey] = settings;

            return settings;
                            
        }
    }
}
