﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

namespace Curse.DownloadSecurity.Throttling
{
    /// <summary>
    /// Holds the BitRate Throttling settings that control the rate at which downloads are sent from the media server
    /// Corresponds to the iis response server variables.
    /// </summary>
    public class ThrottlingSettings
    {

        #region Constructors

        /// <summary>
        /// Default constructor disables throttling
        /// </summary>
        public ThrottlingSettings()
        {
            this.ThrottlerEnabled = false;
        }

        /// <summary>
        /// Set normal bitrate throttling settings
        /// </summary>
        /// <param name="pRate">The throttle rate, specified in kilobits per second (kbps)</param>
        public ThrottlingSettings(long pRate)
        {
            this.ThrottlerEnabled = true;
            this.Rate = pRate;
        }

        /// <summary>
        /// Sets settings mandatory for media streaming throttling
        /// </summary>
        /// <param name="pFileExtension">The file name extension for the request (for example, wmv)</param>
        /// <param name="pEncodedBitRateKbps">The encoded bit rate for the content being served in the response</param>
        public ThrottlingSettings(string pFileExtension, long pEncodedBitRateKbps)
        {
            this.ThrottlerEnabled = true;
            this.FileExtension = pFileExtension;
            this.EncodedBitRateKbps = pEncodedBitRateKbps;
        }

        #endregion

        #region Properties

        /// <summary>
        /// Enables/Disables throttling for the particular request.
        /// </summary>
        public bool ThrottlerEnabled { get; set; }

        /// <summary>
        /// The initial send size, specified in kilobytes (KB), that is initially sent to the client at an unthrottled rate to provide the Fast Start experience. This can be used together with Rate.
        /// </summary>
        public long? InitialSendSize { get; set; }

        /// <summary>
        /// The throttle rate, specified in kilobits per second (kbps), at which the response will be throttled after any specified Fast Start burst is delivered. This may be used together with InitialSendSize.
        /// </summary>
        public long? Rate { get; set; }

        /// <summary>
        /// A string that specifies the file name extension for the request (for example, wmv). This must be specified along with EncodedBitRateKbps. This is applicable for media responses only. 
        /// </summary>
        public string FileExtension { get; set; }

        /// <summary>
        /// The encoded bit rate for the content being served in the response (for example, 1000). This must be specified along with FileExtension. This is applicable to media responses only.
        /// </summary>
        public long? EncodedBitRateKbps { get; set; }

        /// <summary>
        /// Specifies the throttle rate, in percent. to be applied to the request (for example, 110). This is applicable for media responses only. This may be used together with InitialBufferTimeSeconds.
        /// </summary>
        public int? ThrottleRatePercentage { get; set; }

        /// <summary>
        /// Specifies the initial buffer time during which the response in unthrottled for fast startup (for example, 20). This is applicable for media responses only. This may be used together with ThrottleRatePercentage.
        /// </summary>
        public int? InitialBufferTimeSeconds { get; set; }

        #endregion
    }
}
