﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using Microsoft.Build.Framework;
using System.Text.RegularExpressions;

namespace Curse.MSBuild.Packer
{
    public static class FileProcessor
    {
        public static void Process(string outputFile, IEnumerable<string> inputFiles, IEnumerable<KeyValuePair<string,string>> substitutions, Func<string, string> procDelegate, bool verbose, string absolutePathBase)
        {

            if (File.Exists(outputFile)) {
                File.Delete(outputFile);
            }

            foreach (string inputFile in inputFiles)
            {

                if (verbose) {
                    LogMessage("Processing '{0}'\n", inputFile);
                }
                if (File.Exists(inputFile)) {
                    FileInfo info = new FileInfo(inputFile);
                    StreamReader reader = info.OpenText();
                    string fileContents = reader.ReadToEnd();

                    if (substitutions != null)
                    {
                        foreach (KeyValuePair<string, string> sub in substitutions)
                        {
                            if (verbose)
                            {
                                LogMessage("\tSubstituting {0} -> {1}\n", sub.Key, sub.Value);
                            }
                            fileContents = fileContents.Replace(sub.Key, sub.Value);
                        }
                    }
                    if (!string.IsNullOrEmpty(absolutePathBase) && outputFile.EndsWith("css"))
                    {
                        // Match only relative URLs
                        Regex matcher = new Regex(@"url\(['""]?(\.\..+?)['""]?\)");
                        fileContents = matcher.Replace(fileContents, delegate(Match match)
                            {
                                string origUrl = match.Groups[1].ToString();
                                Uri cssUri = new Uri(inputFile);
                                Uri linkedUri = new Uri(cssUri, origUrl);
                                // We only want to change files under the /Content path
                                string contentBaseDir = absolutePathBase;
                                int contentIndex = linkedUri.AbsolutePath.IndexOf(contentBaseDir) + contentBaseDir.Length;
                                if (contentIndex <= 0) {
                                    return WrapUrl(origUrl);
                                }
                                string relPath = linkedUri.AbsolutePath.Substring(contentIndex);
                                return WrapUrl(relPath);
                            });
                    }
                    if (procDelegate != null)
                    {
                        fileContents = procDelegate(fileContents);
                    }
                    fileContents = FileHeader(inputFile) + fileContents + "\n\n";
                    if (verbose)
                    {
                        LogMessage("Appending {0} to {1}\n", inputFile, outputFile);
                    }
                    File.AppendAllText(outputFile, fileContents);
                } else {
                    LogMessage("FILE NOT FOUND: '{0}'\n", inputFile);
                }

            }
        }
        private static string WrapUrl(string url)
        {
            return string.Format("url({0})", url);
        }
        private static string FileHeader(string fileName)
        {
            return string.Format("/* {0} */\n", fileName);
        }

        public static void LogMessage(string msg) {
            Console.WriteLine(msg);
        }

        public static void LogMessage(string format, params object[] args) {
            Console.WriteLine(format, args);
        }
 
    }
}
