﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Configuration;
using System.Net.Mail;
using System.Threading;
using System.Net;

using Curse;
using Curse.Extensions;
using System.IO;
using System.Net.Mime;
using System.Xml.Serialization;

namespace Curse.Mailer
{
    public class MailProvider
    {

        private bool _isAlive = true;

        // Mail Queue
        private object _mailQueueLock = new object();
        private List<MailMessage> _mailQueue;
        private Thread _messageThread;
        
        private MailAddress _defaultSender = new MailAddress("notification-noreply@curse.com", "Curse");

        // Templating        
        private Dictionary<string, string> _templates;
        private Thread _templateThread;
        private ContentType _htmlMimeType = new ContentType("text/html");
        private Encoding _mailEncoding = Encoding.GetEncoding("utf-8");

        // Smtp Providers                            
        private Dictionary<int, SmtpProviderConfigElement> _variableSmtpProviders;
        private SmtpProviderConfigElement _defaultSmtpProvider;
        private Dictionary<string, int> _domainToProviderMap;
        private LogDelegate Log = null;       

        #region Properties 

        private MailStatistics _statistics;
        public MailStatistics Statistics 
        {
            get
            {
                return _statistics;
            }
        }

        public IEnumerable<SmtpProviderConfigElement> ActiveSmtpProviders
        {
            get
            {
                foreach (SmtpProviderConfigElement config in MailProviderConfiguration.SmtpProviders)
                {
                    yield return config;
                }
            }
        }

        #endregion

        #region Singleton Implementation

        private static readonly MailProvider _instance = new MailProvider();

        public static MailProvider Instance
        {
            get
            {
                return _instance;
            }
        }

        #endregion
       
        public MailProvider()
        {
            Log = MailProviderConfiguration.LogDelegate;

            Log(LogLevel.Info, "Mail Provider Initializing...");

            MailProviderConfiguration.Instance.DynamicTemplatePath = Path.Combine(AppDomain.CurrentDomain.BaseDirectory,  @"bin\EmailTemplates");
            
            if (MailProviderConfiguration.HasTemplateConfiguration) {
                _templates = new Dictionary<string, string>();
                CacheTemplates();

                _templateThread = new Thread(TemplateThread) { IsBackground = true };
                _templateThread.Start();
            }
            else {
                _templates = MailProviderConfiguration.Templates;
            }

            // Process SMTP Providers
            ProcessSmtpProviders();
            
            _mailQueue = new List<MailMessage>();                        

            _messageThread = new Thread(MessageThread) { IsBackground = true };
            _messageThread.Start();

            _statistics = new MailStatistics();
            _statistics.ServerIdentity = System.Environment.MachineName;
            _statistics.DateStarted = DateTime.UtcNow;

            Log(LogLevel.Info, "Mail Provider Initialization Successful!");

        }

        private SmtpProviderConfigElement GetSmtpClientForMessage(MailMessage message)
        {

            if (message.To.Count == 0)
            {
                throw new ArgumentException("MailProvider - MailMessage has no recipient defined!");
            }

            MailAddress recipient = message.To[0];
            string domain = recipient.Address.Substring(recipient.Address.IndexOf("@") + 1).ToLower();
            
            int providerID = 0;

            if (!_domainToProviderMap.TryGetValue(domain, out providerID))
            {
                foreach (KeyValuePair<int, SmtpProviderConfigElement> kvp in _variableSmtpProviders)
                {                    
                    if (kvp.Value.DomainSelectorRegex.IsMatch(domain))
                    {
                        providerID = kvp.Value.ID;
                        break;
                    }
                }
                _domainToProviderMap[domain] = providerID;

            }

            if (providerID > 0)
            {
                return _variableSmtpProviders[providerID];
            }
            else
            {
                return _defaultSmtpProvider;
            }            
        }               

        private void ProcessSmtpProviders()
        {
            Dictionary<int, SmtpProviderConfigElement> variableSmtpProviders = new Dictionary<int, SmtpProviderConfigElement>();
            SmtpProviderConfigElement defaultSmtpProvider = null;
            
            // Only load enabled providers
            foreach (SmtpProviderConfigElement provider in MailProviderConfiguration.SmtpProviders)
            {                
                if (!provider.IsEnabled)
                {
                    continue;
                }

                if (variableSmtpProviders.ContainsKey(provider.ID))
                {
                    Log(LogLevel.Error, "MailProvider - Unable to use SMTP provider: '{0}'. The provider ID '{1}' is defined more than once.", provider.Host, provider.ID.ToString());
                }
                else if (provider.IsDefault)
                {
                    defaultSmtpProvider = provider;
                }
                else
                {
                    variableSmtpProviders.Add(provider.ID, provider);
                }
            }

            if (defaultSmtpProvider == null)
            {
                Log(LogLevel.Error, "MailProvider - No default provider has been defined!");
                return;
            }

            _defaultSmtpProvider = defaultSmtpProvider;
            _variableSmtpProviders = variableSmtpProviders;
            _domainToProviderMap = new Dictionary<string, int>();
        }        
        
        private void CacheTemplates()
        {
            if (MailProviderConfiguration.TemplatePath == null)
            {
                Log(LogLevel.Error, "MailProvider - The template folder has not been set. No templated mails will be supported.");
                return;
            }

            DirectoryInfo directoryInfo = new DirectoryInfo(MailProviderConfiguration.TemplatePath);
            if (!directoryInfo.Exists) {
                directoryInfo.Create();
            }
            
            FileInfo[] templateFiles = directoryInfo.GetFiles();

            foreach (FileInfo file in templateFiles)
            {
                using (StreamReader reader = new StreamReader(file.FullName))
                {
                    string contents = reader.ReadToEnd();
                    _templates[file.Name] = contents;
                }
            }
        }

        private void SendMessage(MailMessage message)
        {            
#if _DEBUG
            message.To.Clear();
            message.To.Add(new MailAddress("mcomperda@hotmail.com"));
#endif
            SmtpProviderConfigElement provider = GetSmtpClientForMessage(message);
            SmtpClient client = provider.Client;


            if (client == null)
            {
                Log(LogLevel.Error, "MailProvider - Unable to find a suitable SMTP provider for the following recipient: '{0}'", message.To[0].Address);
                return;
            }
            
            for (int i = 0; i < 5; i++)
            {
                try
                {                    
                    client.Send(message);
                    _statistics.DateLastSend = DateTime.UtcNow;
                    ++_statistics.TotalSendSucesses;
                    ++_statistics.TotalSends;
                    ++provider.TotalSendSuccesses;
                    break;
                }
                catch(Exception ex)
                {
                    if (i == 4)
                    {
                        ++_statistics.TotalSendFailures;
                        ++_statistics.TotalSends;
                        ++provider.TotalSendFailures;
                        _statistics.DateLastFailure = DateTime.UtcNow;
                        _statistics.LastFailureMessage = ex.Message;
                        throw ex;
                    }
                    else
                    {

                        Thread.Sleep(1000);
                    }
                }
            }
        }

        private MailAddress GetMailAddressFromString(string address)
        {
            try
            {
                return new MailAddress(address);
            }
            catch
            {
                Log(LogLevel.Error, "MailProvider - Unable to parse e-mail address from: '{0}'", address);
                return null;
            }
        }

        public void SendFormattedMessage(string senderAddress, string recipientAddress, string subject, string plainTextBody, string htmlBody)
        {           
            MailAddress recipient = GetMailAddressFromString(recipientAddress);
            if (recipient == null)
            {
                return;
            }
            SendFormattedMessage(new MailAddress(senderAddress), recipient, subject, plainTextBody, htmlBody);
        }

        public void SendFormattedMessage(MailAddress sender, string recipientAddress, string subject, string plainTextBody, string htmlBody)
        {
            MailAddress recipient = GetMailAddressFromString(recipientAddress);
            if (recipient == null)
            {
                return;
            }
            SendFormattedMessage(sender, recipient, subject, plainTextBody, htmlBody);
        }

        public void SendFormattedMessage(MailAddress senderAddress, MailAddress recipientAddress, string subject, string plainTextBody, string htmlBody)
        {
            if (!htmlBody.ToLower().StartsWith("<doctype>") && !htmlBody.ToLower().StartsWith("<html>"))
            {
                htmlBody = "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.0 Transitional//EN\"><HTML><HEAD><META http-equiv=Content-Type content=\"text/html; charset=iso-8859-1\"></HEAD><BODY>{0}</BODY></HTML>".FormatWith(htmlBody);
            }

            if (senderAddress == null)
            {
                senderAddress = _defaultSender;
            }

            MailMessage message = new MailMessage();
            message.From = senderAddress;
            message.To.Add(recipientAddress);
            message.Subject = subject;            
            message.BodyEncoding = _mailEncoding;

            AlternateView plainView = AlternateView.CreateAlternateViewFromString(plainTextBody, _mailEncoding, "text/plain");
            message.AlternateViews.Add(plainView);

            AlternateView htmlView = AlternateView.CreateAlternateViewFromString(htmlBody, _mailEncoding, "text/html");
            message.AlternateViews.Add(htmlView);                                    

            MailProvider.Instance.QueueMessage(message);
        }

        public void SendTemplatedMessage(string templateName, MailAddress sender, string recipientAddress, string subject, string[] templateValues)
        {
            MailAddress recipient = GetMailAddressFromString(recipientAddress);
            if (recipient == null)
            {
                return;
            }
            SendTemplatedMessage(templateName, sender, recipient, subject, templateValues);
        }

        public void SendTemplatedMessage(string templateName, string senderAddress, string recipientAddress, string subject, string[] templateValues)
        {
            MailAddress recipient = GetMailAddressFromString(recipientAddress);
            if (recipient == null)
            {
                return;
            }
            SendTemplatedMessage(templateName, new MailAddress(senderAddress), recipient, subject, templateValues);
        }

        public void SendTemplatedMessage(string templateName, MailAddress senderAddress, MailAddress recipient, string subject, string[] templateValues)
        {
            if (MailProviderConfiguration.TemplatePath == null)
            {
                throw new Exception("MailProvider - Attempted to send a templated message, when no valid template folder exists.");                
            }

            try
            {
                string plainTextBody = _templates[templateName + ".txt"].FormatWith(templateValues);
                string htmlBody = _templates[templateName + ".html"].FormatWith(templateValues);

                SendFormattedMessage(senderAddress, recipient, subject, plainTextBody, htmlBody);
            }
            catch(Exception ex)
            {
                Log(LogLevel.Error, "MailProvider - SendTemplatedMessage Exception! Details: {0}", ex.GetExceptionDetails());
            }
        }

        public void QueueMessage(MailMessage message)
        {
            lock (_mailQueueLock)
            {
                _mailQueue.Add(message);
            }            
        }
        
        public void Shutdown()
        {
            _isAlive = false;
        }

        #region Threads

        private void TemplateThread()
        {
            while (_isAlive)
            {
                Thread.Sleep(10000);
                try
                {
                    CacheTemplates();
                }
                catch (Exception ex)
                {
                    Log(LogLevel.Error, "MailProvider TemplateThread Exception! Details: {0}", ex.GetExceptionDetails());   
                }
            }
        }        

        private void MessageThread()
        {
            while (_isAlive)
            {
                Thread.Sleep(2000);
                try
                {
                    this.ProcessQueue();
                }
                catch (Exception ex)
                {
                    Log(LogLevel.Error, "MailProvider Messsage Thread Exception! Details: {0}", ex.GetExceptionDetails());
                }
            }
        }

        #endregion

        private void ProcessQueue()
        {
            List<MailMessage> messages = new List<MailMessage>();

            lock (_mailQueueLock)
            {
                if (_mailQueue.Count == 0)
                {
                    return;
                }

                messages.AddRange(_mailQueue);
                _mailQueue.Clear();
            }

            foreach (MailMessage message in messages)
            {
                try
                {
                    SendMessage(message);
                }
                catch(Exception ex)
                {
                    Log(LogLevel.Error, "MailProvider - Send message failed! Details: {0}", ex.GetExceptionDetails());   
                }
            }
        }

        public void Initialize()
        {
        } 

    }
}
