﻿using System;
using System.Linq;
using System.Threading;
using System.Collections.Generic;
using Curse.Logging;
using Curse.ServiceModels.Configuration;
using Curse.ServiceModels.Authentication;
using System.ServiceModel.Web;
using System.ServiceModel;
using System.ServiceModel.Channels;
using System.Web;
using Curse.ServiceModels.AuthService;
using System.Web.Caching;
using Curse.Extensions;
using System.Data.SqlClient;

namespace Curse.ServiceModels.Authentication
{
    public static class ServiceAuthentication
    {
        #region Variable Declarations
        static readonly AuthenticationType _authenticationType;
        
        // Auth Service
        static readonly string _authServiceUrl;
        static readonly int _siteID;
        static readonly StringCipher _outgoingCipher;
        static readonly StringCipher _incomingCipher;
        static readonly TimeSpan _authSessionLifeSpan;
        static readonly int _authServiceTimeout;
        static readonly bool _freePremium = false;
        
        // Api
        static readonly bool _apiPerUser = false;
        static readonly string _apiDataSource;
        static readonly string _apiCatalog; 
        static readonly string _apiKey;

        static object _loginsLock;
        static Dictionary<string, DateTime> _logins;
        
        static object _sessionLock;
        static Dictionary<string, ServiceAuthenticationSession> _sessions;

        private const string _incomingCipherKey = "31ED549B43318C9CFAE926ADE1720D2202C2DEFFAB06AE9D78FB6F16DF2D70A4";

        static Thread _cleanupThread;

        private const int SubscriptionSeedPremium = 181756351;
        private const int SubscriptionSeedNonPremium = 76172335;
        #endregion

        #region Properties
        public static string ApiKey
        {
            get
            {
                return _apiKey;
            }
        }
        private static string ApiConnectionString
        {
            get
            {
                var baseString = "Data Source={0};Initial Catalog={1};Persist Security Info=True;Integrated Security=True;MultipleActiveResultSets=true;";
                return string.Format(baseString, _apiDataSource, _apiCatalog);
            }
        }
        public static ServiceAuthenticationToken CurrentAuthToken
        {
            get
            {
                var webContext = WebOperationContext.Current;
                var operationContext = OperationContext.Current;

                if (webContext != null && operationContext != null)
                {
                    MessageVersion messageVersion = operationContext.IncomingMessageVersion;

                    if (messageVersion == MessageVersion.None) // Soap 11 Version - Check standard Http headers
                    {
                        string username = webContext.IncomingRequest.Headers["x-curse-username"];
                        string password = webContext.IncomingRequest.Headers["x-curse-password"];
                        if (username != null && password != null)
                        {
                            return new ServiceAuthenticationToken(username, password);
                        }
                    }
                    else if (operationContext.IncomingMessageHeaders.FindHeader(ServiceAuthenticationToken.HeaderName, ServiceAuthenticationToken.HeaderNamespace) != -1) // Soap 12 Version - Check standard Soap headers
                    {
                        return operationContext.IncomingMessageHeaders.GetHeader<ServiceAuthenticationToken>(ServiceAuthenticationToken.HeaderName, ServiceAuthenticationToken.HeaderNamespace);
                    }
                }
                return new ServiceAuthenticationToken { IsAnonymous = true };
            }
        }
        public static ServiceAuthenticationSession CurrentSession
        {
            get
            {
                return GetSessionByUsername(CurrentAuthToken.Username, false);
            }
        }
        public static User CurrentUserProfile
        {
            get
            {
                return GetUserProfile(CurrentSession.UserID);
            }
        }
        #endregion

        static ServiceAuthentication()
        {
            _authenticationType = ServiceConfiguration.Instance.Authentication.AuthenticationType;
            switch (_authenticationType)
            {
                case AuthenticationType.Service:
                    {
                        _authServiceUrl = ServiceConfiguration.Instance.Authentication.ServiceAuthentication.AuthServiceUrl;
                        _siteID = ServiceConfiguration.Instance.Authentication.ServiceAuthentication.AuthSiteID;
                        _incomingCipher = new StringCipher(_incomingCipherKey);

                        var outgoingCypher = ServiceConfiguration.Instance.Authentication.ServiceAuthentication.AuthSiteKey;
                        if (!string.IsNullOrEmpty(outgoingCypher))
                        {
                            _outgoingCipher = new StringCipher(outgoingCypher);
                        }

                        _authSessionLifeSpan = ServiceConfiguration.Instance.Authentication.ServiceAuthentication.AuthSessionLifespan;
                        _authServiceTimeout = ServiceConfiguration.Instance.Authentication.ServiceAuthentication.AuthServiceTimeout;
                        _freePremium = ServiceConfiguration.Instance.Authentication.ServiceAuthentication.FreePremium;
                    } break;
                case AuthenticationType.ApiKey:
                    {
                        _apiPerUser = ServiceConfiguration.Instance.Authentication.ApiAuthentication.PerUser;
                        _apiDataSource = ServiceConfiguration.Instance.Authentication.ApiAuthentication.PerUserDataSource;
                        _apiCatalog = ServiceConfiguration.Instance.Authentication.ApiAuthentication.PerUserCatalog;
                        _apiKey = ServiceConfiguration.Instance.Authentication.ApiAuthentication.ApiKey;
                    } break;
            }           

            _loginsLock = new object();
            _logins = new Dictionary<string, DateTime>();
            
            _sessionLock = new object();
            _sessions = new Dictionary<string, ServiceAuthenticationSession>();

            // Session Cleanup
            _cleanupThread = new Thread(CleanupSessionThread)
            {
                IsBackground = true
            };
            _cleanupThread.Start();
        }

        #region Session Cleanup
        static void CleanupSessionThread()
        {
            while (true)
            {
                Thread.Sleep(5000);
                try
                {
                    CleanupSessions();
                }
                catch (Exception exc)
                {
                    Logger.Error(exc, "Unable to cleanup sessions!");
                }
            }
        }
        static void CleanupSessions()
        {
            DateTime expiredSessionDate = DateTime.UtcNow.AddMinutes(-20);

            Dictionary<string, DateTime> loginsCopy;

            lock (_loginsLock)
            {
                loginsCopy = new Dictionary<string, DateTime>(_logins);
            }

            string[] expiredKeys = loginsCopy.Where(p => p.Value < expiredSessionDate).Select(p => p.Key).ToArray();

            if (expiredKeys.Length == 0)
            {
                return;
            }

            foreach (string expiredKey in expiredKeys)
            {
                loginsCopy.Remove(expiredKey);
            }

            lock (_loginsLock)
            {
                _logins = loginsCopy;
            }
        }
        #endregion

        #region Private Methods
        static ServiceAuthenticationSession GetSession(string username)
        {
            ServiceAuthenticationSession session;

            lock (_sessionLock)
            {
                _sessions.TryGetValue(username, out session);
            }

            return session;
        }
        static string GetUserProfileCacheKey(int userID)
        {
            return "UserProfile-" + userID.ToString();
        }
        static User GetUserProfile(int userID)
        {
            string cacheKey = GetUserProfileCacheKey(userID);
            User profile = HttpRuntime.Cache.Get(cacheKey) as User;

            if (profile == null)
            {
                using (NetworkService authService = new NetworkService())
                {
                    profile = authService.v2GetUserProfile(_siteID, userID);
                }

                HttpRuntime.Cache.Insert(cacheKey, profile, null, Cache.NoAbsoluteExpiration, TimeSpan.FromMinutes(10));
            }
            return profile;
        }
        static bool GetEffectivePremiumStatus(bool hasPremium)
        {
            if (_freePremium || hasPremium)
            {
                return true;
            }
            else
            {
                return false;

            }
        }
        static string GetLoginToken(string username, string password)
        {
            return username.ToLowerInvariant() + "-" + password;
        }
        static void CacheLogin(string username, string encryptedPassword)
        {
            string token = GetLoginToken(username, encryptedPassword);

            lock (_loginsLock)
            {
                if (!_logins.ContainsKey(token))
                {
                    _logins.Add(token, DateTime.UtcNow);
                }
            }
        }
        static void CacheSession(string username, ServiceAuthenticationSession session)
        {
            lock (_sessionLock)
            {
                _sessions[username] = session;
            }
        }
        static bool LoginExists(string username, string plainTextPassword)
        {
            string token = GetLoginToken(username, plainTextPassword);

            lock (_loginsLock)
            {
                return _logins.ContainsKey(token);
            }
        }
        static AuthenticationStatus ToAuthenticationStatus(ELoginStatus status)
        {
            switch (status)
            {
                case ELoginStatus.InvalidPassword:
                    return AuthenticationStatus.InvalidPassword;
                case ELoginStatus.InvalidSession:
                    return AuthenticationStatus.InvalidSession;
                case ELoginStatus.Success:
                    return AuthenticationStatus.Success;
                case ELoginStatus.UnauthorizedLogin:
                    return AuthenticationStatus.UnauthorizedLogin;
                case ELoginStatus.UnknownEmail:
                    return AuthenticationStatus.UnknownEmail;
                case ELoginStatus.UnknownUsername:
                    return AuthenticationStatus.UnknownUsername;
                case ELoginStatus.Unsuccessful:
                default:
                    return AuthenticationStatus.Unsuccessful;
            }
        }
        #endregion

        #region Public Methods
        public static int GetSubscriptionToken(int userID, bool hasPremium)
        {
            return userID ^ (hasPremium ? SubscriptionSeedPremium : SubscriptionSeedNonPremium);
        }        
        public static ServiceAuthenticationSession GetSessionByUsername(string username, bool refreshSubscriptionStatus)
        {
            ServiceAuthenticationSession session = GetSession(username);

            if (session == null)
            {
                Logger.Debug("Unable to find session for username:", username);
                return null;
            }

            if (refreshSubscriptionStatus && session.IsExpired)
            {
                try
                {
                    bool hasPremium = session.ActualPremiumStatus;

                    using (NetworkService authService = new NetworkService())
                    {
                        hasPremium = authService.v2GetSubscriptionStatus(_siteID, session.UserID, ESubscriptionType.Premium);
                    }

                    session.ActualPremiumStatus = hasPremium;
                    session.EffectivePremiumStatus = GetEffectivePremiumStatus(hasPremium);
                    session.SubscriptionToken = GetSubscriptionToken(session.UserID, hasPremium);
                    session.DateRefreshed = DateTime.UtcNow;
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Unable to refresh user's premium status.");
                }
            }
            return session;
        }
        public static void RecordUserActivity(string username)
        {
            ServiceAuthenticationSession session = GetSessionByUsername(username, false);

            if (session != null)
            {
                session.RecordActivity();
            }
        }
        public static AuthenticationStatus AuthenticateApiUser(string apiKey, int? userID = null)
        {
            if (string.IsNullOrEmpty(apiKey)) return AuthenticationStatus.InvalidApiKey;

            if (!_apiPerUser && apiKey == ApiKey)
            {
                return AuthenticationStatus.Success;
            }
            else
            {
                using (var conn = new SqlConnection(ApiConnectionString))
                {
                    SqlCommand cmd;
                    if(userID == null)
                    {
                        cmd = new SqlCommand("SELECT ApiKey, IsBanned FROM ServerModsApiKey WHERE apikey = @apikey", conn);
                        cmd.Parameters.AddWithValue("@apikey", apiKey);    
                    }
                    else
                    {
                        cmd = new SqlCommand("SELECT ApiKey, IsBanned FROM ServerModsApiKey WHERE userID = @userID", conn);
                        cmd.Parameters.AddWithValue("@userID", userID);
                    }

                    try
                    {
                        conn.Open();
                        using (var reader = cmd.ExecuteReader())
                        {
                            if (reader.Read())
                            {
                                var userKey = reader.GetString(reader.GetOrdinal("ApiKey"));
                                var isBanned = reader.GetBoolean(reader.GetOrdinal("IsBanned"));

                                if (isBanned || apiKey != userKey)
                                {
                                    return AuthenticationStatus.InvalidApiKey;
                                }

                                return AuthenticationStatus.Success;
                            }
                            else
                            {
                                return AuthenticationStatus.UnknownUsername;
                            }
                        }
                    }
                    catch (Exception exc)
                    {
                        Logger.Error(exc, "Failed to authenticate ApiKey.");
                        return AuthenticationStatus.UnknownError;
                    }
                }
            }
        }
        public static AuthenticationStatus AuthenticateUser(string username, string encryptedPassword)
        {
            string planTextPassword = _incomingCipher.Decrypt(encryptedPassword);

            LoginResult result = null;

            if (LoginExists(username, planTextPassword))
            {
                RecordUserActivity(username);
                return AuthenticationStatus.Success;
            }

            bool actualPremiumStatus = false;
            using (NetworkService authService = new NetworkService())
            {

                try
                {
                    result = authService.v2ValidateClientUser(_siteID, username, _outgoingCipher.Encrypt(planTextPassword));

                    if (result.Status == ELoginStatus.Success)
                    {
                        actualPremiumStatus = authService.v2GetSubscriptionStatus(_siteID, result.UserId, ESubscriptionType.Premium);
                    }
                }
                catch (Exception ex)
                {
                    Logger.Error(ex, "Exception occurred while validating user!");
                    Logger.Error("Username:", username);
                    return AuthenticationStatus.UnknownError;
                }
            }

            if (result.Status == ELoginStatus.Success)
            {
                bool effectivePremiumStatus = GetEffectivePremiumStatus(actualPremiumStatus);

                ServiceAuthenticationSession session = new ServiceAuthenticationSession()
                {
                    UserID = result.UserId,
                    SessionID = result.SessionId,
                    ActualPremiumStatus = actualPremiumStatus,
                    EffectivePremiumStatus = effectivePremiumStatus,
                    SubscriptionToken = GetSubscriptionToken(result.UserId, effectivePremiumStatus),
                    DateLastActive = DateTime.UtcNow
                };

                CacheLogin(username, planTextPassword);
                CacheSession(username, session);

                return AuthenticationStatus.Success;
            }
            else
            {
                Logger.Debug("Unable to authenticate user with username:", username);
                return ToAuthenticationStatus(result.Status);
            }
        }
        public static StatusCode ValidateSessionKeyToUserId(string sessionId, out int userId)
        {
            userId = 0;

            // try to get from the sessions first
            lock (_sessions)
            {
                ServiceAuthenticationSession session;
                if (_sessions.TryGetValue(sessionId, out session))
                {
                    if (session.IsExpired)
                    {
                        _sessions.Remove(sessionId);
                        return StatusCode.AuthenticationInvalidSession;
                    }
                    else
                    {
                        userId = session.UserID;
                        session.DateRefreshed = DateTime.UtcNow;
                        return StatusCode.Ok;
                    }
                }
                else
                {
                    using (var authService = new NetworkService())
                    {
                        userId = authService.v2ValidateUserSession(_siteID, sessionId);
                        if (userId <= 0)
                        {
                            return StatusCode.AuthenticationInvalidSession;
                        }

                        bool actualPremiumStatus = authService.v2GetSubscriptionStatus(_siteID, userId, ESubscriptionType.Premium);
                        bool effectivePremiumStatus = GetEffectivePremiumStatus(actualPremiumStatus);
                        session = new ServiceAuthenticationSession()
                        {
                            UserID = userId,
                            SessionID = sessionId,
                            ActualPremiumStatus = actualPremiumStatus,
                            EffectivePremiumStatus = effectivePremiumStatus,
                            SubscriptionToken = GetSubscriptionToken(userId, effectivePremiumStatus),
                            DateLastActive = DateTime.UtcNow
                        };
                        _sessions.Add(sessionId, session);

                        return StatusCode.Ok;
                    }
                }
            }
        }
        #endregion
    }
}