﻿using System;
using System.Collections.Generic;
using System.Linq;
using Curse.Extensions;
using Curse.Logging;
using Curse.ServiceModels.Models;
using Curse.ServiceModels.Configuration;
using System.Data.SqlClient;

namespace Curse.ServiceModels.Caching
{
    public abstract class IncrementalCache<T>
    {
        WorkerThread<bool> _updateThread;
        
        List<ICachable> _currentResults;
        public List<T> CacheData
        {
            get { return _currentResults.Select(p => (T)p).ToList(); }
            set { _currentResults = value.Select(p => (ICachable)p).ToList(); } 
        }
                
        DateTime _lastQueryTime = new DateTime(1900, 1, 1);
        public DateTime LastQueryTime
        {
            get
            {
                return _lastQueryTime.AddMinutes(-5);
            }
            set
            {
                _lastQueryTime = value;
            }
        }
        public SqlParameter LastQueryParameter
        {
            get
            {
                var param = new SqlParameter("@LastQueryTime", System.Data.SqlDbType.DateTime);
                param.Value = LastQueryTime;
                return param;
            }
        }
                
        double _updateInterval;
        public double UpdateInterval 
        { 
            get { return _updateInterval; } 
            set { _updateInterval = value; } 
        }

        string _dataSourceName;
        string _updateProcedureName;

        object _cacheLock = new object();
        public object CacheLock { get { return _cacheLock; } }

        public IncrementalCache(string dataSource, string updateProcedureName) 
        {
            var config = ServiceConfiguration.Instance.Caching;
            _updateInterval = config.UpdateThreadInterval;
            _dataSourceName = dataSource;
            _updateProcedureName = updateProcedureName;
            _currentResults = new List<ICachable>();

            try
            {
                UpdateCache();
            }
            catch (Exception exc)
            {
                Logger.Error(exc, "UpdateCache Exception. Details: {0}");
            }

            _updateThread = new WorkerThread<bool>(() => UpdateCache(), TimeSpan.FromSeconds(_updateInterval), true);
        }
        public void Initialize() { }

        public virtual bool UpdateCache()
        {
            var results = new List<ICachable>(_currentResults);
            
            using (var conn = DatabaseConfiguration.Instance[_dataSourceName].GetDatabaseConnection())
            {
                var cmd = new SqlCommand(_updateProcedureName, conn);
                cmd.CommandType = System.Data.CommandType.StoredProcedure;
                cmd.Parameters.Add(LastQueryParameter);

                using (var reader = cmd.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        var cacheObject = Activator.CreateInstance<T>() as ICachable;
                        cacheObject.SetFromDataReader(reader);

                        var existingObject = _currentResults.FirstOrDefault(p => p.ID == cacheObject.ID);
                        if (existingObject != null)
                        {
                            results.Remove(existingObject);
                        }
                        results.Add(cacheObject);
                    }
                }
            }

            lock (CacheLock)
            {
                _currentResults = results;
            }

            return true;
        }
    }
}
