﻿using System;
using System.Threading;
using Curse.Logging;

namespace Curse.ServiceModels.Models
{
    public enum WorkerThreadStatus
    {
        Started,
        Stopped,
        Aborted
    }
    public class WorkerThreadEventArgs<T> : EventArgs
    {
        T _rvalue;
        public T Value { get { return _rvalue; } set { _rvalue = value; } }
        public WorkerThreadEventArgs() { }
        public WorkerThreadEventArgs(T rvalue)
        {
            _rvalue = rvalue;
        }
    }
    public class WorkerThread<T>
    {
        Thread _workerThread;
        Func<T> _action;
        TimeSpan _timeout;

        public event EventHandler<WorkerThreadEventArgs<T>> OnComplete;
        public event EventHandler OnShutDown;

        public WorkerThreadStatus Status = WorkerThreadStatus.Stopped;

        public WorkerThread(Func<T> action) : this(action, false) { }
        public WorkerThread(Func<T> action, bool autoStart)
        {
            _action = action;

            _workerThread = new Thread(DoOnce)
            {
                IsBackground = true
            };

            if (autoStart)
            {
                Start();
            }
        }
        public WorkerThread(Func<T> action, TimeSpan timeout) : this(action, timeout, true) { }
        public WorkerThread(Func<T> action, TimeSpan timeout, bool autoStart)
        {
            _action = action;
            _timeout = timeout;

            _workerThread = new Thread(DoWork)
            {
                IsBackground = true
            };

            if (autoStart)
            {
                Start();
            }
        }

        private void DoWork()
        {
            bool aborted = false;
            while (!aborted)
            {
                Thread.Sleep(_timeout);
                try
                {
                    //Skip if we have intentionally stopped the thread
                    if (Status == WorkerThreadStatus.Stopped) return;

                    var result = _action();
                    if (OnComplete != null)
                    {
                        OnComplete(this, new WorkerThreadEventArgs<T>(result));
                    }
                }
                catch (ThreadAbortException)
                {
                    aborted = true;
                    _workerThread.Join(100);
                    Logger.Info("Thread Abort Exception. Service shutting down.");

                    if (OnShutDown != null)
                    {
                        OnShutDown(this, EventArgs.Empty);
                    }

                    Status = WorkerThreadStatus.Aborted;
                }
                catch (Exception ex)
                {
                    Logger.Info(ex, "Thread Exception:");
                }
            }
        }
        private void DoOnce()
        {
            try
            {
                var result = _action();
                if (OnComplete != null)
                {
                    OnComplete(this, new WorkerThreadEventArgs<T>(result));
                }
            }
            catch (ThreadAbortException)
            {
                _workerThread.Join(100);
                Logger.Info("Thread Abort Exception. Service shutting down.");

                if (OnShutDown != null)
                {
                    OnShutDown(this, EventArgs.Empty);
                }

                Status = WorkerThreadStatus.Aborted;
            }
            catch (Exception ex)
            {
                Logger.Info(ex, "Thread Exception.");
            }

        }

        public void Start()
        {
            Status = WorkerThreadStatus.Started;
            if (_workerThread.ThreadState != ThreadState.WaitSleepJoin || _workerThread.ThreadState != ThreadState.Running)
            {
                _workerThread.Start();
            }
        }
        public void Stop()
        {
            Status = WorkerThreadStatus.Stopped;
        }
    }
}
