﻿using System;
using System.Collections.Generic;
using System.Diagnostics;

namespace Curse
{
    /**
     * Class for reading and writing IPackable elements in a array format with preallocated Capacity
     * Based on inheritance of List<T> and implements IPackable
     * Does not include the known Capacity in the packed data form
     *
     * @author Michael Comperda
     */
    public sealed class PackableArray<T>
        : List<T>, IPackable
        where T : class, IPackable, new()
    {
        /**
         * Initialization constructor
         * Asserts if pSize > UInt16.MaxValue
         * Preallocates the base list capacity, does not allocate elements
         *
         * @param  pSize the number of elements in the array
         */
        public PackableArray(Int32 pSize)
        {
            Debug.Assert(pSize <= UInt16.MaxValue);
            Capacity = pSize;
        }

        /**
         * Reads the array from pBuf based on the pre allocated Capacity
         *
         * @param  pBuf the IReadable buffer in which to read from
         * @return      true if all the elements could be read, false otherwise
         */
        public Boolean Read(IReadable pBuf)
        {
            Boolean ok = true;
            for (Int32 i = 0;
                 ok &&
                 i < Capacity;
                 ++i)
            {
                Add(new T());
                ok = this[i].Read(pBuf);
            }
            return ok;
        }

        /**
         * Writes the array to pBuf based on the pre allocated Capacity
         * Asserts if Capacity > UInt16.MaxValue
         *
         * @param  pBuf the IWriteable buffer in which to write to
         * @return      true if all the elements could be written, false otherwise
         */
        public Boolean Write(IWriteable pBuf)
        {
            Debug.Assert(Capacity <= UInt16.MaxValue);
            return TrueForAll(e => e.Write(pBuf));
        }
    }
}
