﻿using System;
using System.Collections.Generic;
using System.Diagnostics;

namespace Curse
{
    /**
     * Class for reading and writing IPackable elements in a list format
     * Based on inheritance of List<T> and implements IPackable
     * Includes the Count in the packed data form
     *
     * @author Michael Comperda
     */
    public sealed class PackableList<T>
        : List<T>, IPackable
        where T : class, IPackable, new()
    {
        /**
         * Reads the list count and the elements from pBuf
         *
         * @param  pBuf the IReadable buffer in which to read from
         * @return      true if list count and all the elements could be read, false otherwise
         */
        public Boolean Read(IReadable pBuf)
        {
            UInt16 count = 0;
            if (!pBuf.Read(ref count))
            {
                return false;
            }
            Capacity = count;
            Boolean ok = true;
            for (Int16 i = 0;
                 ok &&
                 i < Capacity;
                 ++i)
            {
                Add(new T());
                ok = this[i].Read(pBuf);
            }
            return ok;
        }

        /**
         * Writes the list count and elements to pBuf
         * Asserts if Count > UInt16.MaxValue
         *
         * @param  pBuf the IWriteable buffer in which to write to
         * @return      true if list count and all the elements could be written, false otherwise
         */
        public Boolean Write(IWriteable pBuf)
        {
            Debug.Assert(Count <= UInt16.MaxValue);
            return pBuf.Write(Convert.ToUInt16(Count)) &&
                   TrueForAll(e => e.Write(pBuf));
        }
    }
}
