﻿using System;
using System.Collections.Generic;
using System.Diagnostics;

namespace Curse
{
    /**
     * Class for reading and writing String elements in a list format
     * Based on inheritance of List<String> and implements IPackable
     * Includes the Count in the packed data form
     *
     * @author Michael Comperda
     */
    public sealed class PackableStringList
        : List<String>, IPackable
    {
        /**
         * Reads the list count and the elements from pBuf
         *
         * @param  pBuf the IReadable buffer in which to read from
         * @return      true if list count and all the elements could be read, false otherwise
         */
        public Boolean Read(IReadable stream)
        {
            UInt16 count = 0;
            if (!stream.Read(ref count))
            {
                return false;
            }
            Capacity = count;
            Boolean ok = true;
            String buf = null;
            for (Int32 i = 0;
                 ok &&
                 i < Capacity;
                 ++i)
            {
                if ((ok = stream.Read(ref buf)))
                {
                    Add(buf);
                }
            }
            return ok;
        }

        /**
         * Writes the list count and elements to pBuf
         * Asserts if Count > UInt16.MaxValue
         *
         * @param  pBuf the IWriteable buffer in which to write to
         * @return      true if list count and all the elements could be written, false otherwise
         */
        public Boolean Write(IWriteable stream)
        {
            Debug.Assert(Count <= UInt16.MaxValue);
            return stream.Write(Convert.ToUInt16(Count)) &&
                   TrueForAll(s => stream.Write(s));
        }
    }
}
