﻿using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;
using zlib;

namespace Curse
{
    /**
     * Class for buffering binary data when sending and receiving transmissions
     * Implements IReadable and IWriteable
     *
     * @author Michael Comperda
     */
    public sealed class Package
        : IReadable, IWriteable
    {
        /**
         * Initialization constructor
         * Preallocates internal buffer to INIT size
         *
         * @param  pEncoding the Encoding class to use for string<->byte[] conversions
         */
        public Package(Encoding pEncoding)
        {
            mEncoding = pEncoding;
            mBuffer = new Byte[INIT];
        }

        /**
         * Initialization constructor
         * Allocates and fills internal buffer based on pStream length
         *
         * @param  pEncoding the Encoding class to use for string<->byte[] conversions
         * @param  pStream   the stream class to use for initialization
         */
        public Package(Encoding pEncoding,
                       Stream pStream)
        {
            mEncoding = pEncoding;
            mBuffer = new Byte[pStream.Length];
            mLength = pStream.Read(mBuffer, 0, mBuffer.Length);
        }

        /**
         * Initialization constructor
         * Avoids allocation and uses pData for the internal buffer directly
         *
         * @param  pEncoding the Encoding class to use for string<->byte[] conversions
         * @param  pData     the new array to use for the internal buffer
         */
        public Package(Encoding pEncoding,
                       Byte[] pData)
        {
            mEncoding = pEncoding;
            mBuffer = pData;
            mLength = pData.Length;
        }

        /**
         * Read-only accessor to the internal buffer array
         * Used in combination with Length for writing a complete package to an output stream
         *
         * @return the entire internal buffer
         */
        public byte[] InnerBuffer
        {
            get
            {
                return mBuffer;
            }
        }

        /**
         * Read-only accessor to the current length of data written to this package
         *
         * @return the used portion of the internal buffer
         */
        public int Length
        {
            get
            {
                return mLength;
            }
        }

        /**
         * Read-only accessor to the remaining length of data that has not been read from this package yet
         *
         * @return the remaining portion of readable data from the internal buffer
         */
        public int Available
        {
            get
            {
                return mLength - mCursor;
            }
        }

        /**
         * Primitive Byte array reading method
         *
         * @param  pBuf    an array to read data into
         * @param  pIndex  starting index in pBuf to begin putting data at
         * @param  pLength length of data to read from the package to pBuf
         * @return         true if all data was available to read array, false otherwise
         */
        public Boolean Read(Byte[] pBuf,
                            Int32 pIndex,
                            Int32 pLength)
        {
            if (Available < pLength)
            {
                return false;
            }
            Buffer.BlockCopy(mBuffer,
                             mCursor,
                             pBuf,
                             pIndex,
                             pLength);
            mCursor += pLength;
            return true;
        }

        /**
         * Primitive datatype reading methods
         *
         * @param  pValue a reference to the value which data will be read into
         * @return        true if sufficient data was available to read the primitive type, false otherwise
         */
        public Boolean Read(ref Boolean pValue)
        {
            if (Available < 1)
            {
                return false;
            }
            pValue = mBuffer[mCursor++] != 0;
            return true;
        }

        public Boolean Read(ref Byte pValue)
        {
            if (Available < 1)
            {
                return false;
            }
            pValue = mBuffer[mCursor++];
            return true;
        }

        public Boolean Read(ref SByte pValue)
        {
            if (Available < 1)
            {
                return false;
            }
            pValue = Convert.ToSByte(mBuffer[mCursor++]);
            return true;
        }

        public Boolean Read(ref Int16 pValue)
        {
            if (Available < SIZEOF_16BIT)
            {
                return false;
            }
            pValue = BitConverter.ToInt16(mBuffer,
                                          mCursor);
            mCursor += SIZEOF_16BIT;
            return true;
        }

        public Boolean Read(ref UInt16 pValue)
        {
            if (Available < SIZEOF_16BIT)
            {
                return false;
            }
            pValue = BitConverter.ToUInt16(mBuffer,
                                           mCursor);
            mCursor += SIZEOF_16BIT;
            return true;
        }

        public Boolean Read(ref Int32 pValue)
        {
            if (Available < SIZEOF_32BIT)
            {
                return false;
            }
            pValue = BitConverter.ToInt32(mBuffer,
                                          mCursor);
            mCursor += SIZEOF_32BIT;
            return true;
        }

        public Boolean Read(ref UInt32 pValue)
        {
            if (Available < SIZEOF_32BIT)
            {
                return false;
            }
            pValue = BitConverter.ToUInt32(mBuffer,
                                           mCursor);
            mCursor += SIZEOF_32BIT;
            return true;
        }

        public Boolean Read(ref Single pValue)
        {
            if (Available < SIZEOF_32BIT)
            {
                return false;
            }
            pValue = BitConverter.ToSingle(mBuffer,
                                           mCursor);
            mCursor += SIZEOF_32BIT;
            return true;
        }

        public Boolean Read(ref Int64 pValue)
        {
            if (Available < SIZEOF_64BIT)
            {
                return false;
            }
            pValue = BitConverter.ToInt64(mBuffer,
                                          mCursor);
            mCursor += SIZEOF_64BIT;
            return true;
        }

        public Boolean Read(ref UInt64 pValue)
        {
            if (Available < SIZEOF_64BIT)
            {
                return false;
            }
            pValue = BitConverter.ToUInt64(mBuffer,
                                           mCursor);
            mCursor += SIZEOF_64BIT;
            return true;
        }

        public Boolean Read(ref Double pValue)
        {
            if (Available < SIZEOF_64BIT)
            {
                return false;
            }
            pValue = BitConverter.ToDouble(mBuffer,
                                           mCursor);
            mCursor += SIZEOF_64BIT;
            return true;
        }

        public Boolean Read(ref String pValue)
        {
            UInt16 length = 0;
            if (!Read(ref length))
            {
                return false;
            }
            if (Available < length)
            {
                return false;
            }
            pValue = mEncoding.GetString(mBuffer,
                                         mCursor,
                                         length);
            mCursor += length;
            return true;
        }

        /**
         * Primitive Byte array writing method
         *
         * @param  pBuf    an array to write data from
         * @param  pIndex  starting index in pBuf to begin getting data at
         * @param  pLength length of data to read from pBuf to the package
         * @return         true if all data was able be written, false otherwise
         */
        public Boolean Write(Byte[] pBuf,
                             Int32 pIndex,
                             Int32 pLength)
        {
            if (!Prepare(pLength))
            {
                return false;
            }
            Buffer.BlockCopy(pBuf,
                             pIndex,
                             mBuffer,
                             mLength,
                             pLength);
            mLength += pLength;
            return true;
        }

        /**
         * Primitive datatype writing methods
         *
         * @param  pValue the value to be written
         * @return        true if the value was able to be written, false otherwise
         */
        public Boolean Write(Boolean pValue)
        {
            if (!Prepare(1))
            {
                return false;
            }
            mBuffer[mLength++] = Convert.ToByte(pValue);
            return true;
        }

        public Boolean Write(Byte pValue)
        {
            if (!Prepare(1))
            {
                return false;
            }
            mBuffer[mLength++] = pValue;
            return true;
        }

        public Boolean Write(SByte pValue)
        {
            if (!Prepare(1))
            {
                return false;
            }
            mBuffer[mLength++] = Convert.ToByte(pValue);
            return true;
        }

        public Boolean Write(Int16 pValue)
        {
            if (!Prepare(SIZEOF_16BIT))
            {
                return false;
            }
            Buffer.BlockCopy(BitConverter.GetBytes(pValue),
                             0,
                             mBuffer,
                             mLength,
                             SIZEOF_16BIT);
            mLength += SIZEOF_16BIT;
            return true;
        }

        public Boolean Write(UInt16 pValue)
        {
            if (!Prepare(SIZEOF_16BIT))
            {
                return false;
            }
            Buffer.BlockCopy(BitConverter.GetBytes(pValue),
                             0,
                             mBuffer,
                             mLength,
                             SIZEOF_16BIT);
            mLength += SIZEOF_16BIT;
            return true;
        }

        public Boolean Write(Int32 pValue)
        {
            if (!Prepare(SIZEOF_32BIT))
            {
                return false;
            }
            Buffer.BlockCopy(BitConverter.GetBytes(pValue),
                             0,
                             mBuffer,
                             mLength,
                             SIZEOF_32BIT);
            mLength += SIZEOF_32BIT;
            return true;
        }

        public Boolean Write(UInt32 pValue)
        {
            if (!Prepare(SIZEOF_32BIT))
            {
                return false;
            }
            Buffer.BlockCopy(BitConverter.GetBytes(pValue),
                             0,
                             mBuffer,
                             mLength,
                             SIZEOF_32BIT);
            mLength += SIZEOF_32BIT;
            return true;
        }

        public Boolean Write(Single pValue)
        {
            if (!Prepare(SIZEOF_32BIT))
            {
                return false;
            }
            Buffer.BlockCopy(BitConverter.GetBytes(pValue),
                             0,
                             mBuffer,
                             mLength,
                             SIZEOF_32BIT);
            mLength += SIZEOF_32BIT;
            return true;
        }

        public Boolean Write(Int64 pValue)
        {
            if (!Prepare(SIZEOF_64BIT))
            {
                return false;
            }
            Buffer.BlockCopy(BitConverter.GetBytes(pValue),
                             0,
                             mBuffer,
                             mLength,
                             SIZEOF_64BIT);
            mLength += SIZEOF_64BIT;
            return true;
        }

        public Boolean Write(UInt64 pValue)
        {
            if (!Prepare(SIZEOF_64BIT))
            {
                return false;
            }
            Buffer.BlockCopy(BitConverter.GetBytes(pValue),
                             0,
                             mBuffer,
                             mLength,
                             SIZEOF_64BIT);
            mLength += SIZEOF_64BIT;
            return true;
        }

        public Boolean Write(Double pValue)
        {
            if (!Prepare(SIZEOF_64BIT))
            {
                return false;
            }
            Buffer.BlockCopy(BitConverter.GetBytes(pValue),
                             0,
                             mBuffer,
                             mLength,
                             SIZEOF_64BIT);
            mLength += SIZEOF_64BIT;
            return true;
        }

        public Boolean Write(String pValue)
        {
            Byte[] buf = mEncoding.GetBytes(pValue);
            Debug.Assert(buf.Length <= UInt16.MaxValue);
            return Write(Convert.ToUInt16(buf.Length)) &&
                   Write(buf,
                         0,
                         buf.Length);
        }

        /**
         * Allocates and resizes memory for the internal buffer as required
         * Allocates by shifting size left one bit until it exceeds OVERFLOW
         *
         * @param  pValue the value to be written
         * @return        true if space is available, false if OVERFLOW is exceeded
         */
        private Boolean Prepare(Int32 pNeeded)
        {
            Debug.Assert(pNeeded >= 0);
            if (pNeeded == 0)
            {
                return true;
            }
            Int32 size = mBuffer.Length;
            Int32 required = mLength + pNeeded;
            while (size < required)
            {
                size <<= 1;
                if (size > OVERFLOW)
                {
                    return false;
                }
            }
            if (size != mBuffer.Length)
            {
                Array.Resize<Byte>(ref mBuffer,
                                   size);
            }
            return true;
        }

        /**
         * Compresses data in the buffer using gzip
         * Writes an Int32 for the decompressed size
         *
         * @return        true if space is available, false if OVERFLOW is exceeded
         */
        public Boolean Compress()
        {
            Int32 length = Available;
            Byte[] buf = null;
            using (MemoryStream ms = new MemoryStream(Available))
            {
                using (ZOutputStream zs = new ZOutputStream(ms, zlibConst.Z_DEFAULT_COMPRESSION))
                {
                    zs.Write(mBuffer, mCursor, length);
                }
                buf = ms.ToArray();
            }
            mLength = 0;
            mCursor = 0;
            return Write(length) &&
                   Write(buf, 0, buf.Length);
        }

        /**
         * Decompresses data in the buffer using gzip
         * Reads an Int32 for the decompressed size
         *
         * @return        true if space is available, false if OVERFLOW is exceeded
         */
        public Boolean Decompress()
        {
            Int32 length = 0;
            if (!Read(ref length))
            {
                return false;
            }
            if (length <= 0 ||
                length > OVERFLOW)
            {
                return false;
            }
            Byte[] buf = new Byte[length];
            using (MemoryStream ms = new MemoryStream(length))
            {
                using (ZOutputStream zs = new ZOutputStream(ms))
                {
                    zs.Write(mBuffer, mCursor, Available);
                }
                mBuffer = ms.ToArray();
            }
            mCursor = 0;
            mLength = length;
            
            return true;
        }
        public void DumpBinaryToFile(String pPath)
        {
            using (System.IO.FileStream outFile = new System.IO.FileStream(pPath, System.IO.FileMode.Create, System.IO.FileAccess.Write))
            {
                try
                {             
                    outFile.Write(mBuffer, 0, mBuffer.Length - 1);
                }
                catch (Exception)
                {

                }
            }

        }

        public void DumpToFile(String pPath, String pHeader)
        {
            using (StreamWriter sw = new StreamWriter(pPath, true))
            {
                sw.WriteLine(pHeader);

                String[] split = (mLength > 0 ? BitConverter.ToString(mBuffer, 0, mLength) : "").Split('-');
                StringBuilder hex = new StringBuilder();
                StringBuilder ascii = new StringBuilder();
                Char temp;
                if (mLength > 0)
                {
                    for (Int32 index = 0; index < split.Length; ++index)
                    {
                        temp = Convert.ToChar(mBuffer[index]);
                        hex.Append(split[index] + ' ');

                        if (Char.IsWhiteSpace(temp) || Char.IsControl(temp)) temp = '.';

                        ascii.Append(temp);
                        if ((index + 1) % 16 == 0)
                        {
                            sw.WriteLine("{0} {1}", hex, ascii);
                            hex.Length = 0;
                            ascii.Length = 0;
                        }
                    }
                    if (hex.Length > 0)
                    {
                        if (hex.Length < (16 * 3)) hex.Append(new string(' ', (16 * 3) - hex.Length));
                        sw.WriteLine("{0} {1}", hex, ascii);
                    }
                    sw.WriteLine("----------");
                }
            }
        }

        /**
         * Private constants
         */
        private const Int32 INIT = 1024; // Initialize default buffers as 1KB
        private const Int32 OVERFLOW = 1024 * 1024 * 32; // Overflow after 32MB
        private const Int32 SIZEOF_16BIT = 2;
        private const Int32 SIZEOF_32BIT = 4;
        private const Int32 SIZEOF_64BIT = 8;

        /**
         * Private member data
         */
        private Encoding mEncoding = null;
        private Byte[] mBuffer = null;
        private Int32 mLength = 0;
        private Int32 mCursor = 0;
    }
}
