﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Web;
using System.Threading;
using System.Configuration;
using Curse.GameServers.Configuration;
using Curse.GameServers.Extensions;
using System.Data.SqlClient;
using System.IO;
using System.Runtime.Serialization;
using System.Runtime.Serialization.Formatters.Binary;
using Curse.Extensions;
using System.Runtime.Serialization.Json;
using ICSharpCode.SharpZipLib.BZip2;
using Curse.GameServers.Geocoding;
using System.Net;

namespace Curse.GameServers.Caching
{
    public class GameServerCache
    {
        private List<CGameServer> _gameServers;
        private object _cacheLock = new object();

        private string _staticFileDestination;
        private bool _createFeedFiles;
        private Dictionary<FeedTimespan, Int32> _feedLifespans = new Dictionary<FeedTimespan, Int32>();
        private Dictionary<FeedTimespan, DateTime> _feedTimestamps = new Dictionary<FeedTimespan, DateTime>();

        private Thread _updateThread = null;
        private int _updateThreadInterval;
        private bool _isCacheBuilt;
        private DateTime _lastUpdateTime = DateTime.Parse("01/01/2012");

        private int _gameServerEntityTypeId;

        static GameServerCache _instance = new GameServerCache();
        public static GameServerCache Instance { get { return _instance; } }

        public GameServerCache()
        {
            _gameServers = new List<CGameServer>();

            _staticFileDestination = ConfigurationManager.AppSettings["FeedPath"];
            
            if (_createFeedFiles && !Directory.Exists(_staticFileDestination))
            {
                Directory.CreateDirectory(_staticFileDestination);
            }

            _createFeedFiles = (System.Environment.MachineName.ToLower() == ConfigurationManager.AppSettings["JobMachineName"].ToLower());
            _gameServerEntityTypeId = Int32.Parse(ConfigurationManager.AppSettings["GameServerEntityTypeID"]);

            // Populate feed lifespans
            _feedLifespans.Add(FeedTimespan.Complete, (int)TimeSpan.FromHours(3).TotalSeconds);
            _feedLifespans.Add(FeedTimespan.Weekly, (int)TimeSpan.FromHours(2).TotalSeconds);
            _feedLifespans.Add(FeedTimespan.Daily, (int)TimeSpan.FromHours(1).TotalSeconds);
            _feedLifespans.Add(FeedTimespan.Hourly, (int)TimeSpan.FromMinutes(10).TotalSeconds);
            
            UpdateCache();

            _updateThreadInterval = int.Parse(ConfigurationManager.AppSettings["UpdateThreadInterval"]);
            _updateThread = new Thread(CacheThread) { IsBackground = true };
            _updateThread.Priority = ThreadPriority.Lowest;
            _updateThread.Start();
        }

        public void Initialize() { }

        private void CacheThread()
        {
            Boolean aborted = false;
            while (!aborted)
            {
                Thread.Sleep(_updateThreadInterval);
                GC.Collect();
                try
                {
                    UpdateCache();
                }
                catch (ThreadAbortException)
                {
                    aborted = true;
                    _updateThread.Join(100);
                    Logger.Log(ELogLevel.Info, null, "Thread Abort Exception. Service shutting down.");
                }
                catch (Exception ex)
                {

                    Logger.Log(ELogLevel.Info, null, "Update Thread Exception: {0}", ex.Message + "\n" + ex.StackTrace);
                }

                try
                {
                    var responsiveServers = _gameServers.Where(p => (p.GameServerStatus == GameServerStatus.Normal || p.GameServerStatus == GameServerStatus.PingFailed) || !string.IsNullOrEmpty(p.Title)).ToArray();
                    CreateFeedFiles(responsiveServers);
                }
                catch (Exception ex)
                {
                    Logger.Log("Failed to create feed files: " + ex.Message + ", Stack: " + ex.StackTrace, ELogLevel.Error);
                }
            }
        }
        private void UpdateCache()
        {
            Dictionary<int, CGameServer> gameServers = new Dictionary<int, CGameServer>();
            var updateTime = DateTime.UtcNow;

            using (var conn = DatabaseConfiguration.GetGameServerConnection())
            {
                var lastQueryTime = _lastUpdateTime.AddSeconds(-30);

                // Load the PLayer Cache
                Dictionary<int, List<CGameServerPlayer>> playerCache = GamePlayerCache.Instance.GetGameServerPlayerCache(conn, lastQueryTime);

                // Load the Property Cache
                Dictionary<int, List<CGameServerProperty>> gameProperties = GameServerPropertyCache.Instance.GetGameServerPlayerCache(conn);

                // Load the Tag Cache
                Dictionary<int, List<CTag>> gameServerTags = GameServerTagCache.Instance.GetGameServerTagCache(conn, _gameServerEntityTypeId);

                // Load the Ratings Cache
                List<CGameServerRating> gameServerRatings = GameServerRatingCache.Instance.GetGameServerRatingCache(conn);


                var pluginQuery = "SELECT GameServerID, GameServerPluginID FROM GameServerPluginMap " +
                                  "inner join GameServer on GameServer.ID = GameServerPluginMap.GameServerID " +
                                  "where DateLastMined > @lastUpdateTime";

                var pluginCmd = new SqlCommand(pluginQuery, conn);
                pluginCmd.Parameters.Add("lastUpdateTime", System.Data.SqlDbType.DateTime).Value = lastQueryTime;

                Dictionary<int, List<int>> pluginsByServerID = new Dictionary<int, List<int>>();
                using (SqlDataReader reader = pluginCmd.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        int gameServerID = reader.GetInt32(0);
                        int pluginID = reader.GetInt32(1);

                        if (!pluginsByServerID.ContainsKey(gameServerID))
                        {
                            pluginsByServerID.Add(gameServerID, new List<int>());   
                        }
                        pluginsByServerID[gameServerID].Add(pluginID);
                    }
                }

                var query = "SELECT GameServer.* " +
                    "FROM GameServer WITH(NOLOCK) " +
                    "INNER JOIN GameServerDataSource WITH(NOLOCK) ON GameServerDataSource.ID = GameServer.GameServerDataSourceID " +
                    "INNER JOIN GameServerProcessor WITH(NOLOCK) ON GameServerProcessor.ID = GameServerDataSource.GameServerProcessorID " +
                    "WHERE GameServerProcessor.[Enabled] = 1 " +
                    "AND GameServer.DateLastMined > @lastUpdateTime";

                var cmd = new SqlCommand(query, conn);
                cmd.Parameters.Add("lastUpdateTime", System.Data.SqlDbType.DateTime).Value = lastQueryTime;
                using (SqlDataReader reader = cmd.ExecuteReader())
                {
                    while (reader.Read())
                    {
                        var gameServerId = reader.GetInt32(reader.GetOrdinal("ID"));

                        var gameServer = new CGameServer();
                        gameServer.SetFromDataReader(conn, reader, playerCache, gameProperties, gameServerTags, gameServerRatings, pluginsByServerID);

                        
                        if (gameServers.ContainsKey(gameServerId))
                        {
                            gameServers[gameServerId] = gameServer;
                        }
                        else
                        {
                            gameServers.Add(gameServer.ID, gameServer);
                        }
                        
                    }                  
                }

                lock (_cacheLock)
                {
                    _gameServers = gameServers.Values.ToList();
                }               

                _lastUpdateTime = updateTime;
                _isCacheBuilt = true;
              
            }
        }
        
        public CGameServer GetGameServerByID(int gameServerId)
        {
            return _gameServers.FirstOrDefault(p => p.ID == gameServerId);
        }
        
        public List<CGameServer> GetAllGameServers()
        {
            if (!_isCacheBuilt)
            {
                return null;
            }

            return _gameServers;
        }
        public List<CGameServer> GetFilteredGameServers(int takeLimit)
        {
            if (!_isCacheBuilt)
            {
                return null;
            }

            List<CGameServer> verificationRequestServers = _gameServers.Where(s => s.VerificationStatus == EVerificationStatus.Pending).OrderBy(s => s.DateLastMined).ToList();
            verificationRequestServers.AddRange(_gameServers.Except(verificationRequestServers).OrderBy(s => s.DateLastMined).ToArray());
            return verificationRequestServers;
        }
        public List<CGameServer> GetGameServersByDataSourceID(int dataSourceId)
        {
            if (!_isCacheBuilt)
            {
                return null;
            }
            return _gameServers.Where(p => p.GameServerDataSourceID == dataSourceId).ToList();
        }
        public List<CGameServer> GetAllGameServersSince(DateTime since)
        {
            var gameServers = new List<CGameServer>();
            if (_isCacheBuilt)
            {
                gameServers = _gameServers.Where(p => p.DateLastMined > since).ToList();
            }
            return gameServers;
        }
        public List<CGameServer> GetAllGameServersByDataSourceSince(int dataSourceId, DateTime since)
        {
            var gameServers = new List<CGameServer>();
            if (_isCacheBuilt)
            {
                gameServers = _gameServers.Where(p => p.DateLastMined > since && p.GameServerDataSourceID == dataSourceId).ToList();
            }
            return gameServers;
        }
        public List<CGameServer> GetFilteredGameServersByDataSourceSince(int dataSourceId, DateTime since, int takeLimit)
        {
            if (!_isCacheBuilt)
            {
                return null;
            }

            IEnumerable<CGameServer> gameServers = _gameServers.Where(p => p.GameServerDataSourceID == dataSourceId && p.DateLastMined > since).OrderByDescending(s => s.DateLastMined).Take(takeLimit).ToArray();
            return gameServers.ToList();
        }

        private bool IsFeedExpired(FeedTimespan feedTimespan)
        {
            DateTime timestamp;
            Int32 lifespan = _feedLifespans[feedTimespan];

            if (_feedTimestamps.TryGetValue(feedTimespan, out timestamp))
            {
                int age = (int)DateTime.UtcNow.Subtract(timestamp).TotalSeconds;
                if (age > lifespan)
                {
                    Logger.Log("Feed has expired: {0}", ELogLevel.Info, feedTimespan);
                    return true;
                }
                else
                {
                    return false;
                }

            }
            else
            {
                Logger.Log("Feed has not yet been created: {0}", ELogLevel.Info, feedTimespan);
                return true;
            }

        }
        private void CreateFeedFiles(CGameServer[] gameServers)
        {
            if (!_createFeedFiles) { return; }

            foreach (FeedTimespan timespan in System.Enum.GetValues(typeof(FeedTimespan)))
            {
                try
                {
                    CreateFeedFile(timespan, gameServers);
                }
                catch (Exception exc)
                {
                    Logger.Log("Unable to create feed file! Details: {0}", ELogLevel.Error, exc.GetExceptionDetails());
                }
            }
        }
        private void CreateFeedFile(FeedTimespan feedTimespan, CGameServer[] gameServers)
        {
            if (!IsFeedExpired(feedTimespan))
            {
                return;
            }

            _feedTimestamps[feedTimespan] = DateTime.UtcNow;
            int hours = (int)feedTimespan;
            CGameServer[] incrementalList = null;
            if (hours > 0)
            {
                TimeSpan t = new TimeSpan(hours, 0, 0);
                DateTime date = DateTime.UtcNow.Subtract(t);
                incrementalList = gameServers.Where(p => p.DateModified >= date).ToArray();
            }
            else
            {
                incrementalList = gameServers;
            }
            SaveBinaryToDisk(feedTimespan, incrementalList);
            SaveJsonToDisk(feedTimespan, incrementalList);
            //SaveXmlToDisk(feedTimespan, incrementalList);
        }
        
        private void SaveJsonToDisk(FeedTimespan feedTimespan, CGameServer[] incrementalList)
        {
            byte[] uncompressedBytes = null;

            using (MemoryStream uncompressedStream = new MemoryStream())
            {
                DataContractJsonSerializer xs = new DataContractJsonSerializer(incrementalList.GetType());
                xs.WriteObject(uncompressedStream, incrementalList);
                uncompressedStream.Position = 0;
                uncompressedBytes = uncompressedStream.ToArray();
            }
            byte[] compressedBytes = null;

            using (MemoryStream compressedStream = new MemoryStream())
            {
                using (BZip2OutputStream bz2 = new BZip2OutputStream(compressedStream))
                {
                    bz2.Write(uncompressedBytes, 0, uncompressedBytes.Length);
                }
                compressedBytes = compressedStream.ToArray();
            }

            string basePath = _staticFileDestination;
            string tempfile = Path.Combine(basePath, Utility.UniqueNumber + ".bz2");
            string backupfile = Path.Combine(basePath, Utility.UniqueNumber + ".bz2");
            string currentfile = Path.Combine(basePath, feedTimespan.ToString() + ".json.bz2");
            using (BinaryWriter binWriter = new BinaryWriter(File.Open(tempfile, FileMode.Create)))
            {
                binWriter.Write(compressedBytes);
            }
            try
            {
                if (File.Exists(currentfile))
                {
                    File.Replace(tempfile, currentfile, backupfile, true);
                }
                else
                {
                    File.Copy(tempfile, currentfile, true);
                }
            }
            catch (Exception ex)
            {
                Logger.Log("CGameServerCache - SaveXmlToDisk Exception! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
            }
            finally
            {
                File.Delete(tempfile);
                if (File.Exists(backupfile))
                {
                    File.Delete(backupfile);
                }
            }
        }
        private void SaveBinaryToDisk(FeedTimespan feedTimespan, CGameServer[] incrementalList)
        {
            using (MemoryStream ms = new MemoryStream())
            {
                BinaryFormatter bf = new BinaryFormatter();
                bf.Serialize(ms, incrementalList);

                byte[] uncompressed = ms.ToArray();
                byte[] compressed = Utility.GetCompressedBytes(uncompressed);

                string basePath = _staticFileDestination;
                string tempfile = Path.Combine(basePath, Utility.UniqueNumber + ".zip");
                string backupfile = Path.Combine(basePath, Utility.UniqueNumber + ".zip");
                string currentfile = Path.Combine(basePath, feedTimespan.ToString() + ".zip");
                using (BinaryWriter binWriter = new BinaryWriter(File.Open(tempfile, FileMode.Create)))
                {
                    binWriter.Write(compressed);
                }
                try
                {
                    if (File.Exists(currentfile))
                    {
                        File.Replace(tempfile, currentfile, backupfile, true);
                    }
                    else
                    {
                        File.Copy(tempfile, currentfile, true);
                    }
                }
                catch (Exception ex)
                {
                    Logger.Log("CGameServerCache - SaveBinaryToDisk Exception! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
                }
                finally
                {
                    File.Delete(tempfile);
                    if (File.Exists(backupfile))
                    {
                        File.Delete(backupfile);
                    }
                }
            }
        }

        public CServiceResponse CreateGameServer(CGameServerDataSource dataSource, byte[] ipAddress, int port, string slug, int? queryPort, int? gameServerRatingId)
        {
            if (_gameServers.FirstOrDefault(p => p.IPAddress == ipAddress && p.Port == port) != null)
            {
                Logger.Log("CreateGameServer Failed: This IPAddress and Port combination already exist. {0} : {1}", ELogLevel.Warning, ipAddress, port);
                return new CServiceResponse(EServiceResponseStatus.CreateGameServer_IPAddressInUse);
            }

            try
            {
                using (var conn = DatabaseConfiguration.GetGameServerConnection())
                {
                    var cmd = new SqlCommand("spCreateGameServer", conn);
                    cmd.CommandType = System.Data.CommandType.StoredProcedure;

                    cmd.Parameters.Add("DataSourceID", System.Data.SqlDbType.Int).Value = dataSource.ID;
                    cmd.Parameters.Add("IPAddress", System.Data.SqlDbType.VarBinary).Value = ipAddress;
                    cmd.Parameters.Add("Port", System.Data.SqlDbType.Int).Value = port;
                    cmd.Parameters.Add("Slug", System.Data.SqlDbType.NVarChar).Value = slug;
                    if (queryPort.HasValue)
                    {
                        cmd.Parameters.Add("QueryPort", System.Data.SqlDbType.Int).Value = queryPort;
                    }
                                        
                    if (gameServerRatingId.HasValue)
                    {
                        cmd.Parameters.Add("GameServerRatingID", System.Data.SqlDbType.Int).Value = gameServerRatingId;
                    }

                    cmd.ExecuteNonQuery();

                    return new CServiceResponse(EServiceResponseStatus.Successful);
                }
            }
            catch (Exception exc)
            {
                Logger.Log("CreateGameServer: Failed to create server. Details: {0}", ELogLevel.Error, exc.GetExceptionDetails());
                return new CServiceResponse(EServiceResponseStatus.UnknownException, exc.Message);
            }
        }
        public CServiceResponse SetVerificationToken(int dataSourceId, byte[] ipAddress, int port, Guid verificationToken, EVerificationStatus verificationStatus)
        {
            using (var conn = DatabaseConfiguration.GetGameServerConnection())
            {
                var cmd = new SqlCommand("UPDATE GameServer SET VerificationToken = @Token, VerificationStatus = @Status, DateLastMined = NULL WHERE GameServerDataSourceID = @DataSourceID AND IPAddress = @Address AND Port = @Port", conn);
                cmd.Parameters.Add("Address", System.Data.SqlDbType.VarBinary).Value = ipAddress;
                cmd.Parameters.Add("Port", System.Data.SqlDbType.Int).Value = port;
                cmd.Parameters.Add("Token", System.Data.SqlDbType.UniqueIdentifier).Value = verificationToken;
                cmd.Parameters.Add("Status", System.Data.SqlDbType.TinyInt).Value = verificationStatus;
                cmd.Parameters.Add("DataSourceID", System.Data.SqlDbType.Int).Value = dataSourceId;

                try
                {
                    cmd.ExecuteNonQuery();
                    return new CServiceResponse(EServiceResponseStatus.Successful);
                }
                catch (Exception exc)
                {
                    Logger.Log("Failed to update verification token and status for {0}:{1}. Details: {2}", ELogLevel.Error, ipAddress, port, exc.GetExceptionDetails());
                    return new CServiceResponse(EServiceResponseStatus.UnknownException, exc.Message);
                }
            }
        }
        public CServiceResponse UpdateGameServerStatus(int dataSourceId, byte[] ipAddress, int port, GameServerStatus newStatus)
        {
            using (var conn = DatabaseConfiguration.GetGameServerConnection())
            {
                var cmd = new SqlCommand("UPDATE GameServer SET Status = @NewStatus WHERE GameServerDataSourceID = @DataSourceID AND IPAddress = @Address AND Port = @Port", conn);
                cmd.Parameters.Add("Address", System.Data.SqlDbType.VarBinary).Value = ipAddress;
                cmd.Parameters.Add("Port", System.Data.SqlDbType.Int).Value = port;
                cmd.Parameters.Add("NewStatus", System.Data.SqlDbType.TinyInt).Value = newStatus;
                cmd.Parameters.Add("DataSourceID", System.Data.SqlDbType.Int).Value = dataSourceId;

                try
                {
                    cmd.ExecuteNonQuery();
                    return new CServiceResponse(EServiceResponseStatus.Successful);
                }
                catch (Exception exc)
                {
                    Logger.Log("Failed to update game server status for {0}:{1}. Details: {2}", ELogLevel.Error, ipAddress, port, exc.GetExceptionDetails());
                    return new CServiceResponse(EServiceResponseStatus.UnknownException, exc.Message);
                }
            }
        }
        public CServiceResponse UpdateGameServerCountry(int dataSourceId, byte[] ipAddress, int port, string countryCode)
        {
            using (var conn = DatabaseConfiguration.GetGameServerConnection())
            {
                int countryId = CountryCache.Instance.GetCountryIdFromCode(countryCode);
                if (countryId == 0)
                {
                    return new CServiceResponse(EServiceResponseStatus.UpdateGameServerCountry_UnknownCountryCode);
                }

                var cmd = new SqlCommand("UPDATE GameServer SET ManualCountryID = @CountryId, DateLastMined = @DateModified WHERE GameServerDataSourceID = @DataSourceID AND IPAddress = @Address AND Port = @Port", conn);
                cmd.Parameters.Add("Address", System.Data.SqlDbType.VarBinary).Value = ipAddress;
                cmd.Parameters.Add("Port", System.Data.SqlDbType.Int).Value = port;
                cmd.Parameters.Add("CountryId", System.Data.SqlDbType.TinyInt).Value = countryId;
                cmd.Parameters.Add("DataSourceID", System.Data.SqlDbType.Int).Value = dataSourceId;
                cmd.Parameters.Add("DateModified", System.Data.SqlDbType.DateTime).Value = DateTime.UtcNow;

                try
                {
                    cmd.ExecuteNonQuery();
                    return new CServiceResponse(EServiceResponseStatus.Successful);
                }
                catch (Exception exc)
                {
                    Logger.Log("Failed to update game server Country for {0}:{1}. Details: {2}", ELogLevel.Error, ipAddress, port, exc.GetExceptionDetails());
                    return new CServiceResponse(EServiceResponseStatus.UnknownException, exc.Message);
                }
            }
        }
        public CServiceResponse UpdateGameServerQueryPort(int dataSourceId, byte[] ipAddress, int port, int queryPort)
        {
            using (var conn = DatabaseConfiguration.GetGameServerConnection())
            {
                var cmd = new SqlCommand("UPDATE GameServer SET QueryPort = @QueryPort WHERE GameServerDataSourceID = @DataSourceID AND IPAddress = @Address AND Port = @Port", conn);
                cmd.Parameters.Add("Address", System.Data.SqlDbType.VarBinary).Value = ipAddress;
                cmd.Parameters.Add("Port", System.Data.SqlDbType.Int).Value = port;
                cmd.Parameters.Add("QueryPort", System.Data.SqlDbType.Int).Value = queryPort;
                cmd.Parameters.Add("DataSourceID", System.Data.SqlDbType.Int).Value = dataSourceId;
                
                try
                {
                    cmd.ExecuteNonQuery();
                    return new CServiceResponse(EServiceResponseStatus.Successful);
                }
                catch (Exception exc)
                {
                    Logger.Log("Failed to update game server QueryPort for {0}:{1}. Details: {2}", ELogLevel.Error, ipAddress, port, exc.GetExceptionDetails());
                    return new CServiceResponse(EServiceResponseStatus.UnknownException, exc.Message);
                }
            }
        }
        public CServiceResponse UpdateGameServerRatingID(int dataSourceId, byte[] ipAddress, int port, int? ratingId)
        {
            using (var conn = DatabaseConfiguration.GetGameServerConnection())
            {
                var cmd = new SqlCommand("UPDATE GameServer SET GameServerRatingID = @GameServerRatingID WHERE GameServerDataSourceID = @DataSourceID AND IPAddress = @Address AND Port = @Port", conn);
                cmd.Parameters.Add("Address", System.Data.SqlDbType.VarBinary).Value = ipAddress;
                cmd.Parameters.Add("Port", System.Data.SqlDbType.Int).Value = port;
                cmd.Parameters.Add("GameServerRatingID", System.Data.SqlDbType.Int).Value = DBNull.Value;
                cmd.Parameters.Add("DataSourceID", System.Data.SqlDbType.Int).Value = dataSourceId;

                if (ratingId.HasValue)
                {
                    cmd.Parameters["GameServerRatingID"].Value = ratingId;
                }

                try
                {
                    cmd.ExecuteNonQuery();
                    return new CServiceResponse(EServiceResponseStatus.Successful);
                }
                catch (Exception exc)
                {
                    Logger.Log("Failed to update game server GameServerRatingID for {0}:{1}. Details: {2}", ELogLevel.Error, ipAddress, port, exc.GetExceptionDetails());
                    return new CServiceResponse(EServiceResponseStatus.UnknownException, exc.Message);
                }
            }
        }
        public void QueueServerImmediately(int dataSourceId, byte[] ipAddress, int port)
        {
            using (var conn = DatabaseConfiguration.GetGameServerConnection())
            {
                var cmd = new SqlCommand("UPDATE GameServer SET MinerStatus = @MinerStatus, DateLastMined = NULL WHERE GameServerDataSourceID = @DataSourceID AND IPAddress = @Address AND Port = @Port", conn);
                cmd.Parameters.Add("Address", System.Data.SqlDbType.VarBinary).Value = ipAddress;
                cmd.Parameters.Add("Port", System.Data.SqlDbType.Int).Value = port;
                cmd.Parameters.Add("MinerStatus", System.Data.SqlDbType.TinyInt).Value = 1;
                cmd.Parameters.Add("DataSourceID", System.Data.SqlDbType.Int).Value = dataSourceId;

                try
                {
                    cmd.ExecuteNonQuery();
                }
                catch (Exception exc)
                {
                    Logger.Log("Failed to update game server status for {0}:{1}. Details: {2}", ELogLevel.Error, ipAddress, port, exc.GetExceptionDetails());
                }
            }
        }        
    }
}