﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.Serialization;
using System.ServiceModel;
using System.ServiceModel.Web;
using System.Text;
using Curse;
using Curse.Extensions;
using System.Configuration;
using Curse.GameServers.Caching;
using Curse.GameServers.Extensions;
//using Curse.GameServers.GeoCoding;

namespace Curse.GameServers
{
    [ServiceBehavior(Name = "GameServerService", Namespace = "http://gameservers.curse.local/", AddressFilterMode = AddressFilterMode.Any, ConcurrencyMode = ConcurrencyMode.Multiple, InstanceContextMode = InstanceContextMode.Single)]
    public class GameServerService : IGameServerService
    {
        private static object _startLock = new object();

        static GameServerService()
        {
#if DEBUG
            Logger.SetLogLevel = ELogLevel.Debug;
#else
            Logger.SetLogLevel = ELogLevel.Error;
#endif
            Logger.SetLogPath = ConfigurationManager.AppSettings["LogPath"];

            lock (_startLock)
            {
                try
                {
                    Logger.Log("Initializing Country Cache...", ELogLevel.Info);
                    CountryCache.Instance.Initialize();

                    Logger.Log("Initializing Game Server Data Source Cache...", ELogLevel.Info);
                    GameServerDataSourceCache.Instance.Initialize();

                    Logger.Log("Initializing Game Server Cache...", ELogLevel.Info);
                    GameServerCache.Instance.Initialize();

                    Logger.Log("Initializing Game Release Cache...", ELogLevel.Info);
                    GameReleaseCache.Instance.Initialize();

                    // Get the IPDatabase Updater Running
                    //IPDatabaseUpdater.Initialize();
                }
                catch (Exception ex)
                {
                    Logger.Log("Could not start GameServer Service! Details: {0}", ELogLevel.Error, ex.GetExceptionDetails());
                    throw;
                }
            }

            Logger.Log("Caches Built: Service Started!", ELogLevel.Info);
        }

        #region Game Server Methods
        public CGameServerDataSource[] GetAllDataSources()
        {
            return GameServerDataSourceCache.Instance.GetAllDataSources().ToArray();
        }
        
        public CGameServerDataSource[] GetDataSourcesByGameID(int gameId)
        {
            return GameServerDataSourceCache.Instance.GetDataSourcesByGameID(gameId).ToArray();
        }

        public CGameServer[] GetAllGameServers()
        {
            return GameServerCache.Instance.GetAllGameServers().ToArray();
        }

        public CGameServer[] GetFilteredGameServers(int takeLimit)
        {
            return GameServerCache.Instance.GetFilteredGameServers(takeLimit).ToArray();
        }
        
        public CGameServer[] GetGameServersByDataSourceID(string dataSourceName)
        {
            // Get the DataSource
            var dataSource = GameServerDataSourceCache.Instance.GetByName(dataSourceName);
            if (dataSource == null)
            {
                Logger.Log("Unable to find a DataSource Named {0}", ELogLevel.Warning, dataSourceName);
                return null;
            }

            return GameServerCache.Instance.GetGameServersByDataSourceID(dataSource.ID).ToArray();
        }

        public CGameServer[] GetAllGameServerActivitySince(long since)
        {
            return GameServerCache.Instance.GetAllGameServersSince(since.ToDateTime()).ToArray();
        }

        public CGameServer[] GetAllGameServerActivityByDataSourceSince(string dataSourceName, long since)
        {
            // Get the DataSource
            var dataSource = GameServerDataSourceCache.Instance.GetByName(dataSourceName);
            if (dataSource == null)
            {
                Logger.Log("Unable to find a DataSource Named {0}", ELogLevel.Warning, dataSourceName);
                return null;
            }

            return GameServerCache.Instance.GetAllGameServersByDataSourceSince(dataSource.ID, since.ToDateTime()).ToArray();
        }

        public CGameServer[] GetFilteredGameServersByDataSourceSince(string dataSourceName, long since, int takeLimit)
        {
            var dataSource = GameServerDataSourceCache.Instance.GetByName(dataSourceName);
            if (dataSource == null)
            {
                Logger.Log("Unable to find a DataSource Named {0}", ELogLevel.Warning, dataSourceName);
                return null;
            }

            return GameServerCache.Instance.GetFilteredGameServersByDataSourceSince(dataSource.ID, since.ToDateTime(), takeLimit).ToArray();
        }

        public CServiceResponse CreateGameServer(string dataSourceName, byte[] ipAddress, int port, string slug, int? queryPort, int? gameServerRatingId)
        {
            // Get the DataSource
            var dataSource = GameServerDataSourceCache.Instance.GetByName(dataSourceName);
            if (dataSource == null)
            {
                Logger.Log("Unable to find a DataSource Named {0}", ELogLevel.Warning, dataSourceName);
                return new CServiceResponse(EServiceResponseStatus.CreateGameServer_UnknownDataSource);
            }

            // Initial status to 1 (Normal)
            return GameServerCache.Instance.CreateGameServer(dataSource, ipAddress, port, slug, queryPort, gameServerRatingId);
        }

        public CServiceResponse ClaimGameServer(int gameServerId, int userId)
        {
            var gameServer = GameServerCache.Instance.GetGameServerByID(gameServerId);
            if (gameServer == null)
            {
                return new CServiceResponse(EServiceResponseStatus.ClaimGameServer_InvalidGameServerId);
            }

            return gameServer.Claim(userId);
        }

        public CServiceResponse UploadTagIcon(int tagId, byte[] tagIcon)
        {
            throw new NotImplementedException();
        }

        public CServiceResponse SetServerVerification(string dataSourceName, byte[] ipAddress, int port, Guid verificationToken, EVerificationStatus verificationStatus)
        {
            var dataSource = GameServerDataSourceCache.Instance.GetByName(dataSourceName);
            if (dataSource == null)
            {
                Logger.Log("Unable to find a DataSource Named {0}", ELogLevel.Warning, dataSourceName);
                return new CServiceResponse(EServiceResponseStatus.SetServerVerification_UnknownDataSource);
            }

            return GameServerCache.Instance.SetVerificationToken(dataSource.ID, ipAddress, port, verificationToken, verificationStatus);
        }

        public CServiceResponse UpdateGameServerStatus(string dataSourceName, byte[] ipAddress, int port, GameServerStatus newStatus)
        {
            var dataSource = GameServerDataSourceCache.Instance.GetByName(dataSourceName);
            if (dataSource == null)
            {
                Logger.Log("Unable to find a DataSource Named {0}", ELogLevel.Warning, dataSourceName);
                return new CServiceResponse(EServiceResponseStatus.UpdateGameServerStatus_UnknownDataSource);
            }

            return GameServerCache.Instance.UpdateGameServerStatus(dataSource.ID, ipAddress, port, newStatus);
        }

        public CServiceResponse UpdateGameServerCountry(string dataSourceName, byte[] ipAddress, int port, string countryCode)
        {
            var dataSource = GameServerDataSourceCache.Instance.GetByName(dataSourceName);
            if (dataSource == null)
            {
                Logger.Log("Unable to find a DataSource Named {0}", ELogLevel.Warning, dataSourceName);
                return new CServiceResponse(EServiceResponseStatus.UpdateGameServerStatus_UnknownDataSource);
            }

            return GameServerCache.Instance.UpdateGameServerCountry(dataSource.ID, ipAddress, port, countryCode);
        }

        public void QueueServerImmediately(string dataSourceName, byte[] ipAddress, int port)
        {
            var dataSource = GameServerDataSourceCache.Instance.GetByName(dataSourceName);
            if (dataSource == null)
            {
                Logger.Log("Unable to find a DataSource Named {0}", ELogLevel.Warning, dataSourceName);
            }

            GameServerCache.Instance.QueueServerImmediately(dataSource.ID, ipAddress, port);
        }

        public CServiceResponse UpdateGameServerQueryPort(string dataSourceName, byte[] ipAddress, int port, int queryPort)
        {
            var dataSource = GameServerDataSourceCache.Instance.GetByName(dataSourceName);
            if (dataSource == null)
            {
                Logger.Log("Unable to find a DataSource Named {0}", ELogLevel.Warning, dataSourceName);
                return new CServiceResponse(EServiceResponseStatus.UpdateGameServerQueryPort_UnknownDataSource);
            }

            return GameServerCache.Instance.UpdateGameServerQueryPort(dataSource.ID, ipAddress, port, queryPort);
        }

        public CServiceResponse UpdateGameServerRatingID(string dataSourceName, byte[] ipAddress, int port, int? ratingId)
        {
            var dataSource = GameServerDataSourceCache.Instance.GetByName(dataSourceName);
            if (dataSource == null)
            {
                Logger.Log("Unable to find a DataSource Named {0}", ELogLevel.Warning, dataSourceName);
                return new CServiceResponse(EServiceResponseStatus.UpdateGameServerQueryPort_UnknownDataSource);
            }

            return GameServerCache.Instance.UpdateGameServerRatingID(dataSource.ID, ipAddress, port, ratingId);
        }

        public CServiceResponse CreateGameServerRating(string title, string description, EGameServerRatingStatus status)
        {
            return GameServerRatingCache.Instance.CreateGameServerRating(title, description, status);
        }

        public CServiceResponse UpdateGameServerRating(int gameServerRatingId, string title, string description, EGameServerRatingStatus status)
        {
            return GameServerRatingCache.Instance.UpdateGameServerRating(gameServerRatingId, title, description, status);
        }

        public CServiceResponse DeleteGameServerRating(int gameServerRatingId)
        {
            return GameServerRatingCache.Instance.DeleteGameServerRating(gameServerRatingId);
        }
        #endregion

        #region GameRelease Methods
        public CGameRelease[] GetAllGameReleases()
        {
            return GameReleaseCache.Instance.GetAllReleases().ToArray();
        }

        public CGameRelease[] GetAllGameReleasesByGameID(int gameID)
        {
            return GameReleaseCache.Instance.GetAllReleasesByGameID(gameID).ToArray();
        }

        public CServiceResponse UpdateGameReleaseNotes(int releaseId, string releaseNotes, string releaseNoteLink)
        {
            return GameReleaseCache.Instance.UpdateReleaseNotes(releaseId, releaseNotes, releaseNoteLink);
        }
        #endregion
    }
}
