﻿using Curse;
using System;
using System.Collections.Generic;
using System.Data;
using System.Data.SqlClient;
using System.Text;

namespace HardwareCenter
{
    /**
     * Handles all the internal details of managing data pushes to a database
     *
     * @author Shane Bryldt
     */
    public static partial class DB
    {
        /**
         * Prepares the DB class for use
         * 
         * @param pHardwareDB  the hardware database connection string
         */
        public static void Load(String pHardwareDB)
        {
            sHardwareDB = pHardwareDB;
        }

        /**
         * Escapes and quotes strings for SQL formatting
         * 
         * @param  pValue  the value to escape and quote
         * @return         the escaped and quoted value
         */
        private static String EscapeAndQuote(String pValue)
        {
            return "'" + pValue.Replace("'", "''") + "'";
        }

        /**
         * Handles updating CPU details
         * 
         * @param pUserId  the id of the user sending the update
         * @param pHost    the host in ip address format of the sender
         * @param pCPU     the CPU details
         */
        public static void UpdateCPU(Int32 pUserId,
                                     String pHost,
                                     CPU pCPU,
                                     bool pIsLaptop)
        {
            StringBuilder sql = new StringBuilder(SQL_INITIAL_CAPACITY);
            using (SqlConnection conn = new SqlConnection(sHardwareDB))
            {
                conn.Open();
                using (SqlTransaction trans = conn.BeginTransaction())
                {
                    SqlCommand cmd = new SqlCommand();
                    cmd.Connection = conn;
                    cmd.Transaction = trans;

                    sql.AppendFormat(sCPUQueryFormat,
                                     pCPU.Vendor0Id,
                                     pCPU.Vendor1Id,
                                     pCPU.Vendor2Id,
                                     pCPU.ExtendedFamily,
                                     pCPU.Type,
                                     pCPU.Family,
                                     pCPU.Model,
                                     pCPU.ExtendedModel,
                                     pCPU.Stepping,
                                     pCPU.Brand,
                                     pCPU.Count,
                                     pCPU.Feature0Flags,
                                     pCPU.Feature1Flags,
                                     pCPU.ExtendedBrand,
                                     pCPU.ExtendedFeature0Flags,
                                     pCPU.ExtendedFeature1Flags,
                                     pCPU.Cache0,
                                     pCPU.Cache1,
                                     pCPU.Cache2,
                                     pCPU.Cache3,
                                     pCPU.Cache4,
                                     pCPU.Cache5,
                                     pCPU.Cache6,
                                     pCPU.Cache7,
                                     pCPU.Cache8,
                                     pCPU.Cache9,
                                     pCPU.Cache10,
                                     pCPU.Cache11,
                                     pCPU.Cache12,
                                     pCPU.Cache13,
                                     pCPU.Cache14,
                                     pCPU.Cache15,
                                     pCPU.L1DataCache,
                                     pCPU.L1CodeCache,
                                     pCPU.L2Cache,
                                     pCPU.L3Cache,
                                     pCPU.Frequency,
                                     EscapeAndQuote(pCPU.Description),
                                     pUserId,
                                     pIsLaptop == true ? 1 : 0);

                    cmd.CommandText = sql.ToString();
                    cmd.ExecuteNonQuery();
                    trans.Commit();
                }
            }
        }

        /**
         * Handles updating multiple GPU details
         * 
         * @param pUserId  the id of the user sending the update
         * @param pHost    the host in ip address format of the sender
         * @param pGPUs    the list of GPU details
         */
        public static void UpdateGPUs(Int32 pUserId,
                                      String pHost,
                                      PackableList<GPU> pGPUs)
        {
            StringBuilder sql = new StringBuilder(SQL_INITIAL_CAPACITY);
            using (SqlConnection conn = new SqlConnection(sHardwareDB))
            {
                conn.Open();
                using (SqlTransaction trans = conn.BeginTransaction())
                {
                    SqlCommand cmd = new SqlCommand();
                    cmd.Connection = conn;
                    cmd.Transaction = trans;

                    sql.AppendFormat("DELETE FROM gpu WHERE owner_id={0};",
                                     pUserId);

                    foreach (GPU gpu in pGPUs)
                    {
                        sql.AppendFormat(sGPUInsertQueryFormat,
                                         pUserId,
                                         gpu.Caps0,
                                         gpu.Caps1,
                                         gpu.Caps2,
                                         gpu.PresentationIntervals,
                                         gpu.CursorCaps,
                                         gpu.DevCaps0,
                                         gpu.DevCaps1,
                                         gpu.PrimitiveMiscCaps,
                                         gpu.RasterCaps,
                                         gpu.ZCmpCaps,
                                         gpu.SrcBlendCaps,
                                         gpu.DestBlendCaps,
                                         gpu.AlphaCmpCaps,
                                         gpu.ShadeCaps,
                                         gpu.TextureCaps,
                                         gpu.TextureFilterCaps,
                                         gpu.CubeTextureFilterCaps,
                                         gpu.VolumeTextureFilterCaps,
                                         gpu.TextureAddressCaps,
                                         gpu.VolumeTextureAddressCaps,
                                         gpu.LineCaps,
                                         gpu.MaxTextureWidth,
                                         gpu.MaxTextureHeight,
                                         gpu.MaxVolumeExtent,
                                         gpu.MaxTextureRepeat,
                                         gpu.MaxTextureAspectRatio,
                                         gpu.MaxAnisotropy,
                                         gpu.MaxVertexW,
                                         gpu.GuardBandLeft,
                                         gpu.GuardBandTop,
                                         gpu.GuardBandRight,
                                         gpu.GuardBandBottom,
                                         gpu.ExtentsAdjust,
                                         gpu.StencilCaps,
                                         gpu.FVFCaps,
                                         gpu.TextureOpCaps,
                                         gpu.MaxTextureBlendStages,
                                         gpu.MaxSimultaneousTextures,
                                         gpu.VertexProcessingCaps,
                                         gpu.MaxActiveLights,
                                         gpu.MaxUserClipPlanes,
                                         gpu.MaxVertexBlendMatrices,
                                         gpu.MaxVertexBlendMatrixIndex,
                                         gpu.MaxPointSize,
                                         gpu.MaxPrimitiveCount,
                                         gpu.MaxVertexIndex,
                                         gpu.MaxStreams,
                                         gpu.MaxStreamStride,
                                         gpu.VertexShaderVersion,
                                         gpu.MaxVertexShaderConst,
                                         gpu.PixelShaderVersion,
                                         gpu.PixelShader1xMaxValue,
                                         gpu.MaxNpatchTessellationLevel,
                                         gpu.MasterAdapterOrdinal,
                                         gpu.AdapterOrdinalInGroup,
                                         gpu.NumberOfAdaptersInGroup,
                                         gpu.DeclTypes,
                                         gpu.NumSimultaneousRTs,
                                         gpu.StretchRectFilterCaps,
                                         gpu.VS20Caps,
                                         gpu.VS20DynamicFlowControlDepth,
                                         gpu.VS20NumTemps,
                                         gpu.VS20StaticFlowControlDepth,
                                         gpu.PS20Caps,
                                         gpu.PS20DynamicFlowControlDepth,
                                         gpu.PS20NumTemps,
                                         gpu.PS20StaticFlowControlDepth,
                                         gpu.PS20NumInstructionSlots,
                                         gpu.VertexTextureFilterCaps,
                                         gpu.MaxVShaderInstructionsExecuted,
                                         gpu.MaxVertexShader30InstructionSlots,
                                         gpu.MaxPixelShader30InstructionSlots,
                                         EscapeAndQuote(gpu.Description),
                                         gpu.VendorId,
                                         gpu.DeviceId,
                                         gpu.SubSysId,
                                         gpu.Revision,
                                         gpu.DriverProduct,
                                         gpu.DriverVersion,
                                         gpu.DriverSubVersion,
                                         gpu.DriverBuild,
                                         gpu.MemorySize);

                        if (sql.Length >= SQL_INITIAL_CAPACITY)
                        {
                            cmd.CommandText = sql.ToString();
                            cmd.ExecuteNonQuery();
                            sql.Length = 0;
                        }
                    }
                    if (sql.Length > 0)
                    {
                        cmd.CommandText = sql.ToString();
                        cmd.ExecuteNonQuery();
                    }
                    trans.Commit();
                }
            }
        }

        /**
         * Handles updating multiple HD details
         * 
         * @param pUserId  the id of the user sending the update
         * @param pHost    the host in ip address format of the sender
         * @param pHDs    the list of HD details
         */
        public static void UpdateHDs(Int32 pUserId,
                                     String pHost,
                                     PackableList<HD> pHDs)
        {
            StringBuilder sql = new StringBuilder(SQL_INITIAL_CAPACITY);
            using (SqlConnection conn = new SqlConnection(sHardwareDB))
            {
                conn.Open();
                using (SqlTransaction trans = conn.BeginTransaction())
                {
                    SqlCommand cmd = new SqlCommand();
                    cmd.Connection = conn;
                    cmd.Transaction = trans;

                    sql.AppendFormat("DELETE FROM hd WHERE owner_id={0};",
                                     pUserId);

                    foreach (HD hd in pHDs)
                    {
                        sql.AppendFormat(sHDInsertQueryFormat,
                                         pUserId,
                                         hd.SpaceFree / 1000000,
                                         hd.SpaceTotal / 1000000);


                        if (sql.Length >= SQL_INITIAL_CAPACITY)
                        {
                            cmd.CommandText = sql.ToString();
                            cmd.ExecuteNonQuery();
                            sql.Length = 0;
                        }
                    }
                    if (sql.Length > 0)
                    {
                        cmd.CommandText = sql.ToString();
                        cmd.ExecuteNonQuery();
                    }
                    trans.Commit();
                }
            }
        }

        /**
         * Handles updating RAM details
         * 
         * @param pUserId  the id of the user sending the update
         * @param pHost    the host in ip address format of the sender
         * @param pRAM     the RAM details
         */
        public static void UpdateRAM(Int32 pUserId,
                                     String pHost,
                                     RAM pRAM)
        {
            StringBuilder sql = new StringBuilder(SQL_INITIAL_CAPACITY);
            using (SqlConnection conn = new SqlConnection(sHardwareDB))
            {
                conn.Open();
                using (SqlTransaction trans = conn.BeginTransaction())
                {
                    SqlCommand cmd = new SqlCommand();
                    cmd.Connection = conn;
                    cmd.Transaction = trans;

                    sql.AppendFormat(sRAMQueryFormat,
                                     pRAM.SizeFree / 1000000,
                                     pRAM.SizeTotal / 1000000,
                                     pUserId);

                    cmd.CommandText = sql.ToString();
                    cmd.ExecuteNonQuery();
                    trans.Commit();
                }
            }
        }

        /**
         * Handles updating OS details
         * 
         * @param pUserId  the id of the user sending the update
         * @param pHost    the host in ip address format of the sender
         * @param pOS      the OS details
         */
        public static void UpdateOS(Int32 pUserId,
                                    String pHost,
                                    OS pOS)
        {
            StringBuilder sql = new StringBuilder(SQL_INITIAL_CAPACITY);
            using (SqlConnection conn = new SqlConnection(sHardwareDB))
            {
                conn.Open();
                using (SqlTransaction trans = conn.BeginTransaction())
                {
                    SqlCommand cmd = new SqlCommand();
                    cmd.Connection = conn;
                    cmd.Transaction = trans;

                    sql.AppendFormat(sOSQueryFormat,
                                     pOS.MajorVersion,
                                     pOS.MinorVersion,
                                     pOS.BuildVersion,
                                     pOS.PlatformId,
                                     pUserId);

                    cmd.CommandText = sql.ToString();
                    cmd.ExecuteNonQuery();
                    trans.Commit();
                }
            }
        }

        /**
         * Handles updating Sound details
         * 
         * @param pUserId  the id of the user sending the update
         * @param pHost    the host in ip address format of the sender
         * @param pSound   the Sound details
         */
        public static void UpdateSound(Int32 pUserId,
                                       String pHost,
                                       Sound pSound)
        {
            StringBuilder sql = new StringBuilder(SQL_INITIAL_CAPACITY);
            using (SqlConnection conn = new SqlConnection(sHardwareDB))
            {
                conn.Open();
                using (SqlTransaction trans = conn.BeginTransaction())
                {
                    SqlCommand cmd = new SqlCommand();
                    cmd.Connection = conn;
                    cmd.Transaction = trans;

                    sql.AppendFormat(sSoundQueryFormat,
                                     pSound.Flags,
                                     pSound.MinSecondarySampleRate,
                                     pSound.MaxSecondarySampleRate,
                                     pSound.PrimaryBuffers,
                                     pSound.MaxHwMixingAllBuffers,
                                     pSound.MaxHwMixingStaticBuffers,
                                     pSound.MaxHwMixingStreamingBuffers,
                                     pSound.FreeHwMixingAllBuffers,
                                     pSound.FreeHwMixingStaticBuffers,
                                     pSound.FreeHwMixingStreamingBuffers,
                                     pSound.MaxHw3DAllBuffers,
                                     pSound.MaxHw3DStaticBuffers,
                                     pSound.MaxHw3DStreamingBuffers,
                                     pSound.FreeHw3DAllBuffers,
                                     pSound.FreeHw3DStaticBuffers,
                                     pSound.FreeHw3DStreamingBuffers,
                                     pSound.TotalHwMemBytes,
                                     pSound.FreeHwMemBytes,
                                     pSound.MaxContigFreeHwMemBytes,
                                     pSound.UnlockTransferRateHwBuffers,
                                     pSound.PlayCpuOverheadSwBuffers,
                                     EscapeAndQuote(pSound.Description),
                                     pUserId);

                    cmd.CommandText = sql.ToString();
                    cmd.ExecuteNonQuery();
                    trans.Commit();
                }
            }
        }

        /**
         * Handles updating Display details
         * 
         * @param pUserId  the id of the user sending the update
         * @param pHost    the host in ip address format of the sender
         * @param pDisplay the Display details
         */
        public static void UpdateDisplay(Int32 pUserId,
                                         String pHost,
                                         Display pDisplay)
        {
            StringBuilder sql = new StringBuilder(SQL_INITIAL_CAPACITY);
            using (SqlConnection conn = new SqlConnection(sHardwareDB))
            {
                conn.Open();
                using (SqlTransaction trans = conn.BeginTransaction())
                {
                    SqlCommand cmd = new SqlCommand();
                    cmd.Connection = conn;
                    cmd.Transaction = trans;

                    sql.AppendFormat(sDisplayQueryFormat,
                                     pDisplay.DesktopWidth,
                                     pDisplay.DesktopHeight,
                                     pUserId);

                    cmd.CommandText = sql.ToString();
                    cmd.ExecuteNonQuery();
                    trans.Commit();
                }
            }
        }

        /**
         * Constants
         */
        private const Int32 SQL_INITIAL_CAPACITY = 1024 * 1024;

        /**
         * Static member variables
         */
        private static String sHardwareDB = null;
    }
}
