#if NVP
#else

/*
 * Copyright 2005 PayPal, Inc. All Rights Reserved.
 */
using System;
using System.Collections;
using System.Reflection;
using System.Security;
using System.Security.Cryptography.X509Certificates;
using System.Text;
using System.Threading;
using System.Web.Services.Protocols;
using com.paypal.sdk.exceptions;
using com.paypal.sdk.profiles;
using com.paypal.sdk.rules;
using com.paypal.sdk.util;
using com.paypal.soap.api;
using log4net;

namespace com.paypal.sdk.core.soap
{
	/// <summary>
	/// Main class used to invoke API calls
	/// </summary>
	public class SOAPAPICaller : APICallerBase
	{
		private static readonly ILog log = LogManager.GetLogger("com.paypal.sdk.core.SOAPAPICaller");

		private IList ports;
		private CustomSecurityHeaderType header;

		/// <summary>
		/// Default constructor
		/// </summary>
		public SOAPAPICaller()
		{
			this.ports = new ArrayList();
		}

		/// <summary>
		/// Setup the SOAP header and X509 certificate.
		/// </summary>
		/// <exception cref="PayPalException">If an error ocurrs</exception>
		protected override void SetupConnection(IAPIProfile cprofile)
		{
			for (int i = 0; i < ports.Count; i++)
			{
				SoapHttpClientProtocol binding = (SoapHttpClientProtocol) ports[i];

				// Set the endpoint URL
				String name = binding.GetType().Name;
				int index = name.IndexOf("SoapBinding");
				string environment = cprofile.Environment;
				string port = name.Substring(0, index);
				Endpoint endpoint = Config.Instance.GetEndpoint(environment, port, cprofile is SignatureAPIProfile);
				if (endpoint != null && endpoint.Url != null)
				{
					binding.Url = endpoint.Url;
				}
				else
				{
					throw new TransactionException(MessageResources.GetMessage("ENDPOINT_NOT_FOUND") +
					                               " environment : '" + environment + "', port : '" + port + "' threeToken : " +
					                               (cprofile is SignatureAPIProfile));
				}

				// Set the security header
				header = new CustomSecurityHeaderType();
				header.Credentials = new UserIdPasswordType();
				header.Credentials.Username = cprofile.APIUsername;
				header.Credentials.Password = cprofile.APIPassword;
				header.Credentials.Subject = cprofile.Subject;
				// Clear the binding client certificates
				binding.ClientCertificates.Clear();


				if (cprofile is SignatureAPIProfile)
				{
					string signature = cprofile.APISignature;
					header.Credentials.Signature = signature;
				}
				else
				{
					// Get the client certificate from the keystore.
					X509Certificate x509 = GetCertificate(cprofile.APIUsername);
					binding.ClientCertificates.Add(x509);
				}

				// Set the connection timeout
				if (cprofile.Timeout > 0)
				{
					binding.Timeout = cprofile.Timeout;
				}
			}
		} // SetupConnection


		/// <summary>
		/// Invoke a PayPal API call
		/// </summary>
		/// <param name="operationName">name of the operation to call</param>
		/// <param name="request">request object which encapsulates all request parameters</param>
		/// <returns>response object which encapsulates all response data</returns>
		/// <exception cref="PayPalException">if a transmission error occurs</exception>
		public AbstractResponseType Call(string operationName, AbstractRequestType request)
		{
			this.validateProfile(this.Profile);
			SoapHttpClientProtocol binding = null;
			MethodInfo method = null;

			for (int i = 0; i < ports.Count; i++)
			{
				binding = (SoapHttpClientProtocol) ports[i];
				method = binding.GetType().GetMethod(operationName);
				if (method != null) break;
			}

			if (method == null)
			{
				throw new TransactionException(MessageResources.GetMessage("INVALID_METHOD") + operationName);
			}

			// Set the request header
			FieldInfo requesterCredentials = binding.GetType().GetField("RequesterCredentials");
			requesterCredentials.SetValue(binding, header);

			// Set the version attribute
			request.Version = WSDLVersion;

			Object[] parameters = new Object[1];
			try
			{
				Type requestType = method.GetParameters()[0].ParameterType;
				if (requestType.Name.EndsWith("Req"))
				{
					Object reqObject = Activator.CreateInstance(requestType);
					FieldInfo[] fields = requestType.GetFields(BindingFlags.Public | BindingFlags.Instance | BindingFlags.DeclaredOnly);
					fields[0].SetValue(reqObject, request);
					parameters[0] = reqObject;
				}
				else
				{
					parameters[0] = request;
				}
			}
			catch (Exception e)
			{
				throw new FatalException(e.Message, e);
			}

			DateTime startDate = DateTime.Now;
			if (log.IsInfoEnabled)
			{
				log.Info(method.Name + MessageResources.GetMessage("TRANSACTION_SENT"));
			}

			AbstractResponseType response = this.CallSOAP(method, binding, parameters);

			if (response == null)
			{
				throw new TransactionException("SOAP deserialization error. " +
				                               "Response object is null.");
			}

			// Log the transaction result
			if (log.IsInfoEnabled)
			{
				log.Info(method.Name + " Ack : " + response.Ack +
				         MessageResources.GetMessage("ELAPSED_TIME") + (DateTime.Now - startDate).Milliseconds + " ms");

				// Log the error message
				if (response.Errors != null)
				{
					for (int i = 0; i < response.Errors.Length; i++)
					{
						log.Info(method.Name + MessageResources.GetMessage("TRANSACTION_ERROR")
						         + response.Errors[i].LongMessage);
					}
				}
			}
			return response;
		} // Call


		/// <summary>
		/// Call the SOAP toolkit
		/// </summary>
		/// <param name="method">method info object</param>
		/// <param name="obj">binding object</param>
		/// <param name="args">parameters</param>
		/// <returns>response object which encapsulates all response data</returns>
		/// <exception cref="PayPalException">if a transmission error occurs</exception>
		protected AbstractResponseType CallSOAP(MethodInfo method, Object obj, Object[] args)
		{
			Exception callex = null;
			for (int retry = 0; retry < (Profile.MaximumRetries + 1); retry++)
			{
				if (retry > 0)
				{
					if (log.IsDebugEnabled)
					{
						log.Debug(callex.Message, callex);
						log.Debug("Retry " + method.Name + " #" + retry + "...");
					}
					try
					{
						Thread.Sleep(Profile.DelayTime);
					}
					catch (Exception ex)
					{
						throw new TransactionException(ex.Message, ex);
					}
				}

				try
				{
					return (AbstractResponseType) method.Invoke(obj, args);
				}
				catch (Exception e)
				{
					callex = e;
				}
			}
			// All retries failed, throw a PayPal exception
			if (callex is TargetInvocationException)
			{
				throw new TransactionException(callex.InnerException.Message, callex);
			}
			else
			{
				throw new TransactionException(callex.Message, callex);
			}
		} // CallSOAP


		private static readonly string wsdlVersion = ConfigDefaults.WSDL_VERSION;

		/// <summary>
		/// Version of WSDL being used.
		/// </summary>
		public static string WSDLVersion
		{
			get { return wsdlVersion; }
		}

		/// <summary>
		/// Retrieve the binding objects to the different PayPal ports.
		/// </summary>
		/// <exception cref="PayPalException">If an error ocurrs</exception>
		public void Initialize()
		{
			// Retrieves and stores binding objects
			try
			{
				//Assembly assembly = Assembly.LoadWithPartialName("paypal_stubs");
				Assembly assembly = Assembly.GetExecutingAssembly();
				foreach (Type type in assembly.GetTypes())
				{
					if ((type.BaseType != null) && (type.BaseType.Name.Equals("SoapHttpClientProtocol")))
					{
						ports.Add(Activator.CreateInstance(type));
					}
				}
			}
			catch (SecurityException e)
			{
				throw new FatalException(MessageResources.GetMessage("STUBS_PERMISSION_ERROR"), e);
			}
			catch (Exception e)
			{
				throw new FatalException(MessageResources.GetMessage("STUBS_LOAD_ERROR"), e);
			}
		} // Initialize


		/// <summary>
		/// Retrieve the available PayPal operations
		/// </summary>
		/// <returns>response operation list</returns>
		public IDictionary GetAvailableOperations()
		{
			IDictionary Operations = new SortedList();
			foreach (SoapHttpClientProtocol binding in ports)
			{
				foreach (
					MethodInfo method in
						binding.GetType().GetMethods(BindingFlags.Public | BindingFlags.Instance | BindingFlags.DeclaredOnly))
				{
					Operations.Add(method.Name, method);
				}
			}
			return Operations;
		} // GetAvailableOperations


		protected override void validateProfile(IAPIProfile profile)
		{
			ValidationErrors errors = new ValidationErrors();
			if (Utils.IsEmpty(profile.APIUsername))
			{
				errors.Add(new ValidationError(MessageResources.GetMessage("API_APIUSERNAME_EMPTY")));
			}
			if (Utils.IsEmpty(profile.APIPassword))
			{
				errors.Add(new ValidationError(MessageResources.GetMessage("API_APIPASSWORD_EMPTY")));
			}
			if (Utils.IsEmpty(profile.Environment))
			{
				errors.Add(new ValidationError(MessageResources.GetMessage("API_ENVIRONMENT_EMPTY")));
			}
			if (profile is SignatureAPIProfile)
			{
				if (Utils.IsEmpty(profile.APISignature))
				{
					errors.Add(new ValidationError(MessageResources.GetMessage("API_SIGNATURE_EMPTY")));
				}
			}
			if (!errors.IsEmpty())
			{
				StringBuilder msg = new StringBuilder(MessageResources.GetMessage("PROFILE_INVALID"));
				IEnumerator iEnum = errors.GetEnumerator();
				while (iEnum.MoveNext())
				{
					ValidationError error = (ValidationError) iEnum.Current;
					msg.Append("\n" + error.Key);
				}
				throw new TransactionException(msg.ToString());
			}
		}

		/// <summary>
		/// Retrieve a PayPal operation
		/// </summary>
		/// <param name="operationName"></param>
		/// <returns>response method</returns>
		public MethodInfo GetOperation(String operationName)
		{
			foreach (SoapHttpClientProtocol binding in ports)
			{
				MethodInfo method = binding.GetType().GetMethod(operationName);
				if (method != null)
				{
					return method;
				}
			}
			return null;
		} // GetOperation
	} // SOAPAPICaller	
} // core namespace
#endif