/*
 * Copyright 2005 PayPal, Inc. All Rights Reserved.
 */

using System;
using System.IO;
using System.Reflection;
using System.Text;
using System.Web.Services.Protocols;
using System.Xml;
using System.Xml.Xsl;
using com.paypal.sdk.exceptions;
using com.paypal.sdk.util;
using log4net;

namespace com.paypal.sdk.logging 
{
	/// <summary>
	/// Performs custom actions with web service calls such as logging
	/// </summary>
	public class DefaultSOAPHandler : SoapExtension 
	{
		private static ILog log = LogManager.GetLogger("com.paypal.sdk.logging.DefaultSOAPHandler");

		private static XslTransform xslt;
		
		private Stream oldStream;
		private Stream newStream;
		
		/// <summary>
		/// Initialize the SOAPHandler
		/// </summary>
		private void init()
		{
			if (xslt == null)
			{			
				Stream file = null;
				XmlTextReader reader = null;
				try 
				{
					Assembly assembly = Assembly.GetExecutingAssembly();
					file = assembly.GetManifestResourceStream(Constants.RESOURCE_ROOT + ".resources.soapProtect.xsl");

					reader = new XmlTextReader(file);
					xslt = new XslTransform();
					xslt.Load(reader, null, null);
					
				}
				catch (Exception e)
				{
					throw new FatalException(MessageResources.GetMessage("XSLT_LOAD_ERROR"), e);
				}
				finally
				{
					try 
					{
						if (reader != null) reader.Close();
					}
					catch (IOException) { ;}	
					try 
					{
						if (file != null) file.Close();
					}
					catch (IOException) { ;}
				}
			}
		} // init


		/// <summary>
		/// When the SOAP extension is accessed for the first time, the XML Web
		/// service method it is applied to is accessed to store the file
		/// name passed in, using the corresponding SoapExtensionAttribute.
		/// </summary>
		/// <param name="methodInfo">A LogicalMethodInfo representing the specific function prototype for the XML Web service method to which the SOAP extension is applied.</param>
		/// <param name="attribute">The SoapExtensionAttribute applied to the XML Web service method. </param>
		/// <returns>null</returns>
		public override object GetInitializer(LogicalMethodInfo methodInfo, SoapExtensionAttribute attribute) 
		{
			return null;
		}


		/// <param name="type">The type of the class implementing the XML Web service to which the SOAP extension is applied. </param>
		/// <returns>null</returns>
		public override object GetInitializer(Type type) 
		{
			return null;
		}


		/// <param name="initializer">The Object returned from GetInitializer</param>
		public override void Initialize(object initializer) {}		


		/// <summary>
		/// ChainStream ensures that SOAP extensions with the highest priority can modify the actual data closest to the SOAP message sent or returned over the wire.
		/// </summary>
		/// <param name="stream">A memory buffer containing the SOAP request or response.</param>
		/// <returns>A Stream representing a new memory buffer that this SOAP extension can modify.</returns>
		public override Stream ChainStream(Stream stream)
		{
			oldStream = stream;
			newStream = new MemoryStream();
			return newStream;
		}


		/// <summary>
		///  If the SoapMessageStage is such that the SoapRequest or
		///  SoapResponse is still in the SOAP format to be sent or received,
		///  save it to the log file.
		/// </summary>
		/// <param name="message">The SoapMessage to process</param>
		public override void ProcessMessage(SoapMessage message) 
		{
			// Call init() to ensure XslTransform is set up
			init();

			switch (message.Stage) 
			{
				case SoapMessageStage.BeforeSerialize:
					break;

				// Outgoing to server
				case SoapMessageStage.AfterSerialize:
					try
					{
						WriteOutput(message);
					}
					catch (PayPalException e)
					{
						if (log.IsErrorEnabled)
						{	
							log.Error(e.Message, e);
						}
					}
					break;

				// Incoming from server
				case SoapMessageStage.BeforeDeserialize:
					try
					{
						WriteInput(message);
					}
					catch (PayPalException e)
					{
						if (log.IsErrorEnabled)
						{	
							log.Error(e.Message, e);
						}
					}
					break;

				case SoapMessageStage.AfterDeserialize:
					break;

				default:
					break;
			} // switch
		} // ProcessMessage()


		/// <summary>
		/// Log the SOAP envelope into the log file if IsDebugEnable is true
		/// </summary>
		private void LogSoapEnvelope() 
		{
			newStream.Position = 0;
			TextReader reader = new StreamReader(newStream, Encoding.UTF8);
			XmlDocument document = new XmlDocument();
			document.Load(reader);
			
			Stream output = new MemoryStream(); 
			xslt.Transform(document, null, output, null);
				
			output.Position = 0;
			log.Debug((new StreamReader(output)).ReadToEnd());
		}


		/// <summary>
		/// Log the output SOAP envelope if IsDebugEnable is true
		/// </summary>
		private void WriteOutput(SoapMessage message) 
		{
			if (log.IsDebugEnabled) LogSoapEnvelope();
			newStream.Position = 0;
			Copy(newStream, oldStream);
		}


		/// <summary>
		/// Log the input SOAP envelope if IsDebugEnable is true
		/// </summary>
		private void WriteInput(SoapMessage message) 
		{
			Copy(oldStream, newStream);
			if (log.IsDebugEnabled) LogSoapEnvelope();
			newStream.Position = 0;
		}


		/// <summary>
		/// Write from one Stream to another
		/// </summary>
		/// <param name="from">Original Stream</param>
		/// <param name="to">Copy to Stream</param>
		private void Copy(Stream from, Stream to) 
		{
			TextReader reader = new StreamReader(from);
			TextWriter writer = new StreamWriter(to);
			writer.WriteLine(reader.ReadToEnd());
			writer.Flush();
		} // Copy
	} // DefaultSOAPHandler class
} // logging namespace