/*
 * Copyright 2006 PayPal, Inc. All Rights Reserved.
 */

using System;
using com.paypal.sdk.core;
using log4net;

namespace com.paypal.sdk.profiles
{
	/// <summary>
	/// implementation BaseAPIProfile.
	/// </summary>
	[Serializable]
	public abstract class BaseAPIProfile : IAPIProfile
	{
		private static ILog log = LogManager.GetLogger("com.paypal.sdk.profiles.BaseAPIProfile");

		/// <summary>
		/// The username used to access the PayPal API
		/// </summary>
		private string apiUsername = string.Empty;

		/// <summary>
		/// The password used to access the PayPal API
		/// </summary>
		[NonSerialized] private string apiPassword;

		/// <summary>
		/// The name of the entity on behalf of which this profile is issuing calls
		/// </summary>
		[NonSerialized] private string subject;


		/// <summary>
		///  The PayPal environment (Live, Sadnbox)
		/// </summary>
		private string environment = "sandbox";

		/// <summary>
		///  The connection timeout in milliseconds
		/// </summary>
		private int timeout;


		/// <summary>
		///  The maximum number of retries
		/// </summary>
		private int maximumRetries;


		/// <summary>
		///  The delay time bewteen each retry call in milliseconds
		/// </summary>
		private int delayTime;


		/// <summary>
		///  Default APUI Profile constructor
		/// </summary>
		public BaseAPIProfile()
		{
			try
			{
				this.timeout = ConfigDefaults.CONNECTION_TIMEOUT_MS;
				this.maximumRetries = Config.Instance.MaximumRetries;
				this.delayTime = ConfigDefaults.DELAY_TIME;
			}
			catch (Exception e)
			{
				log.Warn(e.Message, e);
			}
		}


		/// <summary>
		/// The username used to access the PayPal API
		/// </summary>
		public string APIUsername
		{
			get { return this.apiUsername; }
			set { this.apiUsername = value; }
		}


		/// <summary>
		/// The password used to access the PayPal API
		/// </summary>
		public string APIPassword
		{
			get { return this.apiPassword; }
			set { this.apiPassword = value; }
		}


		/// <summary>
		/// The name of the entity on behalf of which this profile is issuing calls
		/// </summary>
		/// <remarks>
		/// This is for Third-Party access
		/// </remarks>
		public string Subject
		{
			get { return this.subject; }
			set { this.subject = value; }
		}


		/// <summary>
		/// The PayPal environment (Live, Sadnbox) 
		/// </summary>
		public string Environment
		{
			get { return this.environment; }
			set { this.environment = value; }
		}


		/// <summary>
		///  The connection timeout in milliseconds
		/// </summary>
		public int Timeout
		{
			get { return this.timeout; }
			set
			{
				if (value >= 0)
					this.timeout = value;
			}
		}


		/// <summary>
		///  The maximum number of retries
		/// </summary>
		public int MaximumRetries
		{
			get { return this.maximumRetries; }
			set { if (value >= 0) this.maximumRetries = value; }
		}


		/// <summary>
		///  The delay time bewteen each retry call in milliseconds
		/// </summary>
		public int DelayTime
		{
			get { return this.delayTime; }
			set { if (value >= 0) this.delayTime = value; }
		}


		/// <summary>
		/// abstract APISignature
		/// </summary>
		public abstract string APISignature { get; set; }

		/// <summary>
		/// abstract CertificateFile
		/// </summary>
		public abstract string CertificateFile { get; set; }

		/// <summary>
		/// abstract PrivateKeyPassword
		/// </summary>
		public abstract string PrivateKeyPassword { get; set; }
	} // BaseAPIProfile
} // profiles namespace