#if NVP
#else
/*
 * Copyright 2005 PayPal, Inc. All Rights Reserved.
 */

using System;
using System.Collections;
using System.Reflection;
using System.Text;
using com.paypal.sdk.core;
using com.paypal.sdk.exceptions;
using com.paypal.sdk.profiles;
using com.paypal.sdk.rules;
using com.paypal.sdk.util;
using com.paypal.soap.api;
using com.paypal.sdk.core.soap;
namespace com.paypal.sdk.services
{
	/// <summary>
	/// Facade class that implements API caller services
	/// </summary>
	public class CallerServices
	{
		/// <summary>
		/// API Caller reference makes actual SOAP calls
		/// </summary>
		//private IAPICaller caller;
		private SOAPAPICaller caller;
		
		/// <summary>
		/// Flag which determines whether or not this object has been initialized
		/// </summary>
		private bool initialized = false;
			
		/// <summary>
		/// Default constructor gets reference to APICaller
		/// </summary>
		public CallerServices() 
		{
			caller = (SOAPAPICaller) APICallerFactory.createSOAPAPICaller();
		}

		/// <summary>
		/// Get/Set the API profile.
		/// </summary>
		public IAPIProfile APIProfile 
		{
			get 
			{
				return caller.Profile;
			}
			set
			{
				if (!initialized) 
				{
					this.Initialize();
				}
				caller.Profile = value;
			}
		}

		/// <summary>
		/// Initializes the object
		/// </summary>
		public void Initialize() 
		{
			caller.Initialize();
			initialized = true;
		}


		/// <summary>
		/// Current version of the WSDL
		/// </summary>
		/// <returns>Current version of the WSDL</returns>
		public string GetWSDLVersion()
		{
			if (!initialized) 
			{
				this.Initialize();
			}
			return SOAPAPICaller.WSDLVersion;
		}


		/// <summary>
		/// Get all available API operations
		/// </summary>
		/// <returns>Collection of operations</returns>
		public IDictionary GetAvailableOperations() 
		{
			if (!initialized) 
			{
				this.Initialize();
			}
			return caller.GetAvailableOperations();
		}
	

		/// <summary>
		/// Generates an XML string that can be populated and used to invoke the CallerServices.Call(String operationName, String requestString) methods
		/// </summary>
		/// <param name="operationName">operation name to generate the request string</param>
		/// <returns>XML string with blank request values</returns>
		public string GetRequestTemplate(string operationName)  
		{
			if (!initialized) 
			{
				this.Initialize();
			}

			MethodInfo method = caller.GetOperation(operationName);
			if (method == null)
			{
				throw new TransactionException(MessageResources.GetMessage("INVALID_METHOD") + operationName);
			}
			
			Object request;
			try
			{
				ParameterInfo[] pars = method.GetParameters();
				Type requestType = pars[0].ParameterType;
				if (requestType.Name.EndsWith("Req")) 
				{ // Handles <<Operation>>Req redirection layer
					FieldInfo[] fields = requestType.GetFields(BindingFlags.Public|BindingFlags.Instance|BindingFlags.DeclaredOnly);
					request = Activator.CreateInstance(fields[0].FieldType);
				} 
				else 
				{ 
					request = Activator.CreateInstance(requestType);
				}
			}
			catch (Exception e)
			{
				throw new TransactionException(MessageResources.GetMessage("TEMPLATE_INSTANTIATION_ERROR") + e.Message, e);
			}
			
			return XMLSerializer.ToXMLTemplate(request);
		} // GetRequestTemplate
	

		/// <summary>
		/// Invoke an API call
		/// </summary>
		/// <param name="operationName">name of the operation, eg. "GetTransactionDetails"</param>
		/// <param name="request">object containing the request parameters</param>
		/// <returns>object containing the response values</returns>
		public AbstractResponseType Call(string operationName, AbstractRequestType request)
		{
			if (!initialized) 
			{
				this.Initialize();
			}
			return caller.Call(operationName, request);
		} // Call(string, AbstractRequestType)

	
		/// <summary>
		/// Invoke an API call
		/// </summary>
		/// <param name="operationName">name of the operation</param>
		/// <param name="requestStr">XML string containing request parameters</param>
		/// <returns>XML string containing the response values</returns>
		/// <exception cref="Exception"></exception>
		public string Call(string operationName, string requestStr) 
		{		
			if (!initialized) 
			{
				this.Initialize();
			}
			
			// Retrieve the MethodInfo object
			MethodInfo method = caller.GetOperation(operationName);
			if (method == null)
			{
				throw new TransactionException(MessageResources.GetMessage("INVALID_METHOD") + " : " + operationName);
			}

			// Deserialize request object
			AbstractRequestType request;
			ParameterInfo[] pars = method.GetParameters();
			Type requestType = pars[0].ParameterType;
			if (requestType.Name.EndsWith("Req")) 
			{ // Handles <<Operation>>Req redirection layer
				FieldInfo[] fields = requestType.GetFields(BindingFlags.Public|BindingFlags.Instance|BindingFlags.DeclaredOnly);
				request = (AbstractRequestType) XMLSerializer.FromXML(requestStr, fields[0].FieldType);
			} 
			else 
			{ 
				request = (AbstractRequestType) XMLSerializer.FromXML(requestStr, requestType);
			}
			
			AbstractResponseType response = this.Call(operationName, request);
			return XMLSerializer.ToXML(response);
		} // Call(string, string)
	} // CallerServices
} // services namespace
#endif