/*
 * Copyright 2005 PayPal, Inc. All Rights Reserved.
 */

using System;
using System.Collections;
using System.IO;
using System.Reflection;
using System.Text;
using System.Xml; 
using System.Xml.Serialization;

using com.paypal.sdk.exceptions;

namespace com.paypal.sdk.util
{
	/// <summary>
	/// Serializes request and response templates
	/// </summary>
	public abstract class XMLSerializer
	{	
		/// <summary>
		/// Create an XML template for an obbject
		/// </summary>
		/// <param name="obj"></param>
		/// <returns></returns>
		public static string ToXMLTemplate(Object obj)
		{
				XMLSerializer.inflate(obj);
				return XMLSerializer.ToXML(obj);
		}

		/// <summary>
		/// Turns an object into its XML representation
		/// </summary>
		/// <param name="obj">object to serialize</param>
		/// <returns>XML representation of the object</returns>
		public static string ToXML(Object obj)
		{
			try
			{
				StringWriter Output = new StringWriter(new StringBuilder());
				XmlSerializer serializer = new XmlSerializer(obj.GetType());
				serializer.Serialize(Output, obj);

				string xml = Output.ToString().Replace("xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\"","");
				xml = xml.Replace("xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"",""); 
				xml = xml.Replace("<?xml version=\"1.0\" encoding=\"utf-16\"?>","").Trim();
				return xml;
			}
			catch (Exception e) 
			{
				throw new TransactionException(MessageResources.GetMessage("SERIALIZE_ERROR"), e);
			}
		}

		/// <summary>
		/// Turns XML into an object
		/// </summary>
		/// <param name="type">type of object to create</param>
		/// <param name="xml">xml representation of object</param>
		/// <returns></returns>
		public static Object FromXML(string xml, Type type) 
		{
			try 
			{
				XmlSerializer serializer = new XmlSerializer(type);
				XmlReader reader = new XmlTextReader(new System.IO.StringReader(xml));
				return serializer.Deserialize(reader);
			}
			catch (Exception e) 
			{
				throw new TransactionException(MessageResources.GetMessage("DESERIALIZE_ERROR"), e);
			}
		}

		private static void inflate(Object obj)
		{
			Type type = obj.GetType();
			foreach (FieldInfo field in type.GetFields(BindingFlags.Public|BindingFlags.Instance|BindingFlags.DeclaredOnly)) 
			{
				Type fieldType = field.FieldType;
				if (!fieldType.IsEnum) 
				{
					if (fieldType == typeof(string))
					{
						field.SetValue(obj, "");
					}
					else if (fieldType == typeof(bool)) 
					{
						field.SetValue(obj, true);
					}
					else if (fieldType.IsArray)
					{
						try
						{
							Type itemType = fieldType.GetElementType();
							object item = Activator.CreateInstance(itemType);
							XMLSerializer.inflate(item);
							Array array = Array.CreateInstance(itemType, 1);
							array.SetValue(item, new long[]{0});
							field.SetValue(obj, array);
						}
						catch (Exception) { ;}
					}
					else
					{
						try 
						{
							object newObj = Activator.CreateInstance(fieldType);
							XMLSerializer.inflate(newObj);
							field.SetValue(obj, newObj);
						} 
						catch (Exception) { ;}
					}
				}
			}
		}
	}
}
