﻿using Curse;
using System;
using System.Configuration;
using System.Collections;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Text;
using System.Threading;
using System.Net;
using System.Web;
using System.Web.Services;
using System.Web.Services.Protocols;
using System.Data.SqlClient;
using Curse.Logging;
using Curse.Logging.Uploader;
using Curse.Mailer;

namespace PaymentService
{    
    
    
    [WebService(Namespace = "http://paymentservice.curse.com/")]
    [WebServiceBinding(ConformsTo = WsiProfiles.BasicProfile1_1)]
    [ToolboxItem(false)]
    public class PaymentService : System.Web.Services.WebService
    {

        // These are constants as they will never, ever change:
        private const String URL_SANDBOX = "https://www.sandbox.paypal.com/cgi-bin/webscr";
        private const String URL_LIVE = "https://www.paypal.com/cgi-bin/webscr";
        private const Int32 NUM_WARNINGS = 3;

        // Configurable secret key info. Must co-incide with our PayPal account settings:
        private static readonly String sSecretKeyName = null;
        private static readonly String sSecretKeyValue = null;

        private static Dictionary<String, Int32> mWarnings = new Dictionary<String, Int32>();
        private static Dictionary<String, Object> mBans = new Dictionary<String, Object>();
        
        static PaymentService()
        {

            sSecretKeyName = ConfigurationManager.AppSettings["SecretKeyName"];
            sSecretKeyValue = ConfigurationManager.AppSettings["SecretKeyValue"];

            #region Init Logger

            Logger.Init(ConfigurationManager.AppSettings["LogPath"], "AchievementService");
            Logger.Info("Achievement Service Starting...");

            LogUploader.Initialize(13, ConfigurationManager.AppSettings["LogServiceUrl"], ConfigurationManager.AppSettings["LogServiceApiKey"]);

            #endregion

            Logger.Info("Mail Provider Initializing...");
            MailProvider.Instance.Initialize();
        }

        private void Warn(string pIpAddress)
        {
            lock (mWarnings)
            {
                if (mWarnings.ContainsKey(pIpAddress))
                {
                    mWarnings[pIpAddress] += 1;
                }
                else
                {
                    mWarnings.Add(pIpAddress, 1);
                }
            }

            if (mWarnings[pIpAddress] == NUM_WARNINGS)
            {
                Ban(pIpAddress);
            }
        }

        private void Ban(string pIpAddress)
        {
            lock(mBans)
            {
                if(!mBans.ContainsKey(pIpAddress))
                {
                    mBans.Add(pIpAddress, new Object());
                }
            }
        }

        private Transaction ProcessSubscriptionSignup()
        {
            
            // No transaction ID for this:
            String subscriptionId = Context.Request.Form["subscr_id"];
            Int32 userId = Int32.Parse(Context.Request.Form["custom"]);
            String transactionData = GetRequestData();
            Transaction transaction = new Transaction(ETransactionType.SubscriptionSignup,Context.Request.Form, subscriptionId, transactionData, IsSandboxMode());

            // Ensure that this subscription has not already been created:
            if (Subscription.Exists(subscriptionId))
            {
                transaction.Status = ETransactionStatus.Failed;
                return transaction;
            }
           
            transaction.Status = ETransactionStatus.Complete;                        
            return transaction;
        }

        private Transaction ProcessSubscriptionModification()
        {

            // No transaction ID for this:
            String subscriptionId = Context.Request.Form["subscr_id"];
            Int32 userId = Int32.Parse(Context.Request.Form["custom"]);
            String transactionData = GetRequestData();
            Transaction transaction = new Transaction(ETransactionType.ModifySubscription, Context.Request.Form, subscriptionId, transactionData, IsSandboxMode());

            // Ensure that this subscription has not already been created:
            if (!Subscription.Exists(subscriptionId))
            {
                transaction.Status = ETransactionStatus.Failed;
                return transaction;
            }

            transaction.Status = ETransactionStatus.Complete;
            return transaction;
        }

        private Transaction ProcessSubscriptionCancellation()
        {

            // No transaction ID for this:
            String subscriptionId = Context.Request.Form["subscr_id"];            
            String transactionData = GetRequestData();
            Transaction transaction = new Transaction(ETransactionType.CancelSubscription, Context.Request.Form, subscriptionId, transactionData, IsSandboxMode());

            // Ensure that this subscription has not already been created:
            if (!Subscription.Exists(subscriptionId))
            {
                transaction.Status = ETransactionStatus.Failed;
                return transaction;
            }

            transaction.Status = ETransactionStatus.Complete;
            return transaction;
        }

        private Transaction ProcessSubscriptionPayment()
        {
            // This has a transaction ID:
            String subscriptionId = Context.Request.Form["subscr_id"];
            String transactionId = Context.Request.Form["txn_id"];
            String transactionData = GetRequestData();
            Transaction transaction = new Transaction(ETransactionType.SubscriptionPayment, Context.Request.Form, transactionId, subscriptionId, transactionData, IsSandboxMode());
            bool isExistingSubscription = Subscription.Exists(subscriptionId);

            if (!isExistingSubscription)
            {
                Logger.Info("ProcessSubscriptionPayment, Sub Not Found! Sub ID: " + subscriptionId);
                transaction.Status = ETransactionStatus.Pending;
                // Make this a pending transaction:
                Transaction.AddPendingPayment(subscriptionId, transaction);                
            }
            else
            {
                Logger.Info("ProcessSubscriptionPayment, Sub Found! Sub ID: " + subscriptionId);
                transaction.Status = ETransactionStatus.Complete;
            }
            
            return transaction;
        }

        private Transaction ProcessNonRecurringPayment()
        {
            
            Int32 userId = Int32.Parse(Context.Request.Form["custom"]);
            String subscriptionId = "N-" + userId.ToString();
            String transactionData = GetRequestData();
            Transaction transaction = new Transaction(ETransactionType.NonRecurringPayment, Context.Request.Form, subscriptionId, transactionData, IsSandboxMode());          
            transaction.Status = ETransactionStatus.Complete;
            return transaction;
        }

        [WebMethod]
        public string TestSendWelcomeEmail(string pAdminKey, int userID)
        {
            if (pAdminKey != ConfigurationManager.AppSettings["AdminKey"])
            {
                Warn(Context.Request.UserHostAddress);
                return "Unauthorized user detected. Access will be banned.";
            }

            // Send a welcome e-mail
            string username = PaymentAuth.GetUsername(userID);
            string emailAddress = PaymentAuth.GetUserEmail(userID);

            MailProvider.Instance.SendTemplatedMessage("PremiumWelcome", (string)null, emailAddress, "Welcome to Curse Premium!", new string[] { username });

            return "Message Sent!";

        }

        [WebMethod]
        public string HealthCheck()
        {
            
            using (SqlConnection conn = new SqlConnection(ConfigurationManager.ConnectionStrings["PaymentDB"].ConnectionString))
            {
                
                conn.Open();
                    
            }

            return "Success";
        }

        [WebMethod]
        public string GetStats(string pAdminKey)
        {
            if (pAdminKey != ConfigurationManager.AppSettings["AdminKey"])
            {
                Warn(Context.Request.UserHostAddress);
                return "Unauthorized user detected. Access will be banned.";
            }
            string stats = "";
            using (SqlConnection conn = new SqlConnection(ConfigurationManager.ConnectionStrings["PaymentDB"].ConnectionString))
            {
                try
                {
                    conn.Open();
                    stats += "\nDatabase connection available";
                }
                catch (Exception ex)
                {
                    stats += "\nDatabase connection failure: " + ex.Message;
                }                
            }


            lock (mBans)
            {
                if (mBans.Count > 0)
                {
                    stats += "\n IP Bans:";
                }
                foreach(KeyValuePair<string, object> kvp in mBans)
                {
                    stats += "\n" + kvp.Key;
                }                
            }

            return stats;

        }

        [WebMethod]
        public void ProcessIPN()
        {

            
            //PrintRequest();

            // Ensures the request is from a legitimate source:
            if (!IsValidRequest())
            {
                return;
            }

            String transactionType = Context.Request.Form["txn_type"];
            Transaction transaction = null;
            
            switch (transactionType)
            {
                case "subscr_signup":
                    transaction = ProcessSubscriptionSignup();
                    break;
                case "subscr_payment":
                    transaction = ProcessSubscriptionPayment();
                    break;
                case "subscr_modify":
                    transaction = ProcessSubscriptionModification();
                    break;
                case "subscr_cancel":
                    transaction = ProcessSubscriptionCancellation();
                    break;
                case "web_accept":
                    transaction = ProcessNonRecurringPayment();
                    break;
                default:
                    Logger.Error("Unknown Transaction Type: " + transactionType);
                    return;

            }
            
            // Ensures the postback to PayPal checks out:
            if (!SendPostback())
            {
                Logger.Info("Send Postback Failed!");
                return;
            }

            if (transaction.Status == ETransactionStatus.Complete)
            {
                transaction.Commit();
            }
                       
        }

        [Serializable]
        public enum SubscriptionSearchType
        {
            SubscriptionID = 1,
            SubscriberID = 2
        }

        [WebMethod]
        public List<Subscription> GetActiveSubscriptionsForUsers(string apiKey, int[] userIDs)
        {
            if (apiKey != ConfigurationManager.AppSettings["ApiKey"])
            {
                Warn(Context.Request.UserHostAddress);
                return null;
            }

            return Subscription.GetActiveFromDatabase(userIDs);
        }

        [WebMethod]
        public List<Subscription> GetActiveCompSubscriptionsForUsers(string apiKey)
        {
            if (apiKey != ConfigurationManager.AppSettings["ApiKey"])
            {
                Warn(Context.Request.UserHostAddress);
                return null;
            }

            return Subscription.GetActiveCompsFromDatabase();
        }

        [WebMethod]
        public List<Subscription> GetSubscriptionsForUser(string apiKey,
            int userID)
        {
            

            if (apiKey != ConfigurationManager.AppSettings["ApiKey"])
            {
                Warn(Context.Request.UserHostAddress);
                return null;
            }

            return Subscription.GetFromDatabaseBySubscriber(userID);
        }

        [WebMethod]
        public Subscription GetSubscriptionsByID(string apiKey, string subscriptionID)
        {
            if (apiKey != ConfigurationManager.AppSettings["ApiKey"])
            {
                Warn(Context.Request.UserHostAddress);
                return null;
            }

            return Subscription.GetFromDatabase(subscriptionID);
        }

        [WebMethod]
        public List<TransactionData> GetTransactionsBySubscriptionID(string apiKey, string subscriptionId)
        {
            if (apiKey != ConfigurationManager.AppSettings["ApiKey"])
            {
                Warn(Context.Request.UserHostAddress);
                return null;
            }

            return Transaction.GetFromDatabase(subscriptionId);
        }
            
        [WebMethod]
        public Subscription CreateCompedSubscription(string apiKey,
            int userID,
            int months)
        {

            if (apiKey != ConfigurationManager.AppSettings["ApiKey"])
            {
                Warn(Context.Request.UserHostAddress);
                return null;
            }

            //create the comped subscription
            CompedSubscription compedSub = CompedSubscription.CreateCompedSubscription(userID, months);

            //create the subscription
            return Subscription.CreateCompedSubscription(compedSub, userID, months);
        }

        /// <summary>
        /// This method will comp an existing user's old comp, no matter how old it is
        /// </summary>
        /// <param name="apiKey">security key</param>
        /// <param name="userID">user's ID</param>
        /// <param name="months">number of months</param>
        /// <returns>user's subscription object</returns>
        [WebMethod]
        public Subscription CreateCompedSubscriptionFromOldComp(string apiKey,
            int userID,
            int months)
        {

            if (apiKey != ConfigurationManager.AppSettings["ApiKey"])
            {
                Warn(Context.Request.UserHostAddress);
                return null;
            }

            //create the comped subscription
            CompedSubscription compedSub = CompedSubscription.CreateCompedSubscription(userID, months);

            //create the subscription
            return Subscription.CreateCompedSubscriptionFromOldComp(compedSub, userID, months);
        }

        private bool IsSandboxMode()
        {
            if (Context.Request["test_ipn"] != null && Context.Request["test_ipn"] == "1")
            {
                return true;
            }
            else
            {
                return false;
            }
        }

        private string GetPostbackUrl()
        {
            if (IsSandboxMode())
            {
                return URL_SANDBOX;
            }
            else
            {
                return URL_LIVE;
            }
        }

        private bool SendPostback()
        {
            

            string formPostData = "cmd = _notify-validate";
            foreach (string key in Context.Request.Form)
            {
                string postValue = HttpUtility.UrlEncode(Context.Request.Form[key]);
                formPostData += "&" + key + "=" + postValue;
            }
            byte[] postByteArray = Encoding.ASCII.GetBytes(formPostData);

            WebClient webClient = new WebClient();
            byte[] responseArray = null;
            try
            {
                responseArray = webClient.UploadData(GetPostbackUrl(), "POST", postByteArray);
            }
            catch (Exception ex)
            {
                Logger.Warn(Context.Request.UserHostAddress + " Postback Error: ", ex.Message);
                return false;
            }

            string response = Encoding.ASCII.GetString(responseArray);
            bool isValidResponse = (response == "VERIFIED");


            return true;
            //return isValidResponse;

        }

        private bool IsBanned()
        {
            bool isBanned = false;
            lock (mBans)
            {
                isBanned = mBans.ContainsKey(Context.Request.UserHostAddress);
            }
            return isBanned;
        }

        private void PrintRequest()
        {
            //System.Diagnostics.Debug.WriteLine("QueryString:");
            //String[] querykeys = Context.Request.QueryString.AllKeys;
            //foreach (string key in querykeys)
            //{
            //    System.Diagnostics.Debug.WriteLine(key + ":" + Context.Request.QueryString[key]);
            //}

            System.Diagnostics.Debug.WriteLine("Form:");
            string[] formkeys = Context.Request.Form.AllKeys;
            foreach (string key in formkeys)
            {
                System.Diagnostics.Debug.WriteLine(key + ":" + Context.Request.Form[key]);
            }     
        }

        private String GetRequestData()
        {
            String requestData = String.Empty;
            foreach (string key in Context.Request.Form)
            {
                requestData += "<br>" + key + ": " + Context.Request.Form[key];
            }

            return requestData.Substring(4);
        }

        private bool IsValidRequest()
        {
            if (IsBanned())
            {
                return false;
            }

            // Ensure the keys match. If they don't, consider a ban:
            if (Context.Request.QueryString[sSecretKeyName] == null || Context.Request.QueryString[sSecretKeyName] != sSecretKeyValue)
            {
                Warn(Context.Request.UserHostAddress);
                return false;
            }
            return true;
        }

    }
}
